/*
    SSSD

    IPA Provider Common Functions

    Authors:
        Simo Sorce <ssorce@redhat.com>

    Copyright (C) 2009 Red Hat

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE   /* For strcasestr() in ipa_set_search_bases() */
#endif

#include <netdb.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <ldb.h>
#include <string.h>

#include "db/sysdb_selinux.h"
#include "providers/ipa/ipa_common.h"
#include "providers/ipa/ipa_dyndns.h"
#include "providers/ldap/sdap_async_private.h"
#include "providers/be_dyndns.h"
#include "util/sss_krb5.h"
#include "db/sysdb_services.h"
#include "db/sysdb_autofs.h"

#include "providers/ipa/ipa_opts.h"
#include "providers/data_provider/dp_private.h"

int ipa_get_options(TALLOC_CTX *memctx,
                    struct confdb_ctx *cdb,
                    const char *conf_path,
                    struct sss_domain_info *dom,
                    struct ipa_options **_opts)
{
    struct ipa_options *opts;
    char *domain;
    char *server;
    char *realm;
    char *ipa_hostname;
    int ret;
    char hostname[HOST_NAME_MAX + 1];

    opts = talloc_zero(memctx, struct ipa_options);
    if (!opts) return ENOMEM;

    ret = dp_get_options(opts, cdb, conf_path,
                         ipa_basic_opts,
                         IPA_OPTS_BASIC,
                         &opts->basic);
    if (ret != EOK) {
        goto done;
    }

    domain = dp_opt_get_string(opts->basic, IPA_DOMAIN);
    if (!domain) {
        ret = dp_opt_set_string(opts->basic, IPA_DOMAIN, dom->name);
        if (ret != EOK) {
            goto done;
        }
        domain = dom->name;
    }

    server = dp_opt_get_string(opts->basic, IPA_SERVER);
    if (!server) {
        DEBUG(SSSDBG_CRIT_FAILURE,
              "No ipa server set, will use service discovery!\n");
    }

    ipa_hostname = dp_opt_get_string(opts->basic, IPA_HOSTNAME);
    if (ipa_hostname == NULL) {
        ret = gethostname(hostname, sizeof(hostname));
        if (ret != EOK) {
            ret = errno;
            DEBUG(SSSDBG_CRIT_FAILURE, "gethostname failed [%d][%s].\n", ret,
                      strerror(ret));
            goto done;
        }
        hostname[HOST_NAME_MAX] = '\0';
        DEBUG(SSSDBG_TRACE_ALL, "Setting ipa_hostname to [%s].\n", hostname);
        ret = dp_opt_set_string(opts->basic, IPA_HOSTNAME, hostname);
        if (ret != EOK) {
            goto done;
        }
    }

    /* First check whether the realm has been manually specified */
    realm = dp_opt_get_string(opts->basic, IPA_KRB5_REALM);
    if (!realm) {
        /* No explicit krb5_realm, use the IPA domain, transform to upper-case */
        realm = get_uppercase_realm(opts, domain);
        if (!realm) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(opts->basic, IPA_KRB5_REALM,
                                realm);
        if (ret != EOK) {
            goto done;
        }
    }

    ret = EOK;
    *_opts = opts;

done:
    if (ret != EOK) {
        talloc_zfree(opts);
    }
    return ret;
}

static errno_t ipa_parse_search_base(TALLOC_CTX *mem_ctx,
                                     struct ldb_context *ldb,
                                     struct dp_option *opts, int class,
                                     struct sdap_search_base ***_search_bases)
{
    const char *class_name;
    char *unparsed_base;

    *_search_bases = NULL;

    switch (class) {
    case IPA_HBAC_SEARCH_BASE:
        class_name = "IPA_HBAC";
        break;
    case IPA_SELINUX_SEARCH_BASE:
        class_name = "IPA_SELINUX";
        break;
    case IPA_SUBDOMAINS_SEARCH_BASE:
        class_name = "IPA_SUBDOMAINS";
        break;
    case IPA_MASTER_DOMAIN_SEARCH_BASE:
        class_name = "IPA_MASTER_DOMAIN";
        break;
    case IPA_RANGES_SEARCH_BASE:
        class_name = "IPA_RANGES";
        break;
    case IPA_VIEWS_SEARCH_BASE:
        class_name = "IPA_VIEWS";
        break;
    case IPA_DESKPROFILE_SEARCH_BASE:
        class_name = "IPA_DESKPROFILE";
        break;
    default:
        DEBUG(SSSDBG_CONF_SETTINGS,
              "Unknown search base type: [%d]\n", class);
        class_name = "UNKNOWN";
        /* Non-fatal */
        break;
    }

    unparsed_base = dp_opt_get_string(opts, class);
    if (!unparsed_base || unparsed_base[0] == '\0') return ENOENT;

    return common_parse_search_base(mem_ctx, unparsed_base, ldb,
                                    class_name, NULL,
                                    _search_bases);
}

int ipa_get_id_options(struct ipa_options *ipa_opts,
                       struct confdb_ctx *cdb,
                       const char *conf_path,
                       struct data_provider *dp,
                       bool sdom_add,
                       struct sdap_options **_opts)
{
    int ret;

    ipa_opts->id = talloc_zero(ipa_opts, struct sdap_options);
    if (!ipa_opts->id) {
        ret = ENOMEM;
        goto done;
    }
    ipa_opts->id->dp = dp;

    if (sdom_add) {
        ret = sdap_domain_add(ipa_opts->id,
                              ipa_opts->id_ctx->sdap_id_ctx->be->domain,
                              NULL);
        if (ret != EOK) {
            goto done;
        }
    }

    /* get sdap options */
    ret = dp_get_options(ipa_opts->id, cdb, conf_path,
                         ipa_def_ldap_opts,
                         SDAP_OPTS_BASIC,
                         &ipa_opts->id->basic);
    if (ret != EOK) {
        goto done;
    }

    /* sssd-ipa can't use simple bind, ignore option that potentially can be set
     * for sssd-ldap in the same domain
     */
    ret = dp_opt_set_string(ipa_opts->id->basic,
                            SDAP_DEFAULT_AUTHTOK_TYPE, NULL);
    if (ret != EOK) {
        goto done;
    }


    ret = EOK;
    *_opts = ipa_opts->id;

done:
    if (ret != EOK) {
        talloc_zfree(ipa_opts->id);
    }
    return ret;
}

errno_t
ipa_set_sdap_options(struct ipa_options *ipa_opts,
                     struct sdap_options *sdap_opts)
{
    TALLOC_CTX *tmpctx;
    errno_t ret;
    char *realm;
    char *value;

    tmpctx = talloc_new(ipa_opts);
    if (!tmpctx) {
        return ENOMEM;
    }

    /* set krb realm */
    if (NULL == dp_opt_get_string(sdap_opts->basic, SDAP_KRB5_REALM)) {
        realm = dp_opt_get_string(ipa_opts->basic, IPA_KRB5_REALM);
        value = talloc_strdup(tmpctx, realm);
        if (value == NULL) {
            DEBUG(SSSDBG_CRIT_FAILURE, "talloc_strdup failed.\n");
            ret = ENOMEM;
            goto done;
        }
        ret = dp_opt_set_string(sdap_opts->basic,
                                SDAP_KRB5_REALM, value);
        if (ret != EOK) {
            goto done;
        }
        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_KRB5_REALM].opt_name,
                  dp_opt_get_string(sdap_opts->basic, SDAP_KRB5_REALM));
    }

    ret = sdap_set_sasl_options(sdap_opts,
                                dp_opt_get_string(ipa_opts->basic,
                                                  IPA_HOSTNAME),
                                dp_opt_get_string(sdap_opts->basic,
                                                  SDAP_KRB5_REALM),
                                dp_opt_get_string(sdap_opts->basic,
                                                  SDAP_KRB5_KEYTAB));
    if (ret != EOK) {
        DEBUG(SSSDBG_OP_FAILURE, "Cannot set the SASL-related options\n");
        goto done;
    }

    /* fix schema to IPAv1 for now */
    sdap_opts->schema_type = SDAP_SCHEMA_IPA_V1;

    ret = EOK;

done:
    talloc_zfree(tmpctx);
    return ret;
}

errno_t
ipa_set_search_bases(struct ipa_options *ipa_opts,
                     struct confdb_ctx *cdb,
                     const char *basedn,
                     const char *conf_path,
                     struct sdap_domain *sdap_dom)
{
    TALLOC_CTX *tmpctx;
    errno_t ret;
    struct sdap_domain *sdom;
    char *value;
    struct ldb_context *ldb;
    struct sdap_options *sdap_opts;
    bool server_mode;
    int i;

    sdap_opts = ipa_opts->id;

    if (sdap_dom != NULL) {
        sdom = sdap_dom;
    } else {
        sdom = ipa_opts->id->sdom;
    }

    tmpctx = talloc_new(ipa_opts);
    if (!tmpctx) {
        return ENOMEM;
    }

    ldb = sysdb_ctx_get_ldb(ipa_opts->id->dp->be_ctx->domain->sysdb);

    if (NULL == dp_opt_get_string(sdap_opts->basic, SDAP_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=accounts,%s", basedn);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }
        ret = dp_opt_set_string(sdap_opts->basic,
                                SDAP_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic, SDAP_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts, ldb, sdap_opts->basic,
                                 SDAP_SEARCH_BASE,
                                 &sdom->search_bases);
    if (ret != EOK) goto done;

    /* set user/group search bases if they are not specified */
    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_USER_SEARCH_BASE)) {
        ret = dp_opt_set_string(sdap_opts->basic, SDAP_USER_SEARCH_BASE,
                                dp_opt_get_string(sdap_opts->basic,
                                                  SDAP_SEARCH_BASE));
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_USER_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic,
                                    SDAP_USER_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts, ldb, sdap_opts->basic,
                                 SDAP_USER_SEARCH_BASE,
                                 &sdom->user_search_bases);
    if (ret != EOK) goto done;

    /* In server mode we need to search both cn=accounts,$SUFFIX and
     * cn=trusts,$SUFFIX to allow trusted domain object accounts to be found.
     * If cn=trusts,$SUFFIX is missing in the user search bases, add one
     */
    server_mode = dp_opt_get_bool(ipa_opts->basic, IPA_SERVER_MODE);
    if (server_mode != false) {
        /* bases is not NULL at this point already */
        struct sdap_search_base **bases = sdom->user_search_bases;
        struct sdap_search_base *new_base = NULL;

        for (i = 0; bases[i] != NULL; i++) {
            if (strcasestr(bases[i]->basedn, "cn=trusts,") != NULL) {
                break;
            }
        }
        if (NULL == bases[i]) {
            /* no cn=trusts in the base, add a new one */
            char *new_dn = talloc_asprintf(bases,
                                           "cn=trusts,%s",
                                           basedn);
            if (NULL == new_dn) {
                ret = ENOMEM;
                goto done;
            }

            ret = sdap_create_search_base(bases, ldb, new_dn,
                                          LDAP_SCOPE_SUBTREE,
                                          "(objectClass=ipaIDObject)",
                                          &new_base);
            if (ret != EOK) {
                goto done;
            }

            bases = talloc_realloc(sdap_opts,
                                   sdom->user_search_bases,
                                   struct sdap_search_base*,
                                   i + 2);

            if (NULL == bases) {
                ret = ENOMEM;
                goto done;
            }

            bases[i] = new_base;
            bases[i+1] = NULL;
            sdom->user_search_bases = bases;

            DEBUG(SSSDBG_TRACE_FUNC,
                  "Option %s expanded to cover cn=trusts base\n",
                  sdap_opts->basic[SDAP_USER_SEARCH_BASE].opt_name);
        }
    }

    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_GROUP_SEARCH_BASE)) {
        ret = dp_opt_set_string(sdap_opts->basic, SDAP_GROUP_SEARCH_BASE,
                                dp_opt_get_string(sdap_opts->basic,
                                                  SDAP_SEARCH_BASE));
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_GROUP_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic,
                                    SDAP_GROUP_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts, ldb, sdap_opts->basic,
                                 SDAP_GROUP_SEARCH_BASE,
                                 &sdom->group_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_NETGROUP_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=ng,cn=alt,%s", basedn);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }
        ret = dp_opt_set_string(sdap_opts->basic, SDAP_NETGROUP_SEARCH_BASE,
                                value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_NETGROUP_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic,
                                    SDAP_NETGROUP_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts, ldb, sdap_opts->basic,
                                 SDAP_NETGROUP_SEARCH_BASE,
                                 &sdom->netgroup_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_HOST_SEARCH_BASE)) {

        value = dp_opt_get_string(ipa_opts->basic, IPA_HOST_SEARCH_BASE);
        if (!value) {
            value = dp_opt_get_string(sdap_opts->basic, SDAP_SEARCH_BASE);
        }

        ret = dp_opt_set_string(sdap_opts->basic, SDAP_HOST_SEARCH_BASE,
                                value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
              sdap_opts->basic[SDAP_HOST_SEARCH_BASE].opt_name,
              value);
    }
    ret = sdap_parse_search_base(sdap_opts->basic, ldb, sdap_opts->basic,
                                 SDAP_HOST_SEARCH_BASE,
                                 &sdom->host_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_HBAC_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=hbac,%s", basedn);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_HBAC_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_HBAC_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_HBAC_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts->basic, ldb, ipa_opts->basic,
                                IPA_HBAC_SEARCH_BASE,
                                &ipa_opts->hbac_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_SELINUX_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=selinux,%s", basedn);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_SELINUX_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_SELINUX_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_SELINUX_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts->basic, ldb, ipa_opts->basic,
                                IPA_SELINUX_SEARCH_BASE,
                                &ipa_opts->selinux_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_DESKPROFILE_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=desktop-profile,%s", basedn);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_DESKPROFILE_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_DESKPROFILE_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_DESKPROFILE_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts->basic, ldb, ipa_opts->basic,
                                IPA_DESKPROFILE_SEARCH_BASE,
                                &ipa_opts->deskprofile_search_bases);
    if (ret != EOK) goto done;

#ifdef BUILD_SUBID
    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_SUBID_RANGES_SEARCH_BASE)) {
        /* check deprecated option */
        value = dp_opt_get_string(ipa_opts->basic, IPA_SUBID_RANGES_SEARCH_BASE);
        if (value == NULL) { /* set a default */
            value = talloc_asprintf(tmpctx, "cn=subids,%s",
                                    sdom->search_bases[0]->basedn);
        }
        if (value == NULL) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(sdap_opts->basic, SDAP_SUBID_RANGES_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_SUBID_RANGES_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic,
                                    SDAP_SUBID_RANGES_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts->basic, ldb, sdap_opts->basic,
                                 SDAP_SUBID_RANGES_SEARCH_BASE,
                                 &sdom->subid_ranges_search_bases);
    if (ret != EOK) goto done;

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_subid_map,
                       SDAP_OPTS_SUBID_RANGE,
                       &sdap_opts->subid_map);
    if (ret != EOK) {
        goto done;
    }
#endif

    value = dp_opt_get_string(sdap_opts->basic, SDAP_DEREF);
    if (value != NULL) {
        ret = deref_string_to_val(value, &i);
        if (ret != EOK) {
            DEBUG(SSSDBG_CRIT_FAILURE, "Failed to verify ldap_deref option.\n");
            goto done;
        }
    }

    if (NULL == dp_opt_get_string(sdap_opts->basic,
                                  SDAP_SERVICE_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=ipservices,%s",
                                dp_opt_get_string(sdap_opts->basic,
                                                  SDAP_SEARCH_BASE));
        if (!value) {
            ret = ENOMEM;
            goto done;
        }
        ret = dp_opt_set_string(sdap_opts->basic,
                                SDAP_SERVICE_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  sdap_opts->basic[SDAP_SERVICE_SEARCH_BASE].opt_name,
                  dp_opt_get_string(sdap_opts->basic,
                                    SDAP_SERVICE_SEARCH_BASE));
    }
    ret = sdap_parse_search_base(sdap_opts, ldb, sdap_opts->basic,
                                 SDAP_SERVICE_SEARCH_BASE,
                                 &sdom->service_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_SUBDOMAINS_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=trusts,%s", basedn);
        if (value == NULL) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_SUBDOMAINS_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_SUBDOMAINS_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_SUBDOMAINS_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts, ldb, ipa_opts->basic,
                                IPA_SUBDOMAINS_SEARCH_BASE,
                                &ipa_opts->subdomains_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_MASTER_DOMAIN_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=ad,cn=etc,%s", basedn);
        if (value == NULL) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_MASTER_DOMAIN_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_MASTER_DOMAIN_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_MASTER_DOMAIN_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts, ldb, ipa_opts->basic,
                                IPA_MASTER_DOMAIN_SEARCH_BASE,
                                &ipa_opts->master_domain_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_RANGES_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=ranges,cn=etc,%s", basedn);
        if (value == NULL) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_RANGES_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_RANGES_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_RANGES_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts, ldb, ipa_opts->basic,
                                IPA_RANGES_SEARCH_BASE,
                                &ipa_opts->ranges_search_bases);
    if (ret != EOK) goto done;

    if (NULL == dp_opt_get_string(ipa_opts->basic,
                                  IPA_VIEWS_SEARCH_BASE)) {
        value = talloc_asprintf(tmpctx, "cn=views,cn=accounts,%s", basedn);
        if (value == NULL) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->basic, IPA_VIEWS_SEARCH_BASE, value);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
                  ipa_opts->basic[IPA_VIEWS_SEARCH_BASE].opt_name,
                  dp_opt_get_string(ipa_opts->basic,
                                    IPA_VIEWS_SEARCH_BASE));
    }
    ret = ipa_parse_search_base(ipa_opts, ldb, ipa_opts->basic,
                                IPA_VIEWS_SEARCH_BASE,
                                &ipa_opts->views_search_bases);
    if (ret != EOK) goto done;

    ret = sdap_get_map(sdap_opts, cdb, conf_path,
                       ipa_attr_map,
                       SDAP_AT_GENERAL,
                       &sdap_opts->gen_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_user_map,
                       SDAP_OPTS_USER,
                       &sdap_opts->user_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_extend_map_with_list(sdap_opts, sdap_opts,
                                    SDAP_USER_EXTRA_ATTRS,
                                    sdap_opts->user_map,
                                    SDAP_OPTS_USER,
                                    &sdap_opts->user_map,
                                    &sdap_opts->user_map_cnt);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_group_map,
                       SDAP_OPTS_GROUP,
                       &sdap_opts->group_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_netgroup_map,
                       IPA_OPTS_NETGROUP,
                       &sdap_opts->netgroup_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_host_map,
                       SDAP_OPTS_HOST,
                       &sdap_opts->host_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_hostgroup_map,
                       IPA_OPTS_HOSTGROUP,
                       &ipa_opts->hostgroup_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_service_map,
                       SDAP_OPTS_SERVICES,
                       &sdap_opts->service_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_selinux_user_map,
                       IPA_OPTS_SELINUX_USERMAP,
                       &ipa_opts->selinuxuser_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_view_map,
                       IPA_OPTS_VIEW,
                       &ipa_opts->view_map);
    if (ret != EOK) {
        goto done;
    }

    ret = sdap_get_map(sdap_opts,
                       cdb, conf_path,
                       ipa_override_map,
                       IPA_OPTS_OVERRIDE,
                       &ipa_opts->override_map);
    if (ret != EOK) {
        goto done;
    }

    ret = EOK;

done:
    talloc_zfree(tmpctx);
    return ret;
}

int ipa_get_auth_options(struct ipa_options *ipa_opts,
                         struct confdb_ctx *cdb,
                         const char *conf_path,
                         struct dp_option **_opts)
{
    char *value;
    char *copy = NULL;
    int ret;

    ipa_opts->auth = talloc_zero(ipa_opts, struct dp_option);
    if (ipa_opts->auth == NULL) {
        ret = ENOMEM;
        goto done;
    }

    /* get krb5 options */
    ret = dp_get_options(ipa_opts, cdb, conf_path,
                         ipa_def_krb5_opts,
                         KRB5_OPTS, &ipa_opts->auth);
    if (ret != EOK) {
        goto done;
    }

    /* If there is no KDC, try the deprecated krb5_kdcip option, too */
    /* FIXME - this can be removed in a future version */
    ret = krb5_try_kdcip(cdb, conf_path, ipa_opts->auth, KRB5_KDC);
    if (ret != EOK) {
        DEBUG(SSSDBG_CRIT_FAILURE, "sss_krb5_try_kdcip failed.\n");
        goto done;
    }

    /* set krb realm */
    if (NULL == dp_opt_get_string(ipa_opts->auth, KRB5_REALM)) {
        value = dp_opt_get_string(ipa_opts->basic, IPA_KRB5_REALM);
        if (!value) {
            ret = ENOMEM;
            goto done;
        }
        copy = talloc_strdup(ipa_opts->auth, value);
        if (copy == NULL) {
            DEBUG(SSSDBG_CRIT_FAILURE, "talloc_strdup failed.\n");
            ret = ENOMEM;
            goto done;
        }
        ret = dp_opt_set_string(ipa_opts->auth, KRB5_REALM, copy);
        if (ret != EOK) {
            goto done;
        }
        DEBUG(SSSDBG_TRACE_FUNC, "Option %s set to %s\n",
                  ipa_opts->auth[KRB5_REALM].opt_name,
                  dp_opt_get_string(ipa_opts->auth, KRB5_REALM));
    }

    /* If krb5_fast_principal was not set explicitly, default to
     * host/$client_hostname@REALM
     */
    value = dp_opt_get_string(ipa_opts->auth, KRB5_FAST_PRINCIPAL);
    if (value == NULL) {
        value = talloc_asprintf(ipa_opts->auth, "host/%s@%s",
                                    dp_opt_get_string(ipa_opts->basic,
                                                      IPA_HOSTNAME),
                                    dp_opt_get_string(ipa_opts->auth,
                                                      KRB5_REALM));
        if (value == NULL) {
            DEBUG(SSSDBG_CRIT_FAILURE, "talloc_asprintf() failed\n");
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->auth, KRB5_FAST_PRINCIPAL,
                                value);
        if (ret != EOK) {
            DEBUG(SSSDBG_CRIT_FAILURE, "Cannot set %s!\n",
                     ipa_opts->auth[KRB5_FAST_PRINCIPAL].opt_name);
            goto done;
        }

        DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
              ipa_opts->auth[KRB5_FAST_PRINCIPAL].opt_name, value);
    }

    /* Set flag that controls whether we want to write the
     * kdcinfo files at all
     */
    ipa_opts->service->krb5_service->write_kdcinfo = \
        dp_opt_get_bool(ipa_opts->auth, KRB5_USE_KDCINFO);
    DEBUG(SSSDBG_CONF_SETTINGS, "Option %s set to %s\n",
          ipa_opts->auth[KRB5_USE_KDCINFO].opt_name,
          ipa_opts->service->krb5_service->write_kdcinfo ? "true" : "false");
    if (ipa_opts->service->krb5_service->write_kdcinfo) {
        sss_krb5_parse_lookahead(
            dp_opt_get_string(ipa_opts->auth, KRB5_KDCINFO_LOOKAHEAD),
            &ipa_opts->service->krb5_service->lookahead_primary,
            &ipa_opts->service->krb5_service->lookahead_backup);
    }

    *_opts = ipa_opts->auth;
    ret = EOK;

done:
    talloc_free(copy);
    if (ret != EOK) {
        talloc_zfree(ipa_opts->auth);
    }
    return ret;
}

static void ipa_resolve_callback(void *private_data, struct fo_server *server)
{
    TALLOC_CTX *tmp_ctx = NULL;
    struct ipa_service *service;
    struct resolv_hostent *srvaddr;
    struct sockaddr *sockaddr;
    char *new_uri;
    const char *srv_name;
    socklen_t sockaddr_len;
    int ret;

    tmp_ctx = talloc_new(NULL);
    if (tmp_ctx == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "talloc_new failed\n");
        return;
    }

    service = talloc_get_type(private_data, struct ipa_service);
    if (!service) {
        DEBUG(SSSDBG_CRIT_FAILURE, "FATAL: Bad private_data\n");
        talloc_free(tmp_ctx);
        return;
    }

    srvaddr = fo_get_server_hostent(server);
    if (!srvaddr) {
        DEBUG(SSSDBG_CRIT_FAILURE,
              "No hostent available for server (%s)\n",
                  fo_get_server_str_name(server));
        talloc_free(tmp_ctx);
        return;
    }

    sockaddr = resolv_get_sockaddr_address(tmp_ctx, srvaddr, LDAP_PORT, &sockaddr_len);
    if (sockaddr == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "resolv_get_sockaddr_address failed.\n");
        talloc_free(tmp_ctx);
        return;
    }

    srv_name = fo_get_server_name(server);
    if (srv_name == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "Could not get server host name\n");
        talloc_free(tmp_ctx);
        return;
    }

    new_uri = talloc_asprintf(service, "ldap://%s", srv_name);
    if (!new_uri) {
        DEBUG(SSSDBG_OP_FAILURE, "Failed to copy URI ...\n");
        talloc_free(tmp_ctx);
        return;
    }
    DEBUG(SSSDBG_TRACE_FUNC, "Constructed uri '%s'\n", new_uri);

    /* free old one and replace with new one */
    talloc_zfree(service->sdap->uri);
    service->sdap->uri = new_uri;
    talloc_zfree(service->sdap->sockaddr);
    service->sdap->sockaddr = talloc_steal(service, sockaddr);
    service->sdap->sockaddr_len = sockaddr_len;

    if (service->krb5_service->write_kdcinfo) {
        ret = write_krb5info_file_from_fo_server(service->krb5_service,
                                                 server,
                                                 true,
                                                 SSS_KRB5KDC_FO_SRV,
                                                 NULL);
        if (ret != EOK) {
            DEBUG(SSSDBG_OP_FAILURE,
                  "write to %s/kdcinfo.%s failed, authentication might fail.\n",
                  PUBCONF_PATH, service->krb5_service->realm);
        }
    }

    talloc_free(tmp_ctx);
}

static errno_t _ipa_servers_init(struct be_ctx *ctx,
                                 const char *fo_service,
                                 struct ipa_service *service,
                                 struct ipa_options *options,
                                 const char *servers,
                                 bool primary)
{
    TALLOC_CTX *tmp_ctx;
    char **list = NULL;
    char *ipa_domain;
    int ret = 0;
    int i;
    int j;

    tmp_ctx = talloc_new(NULL);
    if (!tmp_ctx) {
        return ENOMEM;
    }

    /* split server parm into a list */
    ret = split_on_separator(tmp_ctx, servers, ',', true, true, &list, NULL);
    if (ret != EOK) {
        DEBUG(SSSDBG_CRIT_FAILURE, "Failed to parse server list!\n");
        goto done;
    }

    for (j = 0; list[j]; j++) {
        if (resolv_is_address(list[j])) {
            DEBUG(SSSDBG_IMPORTANT_INFO,
                  "ipa_server [%s] is detected as IP address, "
                  "this can cause GSSAPI problems\n", list[j]);
        }
    }

    /* now for each one add a new server to the failover service */
    for (i = 0; list[i]; i++) {

        talloc_steal(service, list[i]);

        if (be_fo_is_srv_identifier(list[i])) {
            if (!primary) {
                DEBUG(SSSDBG_MINOR_FAILURE,
                      "Failed to add server [%s] to failover service: "
                       "SRV resolution only allowed for primary servers!\n",
                       list[i]);
                continue;
            }

            ipa_domain = dp_opt_get_string(options->basic, IPA_DOMAIN);
            ret = be_fo_add_srv_server(ctx, fo_service, "ldap", ipa_domain,
                                       BE_FO_PROTO_TCP, false, NULL);
            if (ret) {
                DEBUG(SSSDBG_FATAL_FAILURE, "Failed to add server\n");
                goto done;
            }

            DEBUG(SSSDBG_TRACE_FUNC, "Added service lookup for service [%s]\n",
                                     fo_service);
            continue;
        }

        /* It could be ipv6 address in square brackets. Remove
         * the brackets if needed. */
        ret = remove_ipv6_brackets(list[i]);
        if (ret != EOK) {
            goto done;
        }

        ret = be_fo_add_server(ctx, fo_service, list[i], 0, NULL, primary);
        if (ret && ret != EEXIST) {
            DEBUG(SSSDBG_FATAL_FAILURE, "Failed to add server\n");
            goto done;
        }

        DEBUG(SSSDBG_TRACE_FUNC, "Added Server %s\n", list[i]);
    }

done:
    talloc_free(tmp_ctx);
    return ret;
}

static inline errno_t
ipa_primary_servers_init(struct be_ctx *ctx, const char *fo_service,
                         struct ipa_service *service, struct ipa_options *options,
                         const char *servers)
{
    return _ipa_servers_init(ctx, fo_service, service, options, servers, true);
}

static inline errno_t
ipa_backup_servers_init(struct be_ctx *ctx, const char *fo_service,
                        struct ipa_service *service, struct ipa_options *options,
                        const char *servers)
{
    return _ipa_servers_init(ctx, fo_service, service, options, servers, false);
}

static int ipa_user_data_cmp(void *ud1, void *ud2)
{
    return strcasecmp((char*) ud1, (char*) ud2);
}

int ipa_service_init(TALLOC_CTX *memctx, struct be_ctx *ctx,
                     const char *primary_servers,
                     const char *backup_servers,
                     const char *realm,
                     const char *ipa_service,
                     struct ipa_options *options,
                     struct ipa_service **_service)
{
    TALLOC_CTX *tmp_ctx;
    struct ipa_service *service;
    int ret;

    tmp_ctx = talloc_new(NULL);
    if (!tmp_ctx) {
        return ENOMEM;
    }

    service = talloc_zero(tmp_ctx, struct ipa_service);
    if (!service) {
        ret = ENOMEM;
        goto done;
    }
    service->sdap = talloc_zero(service, struct sdap_service);
    if (!service->sdap) {
        ret = ENOMEM;
        goto done;
    }

    service->krb5_service = krb5_service_new(service, ctx,
                                             ipa_service, realm,
                                             true,   /* The configured value */
                                             0,      /* will be set later when */
                                             0);     /* the auth provider is set up */

    if (!service->krb5_service) {
        ret = ENOMEM;
        goto done;
    }

    ret = be_fo_add_service(ctx, ipa_service, ipa_user_data_cmp);
    if (ret != EOK) {
        DEBUG(SSSDBG_CRIT_FAILURE, "Failed to create failover service!\n");
        goto done;
    }

    service->sdap->name = talloc_strdup(service, ipa_service);
    if (!service->sdap->name) {
        ret = ENOMEM;
        goto done;
    }

    service->sdap->kinit_service_name = service->krb5_service->name;

    if (!primary_servers) {
        DEBUG(SSSDBG_CONF_SETTINGS,
              "No primary servers defined, using service discovery\n");
        primary_servers = BE_SRV_IDENTIFIER;
    }

    ret = ipa_primary_servers_init(ctx, ipa_service, service, options, primary_servers);
    if (ret != EOK) {
        goto done;
    }

    if (backup_servers) {
        ret = ipa_backup_servers_init(ctx, ipa_service, service, options, backup_servers);
        if (ret != EOK) {
            goto done;
        }
    }

    ret = be_fo_service_add_callback(memctx, ctx, ipa_service,
                                     ipa_resolve_callback, service);
    if (ret != EOK) {
        DEBUG(SSSDBG_CRIT_FAILURE, "Failed to add failover callback!\n");
        goto done;
    }

    ret = EOK;

done:
    if (ret == EOK) {
        *_service = talloc_steal(memctx, service);
    }
    talloc_zfree(tmp_ctx);
    return ret;
}

int ipa_get_autofs_options(struct ipa_options *ipa_opts,
                           struct ldb_context *ldb,
                           struct confdb_ctx *cdb,
                           const char *conf_path,
                           struct sdap_options **_opts)
{
    TALLOC_CTX *tmp_ctx;
    char *basedn;
    char *autofs_base;
    errno_t ret;

    tmp_ctx = talloc_new(NULL);
    if (!tmp_ctx) {
        return ENOMEM;
    }

    ret = domain_to_basedn(tmp_ctx,
                           dp_opt_get_string(ipa_opts->basic, IPA_KRB5_REALM),
                           &basedn);
    if (ret != EOK) {
        goto done;
    }

    if (NULL == dp_opt_get_string(ipa_opts->id->basic,
                                  SDAP_AUTOFS_SEARCH_BASE)) {

        autofs_base = talloc_asprintf(tmp_ctx, "cn=%s,cn=automount,%s",
                                dp_opt_get_string(ipa_opts->basic,
                                                  IPA_AUTOMOUNT_LOCATION),
                                basedn);
        if (!autofs_base) {
            ret = ENOMEM;
            goto done;
        }

        ret = dp_opt_set_string(ipa_opts->id->basic,
                                SDAP_AUTOFS_SEARCH_BASE,
                                autofs_base);
        if (ret != EOK) {
            goto done;
        }

        DEBUG(SSSDBG_TRACE_LIBS, "Option %s set to %s\n",
              ipa_opts->id->basic[SDAP_AUTOFS_SEARCH_BASE].opt_name,
              dp_opt_get_string(ipa_opts->id->basic,
                                SDAP_AUTOFS_SEARCH_BASE));
    }

    ret = sdap_parse_search_base(ipa_opts->id, ldb, ipa_opts->id->basic,
                                 SDAP_AUTOFS_SEARCH_BASE,
                                 &ipa_opts->id->sdom->autofs_search_bases);
    if (ret != EOK && ret != ENOENT) {
        DEBUG(SSSDBG_OP_FAILURE, "Could not parse autofs search base\n");
        goto done;
    }

    ret = sdap_get_map(ipa_opts->id, cdb, conf_path,
                       ipa_autofs_mobject_map,
                       SDAP_OPTS_AUTOFS_MAP,
                       &ipa_opts->id->autofs_mobject_map);
    if (ret != EOK) {
        DEBUG(SSSDBG_OP_FAILURE,
              "Could not get autofs map object attribute map\n");
        goto done;
    }

    ret = sdap_get_map(ipa_opts->id, cdb, conf_path,
                       ipa_autofs_entry_map,
                       SDAP_OPTS_AUTOFS_ENTRY,
                       &ipa_opts->id->autofs_entry_map);
    if (ret != EOK) {
        DEBUG(SSSDBG_OP_FAILURE,
              "Could not get autofs entry object attribute map\n");
        goto done;
    }

    *_opts = ipa_opts->id;
    ret = EOK;
done:
    talloc_free(tmp_ctx);
    return ret;
}

errno_t ipa_get_host_attrs(struct dp_option *ipa_options,
                           size_t host_count,
                           struct sysdb_attrs **hosts,
                           struct sysdb_attrs **_ipa_host)
{
    const char *ipa_hostname;
    const char *hostname;
    errno_t ret;

    *_ipa_host = NULL;
    ipa_hostname = dp_opt_get_cstring(ipa_options, IPA_HOSTNAME);
    if (ipa_hostname == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE,
              "Missing ipa_hostname, this should never happen.\n");
        ret = EINVAL;
        goto done;
    }

    for (size_t i = 0; i < host_count; i++) {
        ret = sysdb_attrs_get_string(hosts[i], SYSDB_FQDN, &hostname);
        if (ret != EOK) {
            DEBUG(SSSDBG_CRIT_FAILURE, "Could not locate IPA host\n");
            goto done;
        }

        if (strcasecmp(hostname, ipa_hostname) == 0) {
            *_ipa_host = hosts[i];
            break;
        }
    }

    if (*_ipa_host == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "Could not locate IPA host\n");
        ret = EINVAL;
        goto done;
    }

    ret = EOK;

done:
    return ret;
}

struct ipa_options *
ipa_create_trust_options(TALLOC_CTX *mem_ctx,
                         struct be_ctx *be_ctx,
                         struct confdb_ctx *cdb,
                         const char *subdom_conf_path,
                         struct data_provider *dp,
                         struct sss_domain_info *subdom,
                         const char *keytab,
                         const char *sasl_authid)
{
    struct ipa_options *ipa_options = NULL;
    struct ipa_id_ctx *ipa_id_ctx = NULL;
    struct sdap_id_ctx *sdap_id_ctx = NULL;
    const char *ipa_servers;
    const char *ipa_backup_servers;
    const char *service_name;
    errno_t ret;

    DEBUG(SSSDBG_TRACE_FUNC, "Trust is defined to domain '%s'\n",
          subdom->name);

    ret = ipa_get_options(mem_ctx, cdb, subdom_conf_path, subdom, &ipa_options);
    if (ret != EOK || ipa_options == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "ipa_get_options failed\n");
        goto done;
    } else if (ipa_options == NULL) {
        DEBUG(SSSDBG_CRIT_FAILURE, "ipa_options is NULL\n");
        ret = ENOMEM;
        goto done;
    }

    ipa_servers = dp_opt_get_string(ipa_options->basic, IPA_SERVER);
    ipa_backup_servers = dp_opt_get_string(ipa_options->basic, IPA_BACKUP_SERVER);

    service_name = talloc_asprintf(ipa_options, "sd_%s", subdom->name);
    if (service_name == NULL) {
        ret = ENOMEM;
        goto done;
    }

    ret = ipa_service_init(ipa_options, be_ctx, ipa_servers,
                           ipa_backup_servers, subdom->realm,
                           service_name, ipa_options,
                           &ipa_options->service);
    if (ret != EOK) {
        DEBUG(SSSDBG_FATAL_FAILURE, "Failed to init IPA service [%d]: %s\n",
              ret, sss_strerror(ret));
        goto done;
    }

    /* Set IPA SDAP options */
    ipa_id_ctx = talloc_zero(mem_ctx, struct ipa_id_ctx);
    if (ipa_id_ctx == NULL) {
        ret = ENOMEM;
        goto done;
    }

    sdap_id_ctx = sdap_id_ctx_new(mem_ctx, be_ctx, ipa_options->service->sdap);
    if (sdap_id_ctx == NULL) {
        ret = ENOMEM;
        goto done;
    }

    ipa_id_ctx->ipa_options = ipa_options;
    ipa_id_ctx->sdap_id_ctx = sdap_id_ctx;
    ipa_options->id_ctx = ipa_id_ctx;

    ret = ipa_get_id_options(ipa_options,
                             cdb,
                             subdom_conf_path,
                             dp,
                             false,
                             &ipa_options->id);
    if (ret != EOK) {
        DEBUG(SSSDBG_CRIT_FAILURE, "ipa_get_id_options failed [%d]: %s\n",
              ret, sss_strerror(ret));
        goto done;
    }

    if (keytab != NULL) {
        ret = dp_opt_set_string(ipa_options->id->basic, SDAP_KRB5_KEYTAB,
                                keytab);
        if (ret != EOK) {
            DEBUG(SSSDBG_OP_FAILURE, "Cannot set trust keytab\n");
            goto done;
        }
    }

    /* Set SDAP_SASL_AUTHID to the trust principal */
    ret = dp_opt_set_string(ipa_options->id->basic,
                            SDAP_SASL_AUTHID, sasl_authid);
    if (ret != EOK) {
        DEBUG(SSSDBG_OP_FAILURE, "Cannot set SASL authid\n");
        goto done;
    }

done:
    if (ret != EOK) {
        if (ipa_options != NULL) {
            talloc_free(ipa_options);
        }
        if (ipa_id_ctx != NULL) {
            talloc_free(ipa_options);
        }
        return NULL;
    }

    return ipa_options;
}
