/* Copyright (c) 2000, 2013, Oracle and/or its affiliates.
   Copyright (c) 2009, 2017, MariaDB

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; version 2
   of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
   MA 02110-1301, USA */

/* UTF8 according RFC 2279 */
/* Written by Alexander Barkov <bar@udm.net> */

#include "strings_def.h"
#include <m_ctype.h>

#ifndef EILSEQ
#define EILSEQ ENOENT
#endif


#include "ctype-utf8.h"
#include "ctype-unidata.h"


/* Definitions for strcoll.ic */
#define IS_MB1_CHAR(x)              ((uchar) (x) < 0x80)
#define IS_MB1_MBHEAD_UNUSED_GAP(x) ((uchar) (x) < 0xC2)
#define IS_MB2_CHAR(x,y)            IS_UTF8MB2_STEP2(x,y)
#define IS_MB3_CHAR(x,y,z)          IS_UTF8MB3_STEP3(x,y,z)

/* Collation names */
#define MY_UTF8MB3_GENERAL_CI MY_UTF8MB3 "_general_ci"
#define MY_UTF8MB3_GENERAL_CS MY_UTF8MB3 "_general_cs"
#define MY_UTF8MB3_BIN        MY_UTF8MB3 "_bin"
#define MY_UTF8MB4_GENERAL_CI MY_UTF8MB4 "_general_ci"
#define MY_UTF8MB4_GENERAL_CS MY_UTF8MB4 "_general_cs"
#define MY_UTF8MB4_GENERAL_NOPAD_CI MY_UTF8MB4 "_general_nopad_ci"
#define MY_UTF8MB4_BIN        MY_UTF8MB4 "_bin"
#define MY_UTF8MB4_NOPAD_BIN  MY_UTF8MB4 "_nopad_bin"



#ifndef HAVE_CHARSET_utf8
#define HAVE_CHARSET_utf8
#endif

#ifdef HAVE_CHARSET_utf8
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_ucs2
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_utf16
#define HAVE_UNIDATA
#endif

#ifdef HAVE_CHARSET_utf32
#define HAVE_UNIDATA
#endif


#if defined(HAVE_CHARSET_utf8) || defined(HAVE_CHARSET_utf8mb4)

static inline
int my_valid_mbcharlen_utf8mb3(const uchar *s, const uchar *e)
{
  uchar c;

  DBUG_ASSERT(s < e);
  c= s[0];
  if (c < 0x80)
    return 1;

  if (c < 0xc2)
    return MY_CS_ILSEQ;

  if (c < 0xe0)
  {
    if (s+2 > e) /* We need 2 characters */
      return MY_CS_TOOSMALL2;

    if (!(IS_CONTINUATION_BYTE(s[1])))
      return MY_CS_ILSEQ;

    return 2;
  }

  DBUG_ASSERT(c < 0xf0);
  if (s+3 > e) /* We need 3 characters */
    return MY_CS_TOOSMALL3;

  if (!IS_UTF8MB3_STEP2(c, s[1], s[2]))
    return MY_CS_ILSEQ;

  return 3;
}

#endif  /*HAVE_CHARSET_utf8 || HAVE_CHARSET_utf8mb4*/

#ifdef HAVE_UNIDATA

#include "my_uctype.h"

MY_UNICASE_CHARACTER my_unicase_default_page00[]={
  {0x0000,0x0000,0x0000},  {0x0001,0x0001,0x0001},
  {0x0002,0x0002,0x0002},  {0x0003,0x0003,0x0003},
  {0x0004,0x0004,0x0004},  {0x0005,0x0005,0x0005},
  {0x0006,0x0006,0x0006},  {0x0007,0x0007,0x0007},
  {0x0008,0x0008,0x0008},  {0x0009,0x0009,0x0009},
  {0x000A,0x000A,0x000A},  {0x000B,0x000B,0x000B},
  {0x000C,0x000C,0x000C},  {0x000D,0x000D,0x000D},
  {0x000E,0x000E,0x000E},  {0x000F,0x000F,0x000F},
  {0x0010,0x0010,0x0010},  {0x0011,0x0011,0x0011},
  {0x0012,0x0012,0x0012},  {0x0013,0x0013,0x0013},
  {0x0014,0x0014,0x0014},  {0x0015,0x0015,0x0015},
  {0x0016,0x0016,0x0016},  {0x0017,0x0017,0x0017},
  {0x0018,0x0018,0x0018},  {0x0019,0x0019,0x0019},
  {0x001A,0x001A,0x001A},  {0x001B,0x001B,0x001B},
  {0x001C,0x001C,0x001C},  {0x001D,0x001D,0x001D},
  {0x001E,0x001E,0x001E},  {0x001F,0x001F,0x001F},
  {0x0020,0x0020,0x0020},  {0x0021,0x0021,0x0021},
  {0x0022,0x0022,0x0022},  {0x0023,0x0023,0x0023},
  {0x0024,0x0024,0x0024},  {0x0025,0x0025,0x0025},
  {0x0026,0x0026,0x0026},  {0x0027,0x0027,0x0027},
  {0x0028,0x0028,0x0028},  {0x0029,0x0029,0x0029},
  {0x002A,0x002A,0x002A},  {0x002B,0x002B,0x002B},
  {0x002C,0x002C,0x002C},  {0x002D,0x002D,0x002D},
  {0x002E,0x002E,0x002E},  {0x002F,0x002F,0x002F},
  {0x0030,0x0030,0x0030},  {0x0031,0x0031,0x0031},
  {0x0032,0x0032,0x0032},  {0x0033,0x0033,0x0033},
  {0x0034,0x0034,0x0034},  {0x0035,0x0035,0x0035},
  {0x0036,0x0036,0x0036},  {0x0037,0x0037,0x0037},
  {0x0038,0x0038,0x0038},  {0x0039,0x0039,0x0039},
  {0x003A,0x003A,0x003A},  {0x003B,0x003B,0x003B},
  {0x003C,0x003C,0x003C},  {0x003D,0x003D,0x003D},
  {0x003E,0x003E,0x003E},  {0x003F,0x003F,0x003F},
  {0x0040,0x0040,0x0040},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x005B,0x005B,0x005B},
  {0x005C,0x005C,0x005C},  {0x005D,0x005D,0x005D},
  {0x005E,0x005E,0x005E},  {0x005F,0x005F,0x005F},
  {0x0060,0x0060,0x0060},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x007B,0x007B,0x007B},
  {0x007C,0x007C,0x007C},  {0x007D,0x007D,0x007D},
  {0x007E,0x007E,0x007E},  {0x007F,0x007F,0x007F},
  {0x0080,0x0080,0x0080},  {0x0081,0x0081,0x0081},
  {0x0082,0x0082,0x0082},  {0x0083,0x0083,0x0083},
  {0x0084,0x0084,0x0084},  {0x0085,0x0085,0x0085},
  {0x0086,0x0086,0x0086},  {0x0087,0x0087,0x0087},
  {0x0088,0x0088,0x0088},  {0x0089,0x0089,0x0089},
  {0x008A,0x008A,0x008A},  {0x008B,0x008B,0x008B},
  {0x008C,0x008C,0x008C},  {0x008D,0x008D,0x008D},
  {0x008E,0x008E,0x008E},  {0x008F,0x008F,0x008F},
  {0x0090,0x0090,0x0090},  {0x0091,0x0091,0x0091},
  {0x0092,0x0092,0x0092},  {0x0093,0x0093,0x0093},
  {0x0094,0x0094,0x0094},  {0x0095,0x0095,0x0095},
  {0x0096,0x0096,0x0096},  {0x0097,0x0097,0x0097},
  {0x0098,0x0098,0x0098},  {0x0099,0x0099,0x0099},
  {0x009A,0x009A,0x009A},  {0x009B,0x009B,0x009B},
  {0x009C,0x009C,0x009C},  {0x009D,0x009D,0x009D},
  {0x009E,0x009E,0x009E},  {0x009F,0x009F,0x009F},
  {0x00A0,0x00A0,0x00A0},  {0x00A1,0x00A1,0x00A1},
  {0x00A2,0x00A2,0x00A2},  {0x00A3,0x00A3,0x00A3},
  {0x00A4,0x00A4,0x00A4},  {0x00A5,0x00A5,0x00A5},
  {0x00A6,0x00A6,0x00A6},  {0x00A7,0x00A7,0x00A7},
  {0x00A8,0x00A8,0x00A8},  {0x00A9,0x00A9,0x00A9},
  {0x00AA,0x00AA,0x00AA},  {0x00AB,0x00AB,0x00AB},
  {0x00AC,0x00AC,0x00AC},  {0x00AD,0x00AD,0x00AD},
  {0x00AE,0x00AE,0x00AE},  {0x00AF,0x00AF,0x00AF},
  {0x00B0,0x00B0,0x00B0},  {0x00B1,0x00B1,0x00B1},
  {0x00B2,0x00B2,0x00B2},  {0x00B3,0x00B3,0x00B3},
  {0x00B4,0x00B4,0x00B4},  {0x039C,0x00B5,0x039C},
  {0x00B6,0x00B6,0x00B6},  {0x00B7,0x00B7,0x00B7},
  {0x00B8,0x00B8,0x00B8},  {0x00B9,0x00B9,0x00B9},
  {0x00BA,0x00BA,0x00BA},  {0x00BB,0x00BB,0x00BB},
  {0x00BC,0x00BC,0x00BC},  {0x00BD,0x00BD,0x00BD},
  {0x00BE,0x00BE,0x00BE},  {0x00BF,0x00BF,0x00BF},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00D7,0x00D7,0x00D7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x00DF,0x00DF,0x0053},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00F7,0x00F7,0x00F7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x0178,0x00FF,0x0059}
};


/*
  Almost similar to my_unicase_default_page00, but maps sorting order
  for U+00DF to 0x00DF instead of 0x0053.
*/
static MY_UNICASE_CHARACTER plane00_mysql500[]={
  {0x0000,0x0000,0x0000},  {0x0001,0x0001,0x0001},
  {0x0002,0x0002,0x0002},  {0x0003,0x0003,0x0003},
  {0x0004,0x0004,0x0004},  {0x0005,0x0005,0x0005},
  {0x0006,0x0006,0x0006},  {0x0007,0x0007,0x0007},
  {0x0008,0x0008,0x0008},  {0x0009,0x0009,0x0009},
  {0x000A,0x000A,0x000A},  {0x000B,0x000B,0x000B},
  {0x000C,0x000C,0x000C},  {0x000D,0x000D,0x000D},
  {0x000E,0x000E,0x000E},  {0x000F,0x000F,0x000F},
  {0x0010,0x0010,0x0010},  {0x0011,0x0011,0x0011},
  {0x0012,0x0012,0x0012},  {0x0013,0x0013,0x0013},
  {0x0014,0x0014,0x0014},  {0x0015,0x0015,0x0015},
  {0x0016,0x0016,0x0016},  {0x0017,0x0017,0x0017},
  {0x0018,0x0018,0x0018},  {0x0019,0x0019,0x0019},
  {0x001A,0x001A,0x001A},  {0x001B,0x001B,0x001B},
  {0x001C,0x001C,0x001C},  {0x001D,0x001D,0x001D},
  {0x001E,0x001E,0x001E},  {0x001F,0x001F,0x001F},
  {0x0020,0x0020,0x0020},  {0x0021,0x0021,0x0021},
  {0x0022,0x0022,0x0022},  {0x0023,0x0023,0x0023},
  {0x0024,0x0024,0x0024},  {0x0025,0x0025,0x0025},
  {0x0026,0x0026,0x0026},  {0x0027,0x0027,0x0027},
  {0x0028,0x0028,0x0028},  {0x0029,0x0029,0x0029},
  {0x002A,0x002A,0x002A},  {0x002B,0x002B,0x002B},
  {0x002C,0x002C,0x002C},  {0x002D,0x002D,0x002D},
  {0x002E,0x002E,0x002E},  {0x002F,0x002F,0x002F},
  {0x0030,0x0030,0x0030},  {0x0031,0x0031,0x0031},
  {0x0032,0x0032,0x0032},  {0x0033,0x0033,0x0033},
  {0x0034,0x0034,0x0034},  {0x0035,0x0035,0x0035},
  {0x0036,0x0036,0x0036},  {0x0037,0x0037,0x0037},
  {0x0038,0x0038,0x0038},  {0x0039,0x0039,0x0039},
  {0x003A,0x003A,0x003A},  {0x003B,0x003B,0x003B},
  {0x003C,0x003C,0x003C},  {0x003D,0x003D,0x003D},
  {0x003E,0x003E,0x003E},  {0x003F,0x003F,0x003F},
  {0x0040,0x0040,0x0040},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x005B,0x005B,0x005B},
  {0x005C,0x005C,0x005C},  {0x005D,0x005D,0x005D},
  {0x005E,0x005E,0x005E},  {0x005F,0x005F,0x005F},
  {0x0060,0x0060,0x0060},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x007B,0x007B,0x007B},
  {0x007C,0x007C,0x007C},  {0x007D,0x007D,0x007D},
  {0x007E,0x007E,0x007E},  {0x007F,0x007F,0x007F},
  {0x0080,0x0080,0x0080},  {0x0081,0x0081,0x0081},
  {0x0082,0x0082,0x0082},  {0x0083,0x0083,0x0083},
  {0x0084,0x0084,0x0084},  {0x0085,0x0085,0x0085},
  {0x0086,0x0086,0x0086},  {0x0087,0x0087,0x0087},
  {0x0088,0x0088,0x0088},  {0x0089,0x0089,0x0089},
  {0x008A,0x008A,0x008A},  {0x008B,0x008B,0x008B},
  {0x008C,0x008C,0x008C},  {0x008D,0x008D,0x008D},
  {0x008E,0x008E,0x008E},  {0x008F,0x008F,0x008F},
  {0x0090,0x0090,0x0090},  {0x0091,0x0091,0x0091},
  {0x0092,0x0092,0x0092},  {0x0093,0x0093,0x0093},
  {0x0094,0x0094,0x0094},  {0x0095,0x0095,0x0095},
  {0x0096,0x0096,0x0096},  {0x0097,0x0097,0x0097},
  {0x0098,0x0098,0x0098},  {0x0099,0x0099,0x0099},
  {0x009A,0x009A,0x009A},  {0x009B,0x009B,0x009B},
  {0x009C,0x009C,0x009C},  {0x009D,0x009D,0x009D},
  {0x009E,0x009E,0x009E},  {0x009F,0x009F,0x009F},
  {0x00A0,0x00A0,0x00A0},  {0x00A1,0x00A1,0x00A1},
  {0x00A2,0x00A2,0x00A2},  {0x00A3,0x00A3,0x00A3},
  {0x00A4,0x00A4,0x00A4},  {0x00A5,0x00A5,0x00A5},
  {0x00A6,0x00A6,0x00A6},  {0x00A7,0x00A7,0x00A7},
  {0x00A8,0x00A8,0x00A8},  {0x00A9,0x00A9,0x00A9},
  {0x00AA,0x00AA,0x00AA},  {0x00AB,0x00AB,0x00AB},
  {0x00AC,0x00AC,0x00AC},  {0x00AD,0x00AD,0x00AD},
  {0x00AE,0x00AE,0x00AE},  {0x00AF,0x00AF,0x00AF},
  {0x00B0,0x00B0,0x00B0},  {0x00B1,0x00B1,0x00B1},
  {0x00B2,0x00B2,0x00B2},  {0x00B3,0x00B3,0x00B3},
  {0x00B4,0x00B4,0x00B4},  {0x039C,0x00B5,0x039C},
  {0x00B6,0x00B6,0x00B6},  {0x00B7,0x00B7,0x00B7},
  {0x00B8,0x00B8,0x00B8},  {0x00B9,0x00B9,0x00B9},
  {0x00BA,0x00BA,0x00BA},  {0x00BB,0x00BB,0x00BB},
  {0x00BC,0x00BC,0x00BC},  {0x00BD,0x00BD,0x00BD},
  {0x00BE,0x00BE,0x00BE},  {0x00BF,0x00BF,0x00BF},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00D7,0x00D7,0x00D7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x00DF,0x00DF,0x00DF},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00F7,0x00F7,0x00F7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x0178,0x00FF,0x0059}
};


static MY_UNICASE_CHARACTER plane01[]={
  {0x0100,0x0101,0x0041},  {0x0100,0x0101,0x0041},
  {0x0102,0x0103,0x0041},  {0x0102,0x0103,0x0041},
  {0x0104,0x0105,0x0041},  {0x0104,0x0105,0x0041},
  {0x0106,0x0107,0x0043},  {0x0106,0x0107,0x0043},
  {0x0108,0x0109,0x0043},  {0x0108,0x0109,0x0043},
  {0x010A,0x010B,0x0043},  {0x010A,0x010B,0x0043},
  {0x010C,0x010D,0x0043},  {0x010C,0x010D,0x0043},
  {0x010E,0x010F,0x0044},  {0x010E,0x010F,0x0044},
  {0x0110,0x0111,0x0110},  {0x0110,0x0111,0x0110},
  {0x0112,0x0113,0x0045},  {0x0112,0x0113,0x0045},
  {0x0114,0x0115,0x0045},  {0x0114,0x0115,0x0045},
  {0x0116,0x0117,0x0045},  {0x0116,0x0117,0x0045},
  {0x0118,0x0119,0x0045},  {0x0118,0x0119,0x0045},
  {0x011A,0x011B,0x0045},  {0x011A,0x011B,0x0045},
  {0x011C,0x011D,0x0047},  {0x011C,0x011D,0x0047},
  {0x011E,0x011F,0x0047},  {0x011E,0x011F,0x0047},
  {0x0120,0x0121,0x0047},  {0x0120,0x0121,0x0047},
  {0x0122,0x0123,0x0047},  {0x0122,0x0123,0x0047},
  {0x0124,0x0125,0x0048},  {0x0124,0x0125,0x0048},
  {0x0126,0x0127,0x0126},  {0x0126,0x0127,0x0126},
  {0x0128,0x0129,0x0049},  {0x0128,0x0129,0x0049},
  {0x012A,0x012B,0x0049},  {0x012A,0x012B,0x0049},
  {0x012C,0x012D,0x0049},  {0x012C,0x012D,0x0049},
  {0x012E,0x012F,0x0049},  {0x012E,0x012F,0x0049},
  {0x0130,0x0069,0x0049},  {0x0049,0x0131,0x0049},
  {0x0132,0x0133,0x0132},  {0x0132,0x0133,0x0132},
  {0x0134,0x0135,0x004A},  {0x0134,0x0135,0x004A},
  {0x0136,0x0137,0x004B},  {0x0136,0x0137,0x004B},
  {0x0138,0x0138,0x0138},  {0x0139,0x013A,0x004C},
  {0x0139,0x013A,0x004C},  {0x013B,0x013C,0x004C},
  {0x013B,0x013C,0x004C},  {0x013D,0x013E,0x004C},
  {0x013D,0x013E,0x004C},  {0x013F,0x0140,0x013F},
  {0x013F,0x0140,0x013F},  {0x0141,0x0142,0x0141},
  {0x0141,0x0142,0x0141},  {0x0143,0x0144,0x004E},
  {0x0143,0x0144,0x004E},  {0x0145,0x0146,0x004E},
  {0x0145,0x0146,0x004E},  {0x0147,0x0148,0x004E},
  {0x0147,0x0148,0x004E},  {0x0149,0x0149,0x0149},
  {0x014A,0x014B,0x014A},  {0x014A,0x014B,0x014A},
  {0x014C,0x014D,0x004F},  {0x014C,0x014D,0x004F},
  {0x014E,0x014F,0x004F},  {0x014E,0x014F,0x004F},
  {0x0150,0x0151,0x004F},  {0x0150,0x0151,0x004F},
  {0x0152,0x0153,0x0152},  {0x0152,0x0153,0x0152},
  {0x0154,0x0155,0x0052},  {0x0154,0x0155,0x0052},
  {0x0156,0x0157,0x0052},  {0x0156,0x0157,0x0052},
  {0x0158,0x0159,0x0052},  {0x0158,0x0159,0x0052},
  {0x015A,0x015B,0x0053},  {0x015A,0x015B,0x0053},
  {0x015C,0x015D,0x0053},  {0x015C,0x015D,0x0053},
  {0x015E,0x015F,0x0053},  {0x015E,0x015F,0x0053},
  {0x0160,0x0161,0x0053},  {0x0160,0x0161,0x0053},
  {0x0162,0x0163,0x0054},  {0x0162,0x0163,0x0054},
  {0x0164,0x0165,0x0054},  {0x0164,0x0165,0x0054},
  {0x0166,0x0167,0x0166},  {0x0166,0x0167,0x0166},
  {0x0168,0x0169,0x0055},  {0x0168,0x0169,0x0055},
  {0x016A,0x016B,0x0055},  {0x016A,0x016B,0x0055},
  {0x016C,0x016D,0x0055},  {0x016C,0x016D,0x0055},
  {0x016E,0x016F,0x0055},  {0x016E,0x016F,0x0055},
  {0x0170,0x0171,0x0055},  {0x0170,0x0171,0x0055},
  {0x0172,0x0173,0x0055},  {0x0172,0x0173,0x0055},
  {0x0174,0x0175,0x0057},  {0x0174,0x0175,0x0057},
  {0x0176,0x0177,0x0059},  {0x0176,0x0177,0x0059},
  {0x0178,0x00FF,0x0059},  {0x0179,0x017A,0x005A},
  {0x0179,0x017A,0x005A},  {0x017B,0x017C,0x005A},
  {0x017B,0x017C,0x005A},  {0x017D,0x017E,0x005A},
  {0x017D,0x017E,0x005A},  {0x0053,0x017F,0x0053},
  {0x0180,0x0180,0x0180},  {0x0181,0x0253,0x0181},
  {0x0182,0x0183,0x0182},  {0x0182,0x0183,0x0182},
  {0x0184,0x0185,0x0184},  {0x0184,0x0185,0x0184},
  {0x0186,0x0254,0x0186},  {0x0187,0x0188,0x0187},
  {0x0187,0x0188,0x0187},  {0x0189,0x0256,0x0189},
  {0x018A,0x0257,0x018A},  {0x018B,0x018C,0x018B},
  {0x018B,0x018C,0x018B},  {0x018D,0x018D,0x018D},
  {0x018E,0x01DD,0x018E},  {0x018F,0x0259,0x018F},
  {0x0190,0x025B,0x0190},  {0x0191,0x0192,0x0191},
  {0x0191,0x0192,0x0191},  {0x0193,0x0260,0x0193},
  {0x0194,0x0263,0x0194},  {0x01F6,0x0195,0x01F6},
  {0x0196,0x0269,0x0196},  {0x0197,0x0268,0x0197},
  {0x0198,0x0199,0x0198},  {0x0198,0x0199,0x0198},
  {0x019A,0x019A,0x019A},  {0x019B,0x019B,0x019B},
  {0x019C,0x026F,0x019C},  {0x019D,0x0272,0x019D},
  {0x019E,0x019E,0x019E},  {0x019F,0x0275,0x019F},
  {0x01A0,0x01A1,0x004F},  {0x01A0,0x01A1,0x004F},
  {0x01A2,0x01A3,0x01A2},  {0x01A2,0x01A3,0x01A2},
  {0x01A4,0x01A5,0x01A4},  {0x01A4,0x01A5,0x01A4},
  {0x01A6,0x0280,0x01A6},  {0x01A7,0x01A8,0x01A7},
  {0x01A7,0x01A8,0x01A7},  {0x01A9,0x0283,0x01A9},
  {0x01AA,0x01AA,0x01AA},  {0x01AB,0x01AB,0x01AB},
  {0x01AC,0x01AD,0x01AC},  {0x01AC,0x01AD,0x01AC},
  {0x01AE,0x0288,0x01AE},  {0x01AF,0x01B0,0x0055},
  {0x01AF,0x01B0,0x0055},  {0x01B1,0x028A,0x01B1},
  {0x01B2,0x028B,0x01B2},  {0x01B3,0x01B4,0x01B3},
  {0x01B3,0x01B4,0x01B3},  {0x01B5,0x01B6,0x01B5},
  {0x01B5,0x01B6,0x01B5},  {0x01B7,0x0292,0x01B7},
  {0x01B8,0x01B9,0x01B8},  {0x01B8,0x01B9,0x01B8},
  {0x01BA,0x01BA,0x01BA},  {0x01BB,0x01BB,0x01BB},
  {0x01BC,0x01BD,0x01BC},  {0x01BC,0x01BD,0x01BC},
  {0x01BE,0x01BE,0x01BE},  {0x01F7,0x01BF,0x01F7},
  {0x01C0,0x01C0,0x01C0},  {0x01C1,0x01C1,0x01C1},
  {0x01C2,0x01C2,0x01C2},  {0x01C3,0x01C3,0x01C3},
  {0x01C4,0x01C6,0x01C4},  {0x01C4,0x01C6,0x01C4},
  {0x01C4,0x01C6,0x01C4},  {0x01C7,0x01C9,0x01C7},
  {0x01C7,0x01C9,0x01C7},  {0x01C7,0x01C9,0x01C7},
  {0x01CA,0x01CC,0x01CA},  {0x01CA,0x01CC,0x01CA},
  {0x01CA,0x01CC,0x01CA},  {0x01CD,0x01CE,0x0041},
  {0x01CD,0x01CE,0x0041},  {0x01CF,0x01D0,0x0049},
  {0x01CF,0x01D0,0x0049},  {0x01D1,0x01D2,0x004F},
  {0x01D1,0x01D2,0x004F},  {0x01D3,0x01D4,0x0055},
  {0x01D3,0x01D4,0x0055},  {0x01D5,0x01D6,0x0055},
  {0x01D5,0x01D6,0x0055},  {0x01D7,0x01D8,0x0055},
  {0x01D7,0x01D8,0x0055},  {0x01D9,0x01DA,0x0055},
  {0x01D9,0x01DA,0x0055},  {0x01DB,0x01DC,0x0055},
  {0x01DB,0x01DC,0x0055},  {0x018E,0x01DD,0x018E},
  {0x01DE,0x01DF,0x0041},  {0x01DE,0x01DF,0x0041},
  {0x01E0,0x01E1,0x0041},  {0x01E0,0x01E1,0x0041},
  {0x01E2,0x01E3,0x00C6},  {0x01E2,0x01E3,0x00C6},
  {0x01E4,0x01E5,0x01E4},  {0x01E4,0x01E5,0x01E4},
  {0x01E6,0x01E7,0x0047},  {0x01E6,0x01E7,0x0047},
  {0x01E8,0x01E9,0x004B},  {0x01E8,0x01E9,0x004B},
  {0x01EA,0x01EB,0x004F},  {0x01EA,0x01EB,0x004F},
  {0x01EC,0x01ED,0x004F},  {0x01EC,0x01ED,0x004F},
  {0x01EE,0x01EF,0x01B7},  {0x01EE,0x01EF,0x01B7},
  {0x01F0,0x01F0,0x004A},  {0x01F1,0x01F3,0x01F1},
  {0x01F1,0x01F3,0x01F1},  {0x01F1,0x01F3,0x01F1},
  {0x01F4,0x01F5,0x0047},  {0x01F4,0x01F5,0x0047},
  {0x01F6,0x0195,0x01F6},  {0x01F7,0x01BF,0x01F7},
  {0x01F8,0x01F9,0x004E},  {0x01F8,0x01F9,0x004E},
  {0x01FA,0x01FB,0x0041},  {0x01FA,0x01FB,0x0041},
  {0x01FC,0x01FD,0x00C6},  {0x01FC,0x01FD,0x00C6},
  {0x01FE,0x01FF,0x00D8},  {0x01FE,0x01FF,0x00D8}
};

static MY_UNICASE_CHARACTER plane02[]={
  {0x0200,0x0201,0x0041},  {0x0200,0x0201,0x0041},
  {0x0202,0x0203,0x0041},  {0x0202,0x0203,0x0041},
  {0x0204,0x0205,0x0045},  {0x0204,0x0205,0x0045},
  {0x0206,0x0207,0x0045},  {0x0206,0x0207,0x0045},
  {0x0208,0x0209,0x0049},  {0x0208,0x0209,0x0049},
  {0x020A,0x020B,0x0049},  {0x020A,0x020B,0x0049},
  {0x020C,0x020D,0x004F},  {0x020C,0x020D,0x004F},
  {0x020E,0x020F,0x004F},  {0x020E,0x020F,0x004F},
  {0x0210,0x0211,0x0052},  {0x0210,0x0211,0x0052},
  {0x0212,0x0213,0x0052},  {0x0212,0x0213,0x0052},
  {0x0214,0x0215,0x0055},  {0x0214,0x0215,0x0055},
  {0x0216,0x0217,0x0055},  {0x0216,0x0217,0x0055},
  {0x0218,0x0219,0x0053},  {0x0218,0x0219,0x0053},
  {0x021A,0x021B,0x0054},  {0x021A,0x021B,0x0054},
  {0x021C,0x021D,0x021C},  {0x021C,0x021D,0x021C},
  {0x021E,0x021F,0x0048},  {0x021E,0x021F,0x0048},
  {0x0220,0x0220,0x0220},  {0x0221,0x0221,0x0221},
  {0x0222,0x0223,0x0222},  {0x0222,0x0223,0x0222},
  {0x0224,0x0225,0x0224},  {0x0224,0x0225,0x0224},
  {0x0226,0x0227,0x0041},  {0x0226,0x0227,0x0041},
  {0x0228,0x0229,0x0045},  {0x0228,0x0229,0x0045},
  {0x022A,0x022B,0x004F},  {0x022A,0x022B,0x004F},
  {0x022C,0x022D,0x004F},  {0x022C,0x022D,0x004F},
  {0x022E,0x022F,0x004F},  {0x022E,0x022F,0x004F},
  {0x0230,0x0231,0x004F},  {0x0230,0x0231,0x004F},
  {0x0232,0x0233,0x0059},  {0x0232,0x0233,0x0059},
  {0x0234,0x0234,0x0234},  {0x0235,0x0235,0x0235},
  {0x0236,0x0236,0x0236},  {0x0237,0x0237,0x0237},
  {0x0238,0x0238,0x0238},  {0x0239,0x0239,0x0239},
  {0x023A,0x023A,0x023A},  {0x023B,0x023B,0x023B},
  {0x023C,0x023C,0x023C},  {0x023D,0x023D,0x023D},
  {0x023E,0x023E,0x023E},  {0x023F,0x023F,0x023F},
  {0x0240,0x0240,0x0240},  {0x0241,0x0241,0x0241},
  {0x0242,0x0242,0x0242},  {0x0243,0x0243,0x0243},
  {0x0244,0x0244,0x0244},  {0x0245,0x0245,0x0245},
  {0x0246,0x0246,0x0246},  {0x0247,0x0247,0x0247},
  {0x0248,0x0248,0x0248},  {0x0249,0x0249,0x0249},
  {0x024A,0x024A,0x024A},  {0x024B,0x024B,0x024B},
  {0x024C,0x024C,0x024C},  {0x024D,0x024D,0x024D},
  {0x024E,0x024E,0x024E},  {0x024F,0x024F,0x024F},
  {0x0250,0x0250,0x0250},  {0x0251,0x0251,0x0251},
  {0x0252,0x0252,0x0252},  {0x0181,0x0253,0x0181},
  {0x0186,0x0254,0x0186},  {0x0255,0x0255,0x0255},
  {0x0189,0x0256,0x0189},  {0x018A,0x0257,0x018A},
  {0x0258,0x0258,0x0258},  {0x018F,0x0259,0x018F},
  {0x025A,0x025A,0x025A},  {0x0190,0x025B,0x0190},
  {0x025C,0x025C,0x025C},  {0x025D,0x025D,0x025D},
  {0x025E,0x025E,0x025E},  {0x025F,0x025F,0x025F},
  {0x0193,0x0260,0x0193},  {0x0261,0x0261,0x0261},
  {0x0262,0x0262,0x0262},  {0x0194,0x0263,0x0194},
  {0x0264,0x0264,0x0264},  {0x0265,0x0265,0x0265},
  {0x0266,0x0266,0x0266},  {0x0267,0x0267,0x0267},
  {0x0197,0x0268,0x0197},  {0x0196,0x0269,0x0196},
  {0x026A,0x026A,0x026A},  {0x026B,0x026B,0x026B},
  {0x026C,0x026C,0x026C},  {0x026D,0x026D,0x026D},
  {0x026E,0x026E,0x026E},  {0x019C,0x026F,0x019C},
  {0x0270,0x0270,0x0270},  {0x0271,0x0271,0x0271},
  {0x019D,0x0272,0x019D},  {0x0273,0x0273,0x0273},
  {0x0274,0x0274,0x0274},  {0x019F,0x0275,0x019F},
  {0x0276,0x0276,0x0276},  {0x0277,0x0277,0x0277},
  {0x0278,0x0278,0x0278},  {0x0279,0x0279,0x0279},
  {0x027A,0x027A,0x027A},  {0x027B,0x027B,0x027B},
  {0x027C,0x027C,0x027C},  {0x027D,0x027D,0x027D},
  {0x027E,0x027E,0x027E},  {0x027F,0x027F,0x027F},
  {0x01A6,0x0280,0x01A6},  {0x0281,0x0281,0x0281},
  {0x0282,0x0282,0x0282},  {0x01A9,0x0283,0x01A9},
  {0x0284,0x0284,0x0284},  {0x0285,0x0285,0x0285},
  {0x0286,0x0286,0x0286},  {0x0287,0x0287,0x0287},
  {0x01AE,0x0288,0x01AE},  {0x0289,0x0289,0x0289},
  {0x01B1,0x028A,0x01B1},  {0x01B2,0x028B,0x01B2},
  {0x028C,0x028C,0x028C},  {0x028D,0x028D,0x028D},
  {0x028E,0x028E,0x028E},  {0x028F,0x028F,0x028F},
  {0x0290,0x0290,0x0290},  {0x0291,0x0291,0x0291},
  {0x01B7,0x0292,0x01B7},  {0x0293,0x0293,0x0293},
  {0x0294,0x0294,0x0294},  {0x0295,0x0295,0x0295},
  {0x0296,0x0296,0x0296},  {0x0297,0x0297,0x0297},
  {0x0298,0x0298,0x0298},  {0x0299,0x0299,0x0299},
  {0x029A,0x029A,0x029A},  {0x029B,0x029B,0x029B},
  {0x029C,0x029C,0x029C},  {0x029D,0x029D,0x029D},
  {0x029E,0x029E,0x029E},  {0x029F,0x029F,0x029F},
  {0x02A0,0x02A0,0x02A0},  {0x02A1,0x02A1,0x02A1},
  {0x02A2,0x02A2,0x02A2},  {0x02A3,0x02A3,0x02A3},
  {0x02A4,0x02A4,0x02A4},  {0x02A5,0x02A5,0x02A5},
  {0x02A6,0x02A6,0x02A6},  {0x02A7,0x02A7,0x02A7},
  {0x02A8,0x02A8,0x02A8},  {0x02A9,0x02A9,0x02A9},
  {0x02AA,0x02AA,0x02AA},  {0x02AB,0x02AB,0x02AB},
  {0x02AC,0x02AC,0x02AC},  {0x02AD,0x02AD,0x02AD},
  {0x02AE,0x02AE,0x02AE},  {0x02AF,0x02AF,0x02AF},
  {0x02B0,0x02B0,0x02B0},  {0x02B1,0x02B1,0x02B1},
  {0x02B2,0x02B2,0x02B2},  {0x02B3,0x02B3,0x02B3},
  {0x02B4,0x02B4,0x02B4},  {0x02B5,0x02B5,0x02B5},
  {0x02B6,0x02B6,0x02B6},  {0x02B7,0x02B7,0x02B7},
  {0x02B8,0x02B8,0x02B8},  {0x02B9,0x02B9,0x02B9},
  {0x02BA,0x02BA,0x02BA},  {0x02BB,0x02BB,0x02BB},
  {0x02BC,0x02BC,0x02BC},  {0x02BD,0x02BD,0x02BD},
  {0x02BE,0x02BE,0x02BE},  {0x02BF,0x02BF,0x02BF},
  {0x02C0,0x02C0,0x02C0},  {0x02C1,0x02C1,0x02C1},
  {0x02C2,0x02C2,0x02C2},  {0x02C3,0x02C3,0x02C3},
  {0x02C4,0x02C4,0x02C4},  {0x02C5,0x02C5,0x02C5},
  {0x02C6,0x02C6,0x02C6},  {0x02C7,0x02C7,0x02C7},
  {0x02C8,0x02C8,0x02C8},  {0x02C9,0x02C9,0x02C9},
  {0x02CA,0x02CA,0x02CA},  {0x02CB,0x02CB,0x02CB},
  {0x02CC,0x02CC,0x02CC},  {0x02CD,0x02CD,0x02CD},
  {0x02CE,0x02CE,0x02CE},  {0x02CF,0x02CF,0x02CF},
  {0x02D0,0x02D0,0x02D0},  {0x02D1,0x02D1,0x02D1},
  {0x02D2,0x02D2,0x02D2},  {0x02D3,0x02D3,0x02D3},
  {0x02D4,0x02D4,0x02D4},  {0x02D5,0x02D5,0x02D5},
  {0x02D6,0x02D6,0x02D6},  {0x02D7,0x02D7,0x02D7},
  {0x02D8,0x02D8,0x02D8},  {0x02D9,0x02D9,0x02D9},
  {0x02DA,0x02DA,0x02DA},  {0x02DB,0x02DB,0x02DB},
  {0x02DC,0x02DC,0x02DC},  {0x02DD,0x02DD,0x02DD},
  {0x02DE,0x02DE,0x02DE},  {0x02DF,0x02DF,0x02DF},
  {0x02E0,0x02E0,0x02E0},  {0x02E1,0x02E1,0x02E1},
  {0x02E2,0x02E2,0x02E2},  {0x02E3,0x02E3,0x02E3},
  {0x02E4,0x02E4,0x02E4},  {0x02E5,0x02E5,0x02E5},
  {0x02E6,0x02E6,0x02E6},  {0x02E7,0x02E7,0x02E7},
  {0x02E8,0x02E8,0x02E8},  {0x02E9,0x02E9,0x02E9},
  {0x02EA,0x02EA,0x02EA},  {0x02EB,0x02EB,0x02EB},
  {0x02EC,0x02EC,0x02EC},  {0x02ED,0x02ED,0x02ED},
  {0x02EE,0x02EE,0x02EE},  {0x02EF,0x02EF,0x02EF},
  {0x02F0,0x02F0,0x02F0},  {0x02F1,0x02F1,0x02F1},
  {0x02F2,0x02F2,0x02F2},  {0x02F3,0x02F3,0x02F3},
  {0x02F4,0x02F4,0x02F4},  {0x02F5,0x02F5,0x02F5},
  {0x02F6,0x02F6,0x02F6},  {0x02F7,0x02F7,0x02F7},
  {0x02F8,0x02F8,0x02F8},  {0x02F9,0x02F9,0x02F9},
  {0x02FA,0x02FA,0x02FA},  {0x02FB,0x02FB,0x02FB},
  {0x02FC,0x02FC,0x02FC},  {0x02FD,0x02FD,0x02FD},
  {0x02FE,0x02FE,0x02FE},  {0x02FF,0x02FF,0x02FF}
};

static MY_UNICASE_CHARACTER plane03[]={
  {0x0300,0x0300,0x0300},  {0x0301,0x0301,0x0301},
  {0x0302,0x0302,0x0302},  {0x0303,0x0303,0x0303},
  {0x0304,0x0304,0x0304},  {0x0305,0x0305,0x0305},
  {0x0306,0x0306,0x0306},  {0x0307,0x0307,0x0307},
  {0x0308,0x0308,0x0308},  {0x0309,0x0309,0x0309},
  {0x030A,0x030A,0x030A},  {0x030B,0x030B,0x030B},
  {0x030C,0x030C,0x030C},  {0x030D,0x030D,0x030D},
  {0x030E,0x030E,0x030E},  {0x030F,0x030F,0x030F},
  {0x0310,0x0310,0x0310},  {0x0311,0x0311,0x0311},
  {0x0312,0x0312,0x0312},  {0x0313,0x0313,0x0313},
  {0x0314,0x0314,0x0314},  {0x0315,0x0315,0x0315},
  {0x0316,0x0316,0x0316},  {0x0317,0x0317,0x0317},
  {0x0318,0x0318,0x0318},  {0x0319,0x0319,0x0319},
  {0x031A,0x031A,0x031A},  {0x031B,0x031B,0x031B},
  {0x031C,0x031C,0x031C},  {0x031D,0x031D,0x031D},
  {0x031E,0x031E,0x031E},  {0x031F,0x031F,0x031F},
  {0x0320,0x0320,0x0320},  {0x0321,0x0321,0x0321},
  {0x0322,0x0322,0x0322},  {0x0323,0x0323,0x0323},
  {0x0324,0x0324,0x0324},  {0x0325,0x0325,0x0325},
  {0x0326,0x0326,0x0326},  {0x0327,0x0327,0x0327},
  {0x0328,0x0328,0x0328},  {0x0329,0x0329,0x0329},
  {0x032A,0x032A,0x032A},  {0x032B,0x032B,0x032B},
  {0x032C,0x032C,0x032C},  {0x032D,0x032D,0x032D},
  {0x032E,0x032E,0x032E},  {0x032F,0x032F,0x032F},
  {0x0330,0x0330,0x0330},  {0x0331,0x0331,0x0331},
  {0x0332,0x0332,0x0332},  {0x0333,0x0333,0x0333},
  {0x0334,0x0334,0x0334},  {0x0335,0x0335,0x0335},
  {0x0336,0x0336,0x0336},  {0x0337,0x0337,0x0337},
  {0x0338,0x0338,0x0338},  {0x0339,0x0339,0x0339},
  {0x033A,0x033A,0x033A},  {0x033B,0x033B,0x033B},
  {0x033C,0x033C,0x033C},  {0x033D,0x033D,0x033D},
  {0x033E,0x033E,0x033E},  {0x033F,0x033F,0x033F},
  {0x0340,0x0340,0x0340},  {0x0341,0x0341,0x0341},
  {0x0342,0x0342,0x0342},  {0x0343,0x0343,0x0343},
  {0x0344,0x0344,0x0344},  {0x0399,0x0345,0x0399},
  {0x0346,0x0346,0x0346},  {0x0347,0x0347,0x0347},
  {0x0348,0x0348,0x0348},  {0x0349,0x0349,0x0349},
  {0x034A,0x034A,0x034A},  {0x034B,0x034B,0x034B},
  {0x034C,0x034C,0x034C},  {0x034D,0x034D,0x034D},
  {0x034E,0x034E,0x034E},  {0x034F,0x034F,0x034F},
  {0x0350,0x0350,0x0350},  {0x0351,0x0351,0x0351},
  {0x0352,0x0352,0x0352},  {0x0353,0x0353,0x0353},
  {0x0354,0x0354,0x0354},  {0x0355,0x0355,0x0355},
  {0x0356,0x0356,0x0356},  {0x0357,0x0357,0x0357},
  {0x0358,0x0358,0x0358},  {0x0359,0x0359,0x0359},
  {0x035A,0x035A,0x035A},  {0x035B,0x035B,0x035B},
  {0x035C,0x035C,0x035C},  {0x035D,0x035D,0x035D},
  {0x035E,0x035E,0x035E},  {0x035F,0x035F,0x035F},
  {0x0360,0x0360,0x0360},  {0x0361,0x0361,0x0361},
  {0x0362,0x0362,0x0362},  {0x0363,0x0363,0x0363},
  {0x0364,0x0364,0x0364},  {0x0365,0x0365,0x0365},
  {0x0366,0x0366,0x0366},  {0x0367,0x0367,0x0367},
  {0x0368,0x0368,0x0368},  {0x0369,0x0369,0x0369},
  {0x036A,0x036A,0x036A},  {0x036B,0x036B,0x036B},
  {0x036C,0x036C,0x036C},  {0x036D,0x036D,0x036D},
  {0x036E,0x036E,0x036E},  {0x036F,0x036F,0x036F},
  {0x0370,0x0370,0x0370},  {0x0371,0x0371,0x0371},
  {0x0372,0x0372,0x0372},  {0x0373,0x0373,0x0373},
  {0x0374,0x0374,0x0374},  {0x0375,0x0375,0x0375},
  {0x0376,0x0376,0x0376},  {0x0377,0x0377,0x0377},
  {0x0378,0x0378,0x0378},  {0x0379,0x0379,0x0379},
  {0x037A,0x037A,0x037A},  {0x037B,0x037B,0x037B},
  {0x037C,0x037C,0x037C},  {0x037D,0x037D,0x037D},
  {0x037E,0x037E,0x037E},  {0x037F,0x037F,0x037F},
  {0x0380,0x0380,0x0380},  {0x0381,0x0381,0x0381},
  {0x0382,0x0382,0x0382},  {0x0383,0x0383,0x0383},
  {0x0384,0x0384,0x0384},  {0x0385,0x0385,0x0385},
  {0x0386,0x03AC,0x0391},  {0x0387,0x0387,0x0387},
  {0x0388,0x03AD,0x0395},  {0x0389,0x03AE,0x0397},
  {0x038A,0x03AF,0x0399},  {0x038B,0x038B,0x038B},
  {0x038C,0x03CC,0x039F},  {0x038D,0x038D,0x038D},
  {0x038E,0x03CD,0x03A5},  {0x038F,0x03CE,0x03A9},
  {0x0390,0x0390,0x0399},  {0x0391,0x03B1,0x0391},
  {0x0392,0x03B2,0x0392},  {0x0393,0x03B3,0x0393},
  {0x0394,0x03B4,0x0394},  {0x0395,0x03B5,0x0395},
  {0x0396,0x03B6,0x0396},  {0x0397,0x03B7,0x0397},
  {0x0398,0x03B8,0x0398},  {0x0399,0x03B9,0x0399},
  {0x039A,0x03BA,0x039A},  {0x039B,0x03BB,0x039B},
  {0x039C,0x03BC,0x039C},  {0x039D,0x03BD,0x039D},
  {0x039E,0x03BE,0x039E},  {0x039F,0x03BF,0x039F},
  {0x03A0,0x03C0,0x03A0},  {0x03A1,0x03C1,0x03A1},
  {0x03A2,0x03A2,0x03A2},  {0x03A3,0x03C3,0x03A3},
  {0x03A4,0x03C4,0x03A4},  {0x03A5,0x03C5,0x03A5},
  {0x03A6,0x03C6,0x03A6},  {0x03A7,0x03C7,0x03A7},
  {0x03A8,0x03C8,0x03A8},  {0x03A9,0x03C9,0x03A9},
  {0x03AA,0x03CA,0x0399},  {0x03AB,0x03CB,0x03A5},
  {0x0386,0x03AC,0x0391},  {0x0388,0x03AD,0x0395},
  {0x0389,0x03AE,0x0397},  {0x038A,0x03AF,0x0399},
  {0x03B0,0x03B0,0x03A5},  {0x0391,0x03B1,0x0391},
  {0x0392,0x03B2,0x0392},  {0x0393,0x03B3,0x0393},
  {0x0394,0x03B4,0x0394},  {0x0395,0x03B5,0x0395},
  {0x0396,0x03B6,0x0396},  {0x0397,0x03B7,0x0397},
  {0x0398,0x03B8,0x0398},  {0x0399,0x03B9,0x0399},
  {0x039A,0x03BA,0x039A},  {0x039B,0x03BB,0x039B},
  {0x039C,0x03BC,0x039C},  {0x039D,0x03BD,0x039D},
  {0x039E,0x03BE,0x039E},  {0x039F,0x03BF,0x039F},
  {0x03A0,0x03C0,0x03A0},  {0x03A1,0x03C1,0x03A1},
  {0x03A3,0x03C2,0x03A3},  {0x03A3,0x03C3,0x03A3},
  {0x03A4,0x03C4,0x03A4},  {0x03A5,0x03C5,0x03A5},
  {0x03A6,0x03C6,0x03A6},  {0x03A7,0x03C7,0x03A7},
  {0x03A8,0x03C8,0x03A8},  {0x03A9,0x03C9,0x03A9},
  {0x03AA,0x03CA,0x0399},  {0x03AB,0x03CB,0x03A5},
  {0x038C,0x03CC,0x039F},  {0x038E,0x03CD,0x03A5},
  {0x038F,0x03CE,0x03A9},  {0x03CF,0x03CF,0x03CF},
  {0x0392,0x03D0,0x0392},  {0x0398,0x03D1,0x0398},
  {0x03D2,0x03D2,0x03D2},  {0x03D3,0x03D3,0x03D2},
  {0x03D4,0x03D4,0x03D2},  {0x03A6,0x03D5,0x03A6},
  {0x03A0,0x03D6,0x03A0},  {0x03D7,0x03D7,0x03D7},
  {0x03D8,0x03D8,0x03D8},  {0x03D9,0x03D9,0x03D9},
  {0x03DA,0x03DB,0x03DA},  {0x03DA,0x03DB,0x03DA},
  {0x03DC,0x03DD,0x03DC},  {0x03DC,0x03DD,0x03DC},
  {0x03DE,0x03DF,0x03DE},  {0x03DE,0x03DF,0x03DE},
  {0x03E0,0x03E1,0x03E0},  {0x03E0,0x03E1,0x03E0},
  {0x03E2,0x03E3,0x03E2},  {0x03E2,0x03E3,0x03E2},
  {0x03E4,0x03E5,0x03E4},  {0x03E4,0x03E5,0x03E4},
  {0x03E6,0x03E7,0x03E6},  {0x03E6,0x03E7,0x03E6},
  {0x03E8,0x03E9,0x03E8},  {0x03E8,0x03E9,0x03E8},
  {0x03EA,0x03EB,0x03EA},  {0x03EA,0x03EB,0x03EA},
  {0x03EC,0x03ED,0x03EC},  {0x03EC,0x03ED,0x03EC},
  {0x03EE,0x03EF,0x03EE},  {0x03EE,0x03EF,0x03EE},
  {0x039A,0x03F0,0x039A},  {0x03A1,0x03F1,0x03A1},
  {0x03A3,0x03F2,0x03A3},  {0x03F3,0x03F3,0x03F3},
  {0x03F4,0x03F4,0x03F4},  {0x03F5,0x03F5,0x03F5},
  {0x03F6,0x03F6,0x03F6},  {0x03F7,0x03F7,0x03F7},
  {0x03F8,0x03F8,0x03F8},  {0x03F9,0x03F9,0x03F9},
  {0x03FA,0x03FA,0x03FA},  {0x03FB,0x03FB,0x03FB},
  {0x03FC,0x03FC,0x03FC},  {0x03FD,0x03FD,0x03FD},
  {0x03FE,0x03FE,0x03FE},  {0x03FF,0x03FF,0x03FF}
};

static MY_UNICASE_CHARACTER plane04[]={
  {0x0400,0x0450,0x0415},  {0x0401,0x0451,0x0415},
  {0x0402,0x0452,0x0402},  {0x0403,0x0453,0x0413},
  {0x0404,0x0454,0x0404},  {0x0405,0x0455,0x0405},
  {0x0406,0x0456,0x0406},  {0x0407,0x0457,0x0406},
  {0x0408,0x0458,0x0408},  {0x0409,0x0459,0x0409},
  {0x040A,0x045A,0x040A},  {0x040B,0x045B,0x040B},
  {0x040C,0x045C,0x041A},  {0x040D,0x045D,0x0418},
  {0x040E,0x045E,0x0423},  {0x040F,0x045F,0x040F},
  {0x0410,0x0430,0x0410},  {0x0411,0x0431,0x0411},
  {0x0412,0x0432,0x0412},  {0x0413,0x0433,0x0413},
  {0x0414,0x0434,0x0414},  {0x0415,0x0435,0x0415},
  {0x0416,0x0436,0x0416},  {0x0417,0x0437,0x0417},
  {0x0418,0x0438,0x0418},  {0x0419,0x0439,0x0419},
  {0x041A,0x043A,0x041A},  {0x041B,0x043B,0x041B},
  {0x041C,0x043C,0x041C},  {0x041D,0x043D,0x041D},
  {0x041E,0x043E,0x041E},  {0x041F,0x043F,0x041F},
  {0x0420,0x0440,0x0420},  {0x0421,0x0441,0x0421},
  {0x0422,0x0442,0x0422},  {0x0423,0x0443,0x0423},
  {0x0424,0x0444,0x0424},  {0x0425,0x0445,0x0425},
  {0x0426,0x0446,0x0426},  {0x0427,0x0447,0x0427},
  {0x0428,0x0448,0x0428},  {0x0429,0x0449,0x0429},
  {0x042A,0x044A,0x042A},  {0x042B,0x044B,0x042B},
  {0x042C,0x044C,0x042C},  {0x042D,0x044D,0x042D},
  {0x042E,0x044E,0x042E},  {0x042F,0x044F,0x042F},
  {0x0410,0x0430,0x0410},  {0x0411,0x0431,0x0411},
  {0x0412,0x0432,0x0412},  {0x0413,0x0433,0x0413},
  {0x0414,0x0434,0x0414},  {0x0415,0x0435,0x0415},
  {0x0416,0x0436,0x0416},  {0x0417,0x0437,0x0417},
  {0x0418,0x0438,0x0418},  {0x0419,0x0439,0x0419},
  {0x041A,0x043A,0x041A},  {0x041B,0x043B,0x041B},
  {0x041C,0x043C,0x041C},  {0x041D,0x043D,0x041D},
  {0x041E,0x043E,0x041E},  {0x041F,0x043F,0x041F},
  {0x0420,0x0440,0x0420},  {0x0421,0x0441,0x0421},
  {0x0422,0x0442,0x0422},  {0x0423,0x0443,0x0423},
  {0x0424,0x0444,0x0424},  {0x0425,0x0445,0x0425},
  {0x0426,0x0446,0x0426},  {0x0427,0x0447,0x0427},
  {0x0428,0x0448,0x0428},  {0x0429,0x0449,0x0429},
  {0x042A,0x044A,0x042A},  {0x042B,0x044B,0x042B},
  {0x042C,0x044C,0x042C},  {0x042D,0x044D,0x042D},
  {0x042E,0x044E,0x042E},  {0x042F,0x044F,0x042F},
  {0x0400,0x0450,0x0415},  {0x0401,0x0451,0x0415},
  {0x0402,0x0452,0x0402},  {0x0403,0x0453,0x0413},
  {0x0404,0x0454,0x0404},  {0x0405,0x0455,0x0405},
  {0x0406,0x0456,0x0406},  {0x0407,0x0457,0x0406},
  {0x0408,0x0458,0x0408},  {0x0409,0x0459,0x0409},
  {0x040A,0x045A,0x040A},  {0x040B,0x045B,0x040B},
  {0x040C,0x045C,0x041A},  {0x040D,0x045D,0x0418},
  {0x040E,0x045E,0x0423},  {0x040F,0x045F,0x040F},
  {0x0460,0x0461,0x0460},  {0x0460,0x0461,0x0460},
  {0x0462,0x0463,0x0462},  {0x0462,0x0463,0x0462},
  {0x0464,0x0465,0x0464},  {0x0464,0x0465,0x0464},
  {0x0466,0x0467,0x0466},  {0x0466,0x0467,0x0466},
  {0x0468,0x0469,0x0468},  {0x0468,0x0469,0x0468},
  {0x046A,0x046B,0x046A},  {0x046A,0x046B,0x046A},
  {0x046C,0x046D,0x046C},  {0x046C,0x046D,0x046C},
  {0x046E,0x046F,0x046E},  {0x046E,0x046F,0x046E},
  {0x0470,0x0471,0x0470},  {0x0470,0x0471,0x0470},
  {0x0472,0x0473,0x0472},  {0x0472,0x0473,0x0472},
  {0x0474,0x0475,0x0474},  {0x0474,0x0475,0x0474},
  {0x0476,0x0477,0x0474},  {0x0476,0x0477,0x0474},
  {0x0478,0x0479,0x0478},  {0x0478,0x0479,0x0478},
  {0x047A,0x047B,0x047A},  {0x047A,0x047B,0x047A},
  {0x047C,0x047D,0x047C},  {0x047C,0x047D,0x047C},
  {0x047E,0x047F,0x047E},  {0x047E,0x047F,0x047E},
  {0x0480,0x0481,0x0480},  {0x0480,0x0481,0x0480},
  {0x0482,0x0482,0x0482},  {0x0483,0x0483,0x0483},
  {0x0484,0x0484,0x0484},  {0x0485,0x0485,0x0485},
  {0x0486,0x0486,0x0486},  {0x0487,0x0487,0x0487},
  {0x0488,0x0488,0x0488},  {0x0489,0x0489,0x0489},
  {0x048A,0x048A,0x048A},  {0x048B,0x048B,0x048B},
  {0x048C,0x048D,0x048C},  {0x048C,0x048D,0x048C},
  {0x048E,0x048F,0x048E},  {0x048E,0x048F,0x048E},
  {0x0490,0x0491,0x0490},  {0x0490,0x0491,0x0490},
  {0x0492,0x0493,0x0492},  {0x0492,0x0493,0x0492},
  {0x0494,0x0495,0x0494},  {0x0494,0x0495,0x0494},
  {0x0496,0x0497,0x0496},  {0x0496,0x0497,0x0496},
  {0x0498,0x0499,0x0498},  {0x0498,0x0499,0x0498},
  {0x049A,0x049B,0x049A},  {0x049A,0x049B,0x049A},
  {0x049C,0x049D,0x049C},  {0x049C,0x049D,0x049C},
  {0x049E,0x049F,0x049E},  {0x049E,0x049F,0x049E},
  {0x04A0,0x04A1,0x04A0},  {0x04A0,0x04A1,0x04A0},
  {0x04A2,0x04A3,0x04A2},  {0x04A2,0x04A3,0x04A2},
  {0x04A4,0x04A5,0x04A4},  {0x04A4,0x04A5,0x04A4},
  {0x04A6,0x04A7,0x04A6},  {0x04A6,0x04A7,0x04A6},
  {0x04A8,0x04A9,0x04A8},  {0x04A8,0x04A9,0x04A8},
  {0x04AA,0x04AB,0x04AA},  {0x04AA,0x04AB,0x04AA},
  {0x04AC,0x04AD,0x04AC},  {0x04AC,0x04AD,0x04AC},
  {0x04AE,0x04AF,0x04AE},  {0x04AE,0x04AF,0x04AE},
  {0x04B0,0x04B1,0x04B0},  {0x04B0,0x04B1,0x04B0},
  {0x04B2,0x04B3,0x04B2},  {0x04B2,0x04B3,0x04B2},
  {0x04B4,0x04B5,0x04B4},  {0x04B4,0x04B5,0x04B4},
  {0x04B6,0x04B7,0x04B6},  {0x04B6,0x04B7,0x04B6},
  {0x04B8,0x04B9,0x04B8},  {0x04B8,0x04B9,0x04B8},
  {0x04BA,0x04BB,0x04BA},  {0x04BA,0x04BB,0x04BA},
  {0x04BC,0x04BD,0x04BC},  {0x04BC,0x04BD,0x04BC},
  {0x04BE,0x04BF,0x04BE},  {0x04BE,0x04BF,0x04BE},
  {0x04C0,0x04C0,0x04C0},  {0x04C1,0x04C2,0x0416},
  {0x04C1,0x04C2,0x0416},  {0x04C3,0x04C4,0x04C3},
  {0x04C3,0x04C4,0x04C3},  {0x04C5,0x04C5,0x04C5},
  {0x04C6,0x04C6,0x04C6},  {0x04C7,0x04C8,0x04C7},
  {0x04C7,0x04C8,0x04C7},  {0x04C9,0x04C9,0x04C9},
  {0x04CA,0x04CA,0x04CA},  {0x04CB,0x04CC,0x04CB},
  {0x04CB,0x04CC,0x04CB},  {0x04CD,0x04CD,0x04CD},
  {0x04CE,0x04CE,0x04CE},  {0x04CF,0x04CF,0x04CF},
  {0x04D0,0x04D1,0x0410},  {0x04D0,0x04D1,0x0410},
  {0x04D2,0x04D3,0x0410},  {0x04D2,0x04D3,0x0410},
  {0x04D4,0x04D5,0x04D4},  {0x04D4,0x04D5,0x04D4},
  {0x04D6,0x04D7,0x0415},  {0x04D6,0x04D7,0x0415},
  {0x04D8,0x04D9,0x04D8},  {0x04D8,0x04D9,0x04D8},
  {0x04DA,0x04DB,0x04D8},  {0x04DA,0x04DB,0x04D8},
  {0x04DC,0x04DD,0x0416},  {0x04DC,0x04DD,0x0416},
  {0x04DE,0x04DF,0x0417},  {0x04DE,0x04DF,0x0417},
  {0x04E0,0x04E1,0x04E0},  {0x04E0,0x04E1,0x04E0},
  {0x04E2,0x04E3,0x0418},  {0x04E2,0x04E3,0x0418},
  {0x04E4,0x04E5,0x0418},  {0x04E4,0x04E5,0x0418},
  {0x04E6,0x04E7,0x041E},  {0x04E6,0x04E7,0x041E},
  {0x04E8,0x04E9,0x04E8},  {0x04E8,0x04E9,0x04E8},
  {0x04EA,0x04EB,0x04E8},  {0x04EA,0x04EB,0x04E8},
  {0x04EC,0x04ED,0x042D},  {0x04EC,0x04ED,0x042D},
  {0x04EE,0x04EF,0x0423},  {0x04EE,0x04EF,0x0423},
  {0x04F0,0x04F1,0x0423},  {0x04F0,0x04F1,0x0423},
  {0x04F2,0x04F3,0x0423},  {0x04F2,0x04F3,0x0423},
  {0x04F4,0x04F5,0x0427},  {0x04F4,0x04F5,0x0427},
  {0x04F6,0x04F6,0x04F6},  {0x04F7,0x04F7,0x04F7},
  {0x04F8,0x04F9,0x042B},  {0x04F8,0x04F9,0x042B},
  {0x04FA,0x04FA,0x04FA},  {0x04FB,0x04FB,0x04FB},
  {0x04FC,0x04FC,0x04FC},  {0x04FD,0x04FD,0x04FD},
  {0x04FE,0x04FE,0x04FE},  {0x04FF,0x04FF,0x04FF}
};

static MY_UNICASE_CHARACTER plane05[]={
  {0x0500,0x0500,0x0500},  {0x0501,0x0501,0x0501},
  {0x0502,0x0502,0x0502},  {0x0503,0x0503,0x0503},
  {0x0504,0x0504,0x0504},  {0x0505,0x0505,0x0505},
  {0x0506,0x0506,0x0506},  {0x0507,0x0507,0x0507},
  {0x0508,0x0508,0x0508},  {0x0509,0x0509,0x0509},
  {0x050A,0x050A,0x050A},  {0x050B,0x050B,0x050B},
  {0x050C,0x050C,0x050C},  {0x050D,0x050D,0x050D},
  {0x050E,0x050E,0x050E},  {0x050F,0x050F,0x050F},
  {0x0510,0x0510,0x0510},  {0x0511,0x0511,0x0511},
  {0x0512,0x0512,0x0512},  {0x0513,0x0513,0x0513},
  {0x0514,0x0514,0x0514},  {0x0515,0x0515,0x0515},
  {0x0516,0x0516,0x0516},  {0x0517,0x0517,0x0517},
  {0x0518,0x0518,0x0518},  {0x0519,0x0519,0x0519},
  {0x051A,0x051A,0x051A},  {0x051B,0x051B,0x051B},
  {0x051C,0x051C,0x051C},  {0x051D,0x051D,0x051D},
  {0x051E,0x051E,0x051E},  {0x051F,0x051F,0x051F},
  {0x0520,0x0520,0x0520},  {0x0521,0x0521,0x0521},
  {0x0522,0x0522,0x0522},  {0x0523,0x0523,0x0523},
  {0x0524,0x0524,0x0524},  {0x0525,0x0525,0x0525},
  {0x0526,0x0526,0x0526},  {0x0527,0x0527,0x0527},
  {0x0528,0x0528,0x0528},  {0x0529,0x0529,0x0529},
  {0x052A,0x052A,0x052A},  {0x052B,0x052B,0x052B},
  {0x052C,0x052C,0x052C},  {0x052D,0x052D,0x052D},
  {0x052E,0x052E,0x052E},  {0x052F,0x052F,0x052F},
  {0x0530,0x0530,0x0530},  {0x0531,0x0561,0x0531},
  {0x0532,0x0562,0x0532},  {0x0533,0x0563,0x0533},
  {0x0534,0x0564,0x0534},  {0x0535,0x0565,0x0535},
  {0x0536,0x0566,0x0536},  {0x0537,0x0567,0x0537},
  {0x0538,0x0568,0x0538},  {0x0539,0x0569,0x0539},
  {0x053A,0x056A,0x053A},  {0x053B,0x056B,0x053B},
  {0x053C,0x056C,0x053C},  {0x053D,0x056D,0x053D},
  {0x053E,0x056E,0x053E},  {0x053F,0x056F,0x053F},
  {0x0540,0x0570,0x0540},  {0x0541,0x0571,0x0541},
  {0x0542,0x0572,0x0542},  {0x0543,0x0573,0x0543},
  {0x0544,0x0574,0x0544},  {0x0545,0x0575,0x0545},
  {0x0546,0x0576,0x0546},  {0x0547,0x0577,0x0547},
  {0x0548,0x0578,0x0548},  {0x0549,0x0579,0x0549},
  {0x054A,0x057A,0x054A},  {0x054B,0x057B,0x054B},
  {0x054C,0x057C,0x054C},  {0x054D,0x057D,0x054D},
  {0x054E,0x057E,0x054E},  {0x054F,0x057F,0x054F},
  {0x0550,0x0580,0x0550},  {0x0551,0x0581,0x0551},
  {0x0552,0x0582,0x0552},  {0x0553,0x0583,0x0553},
  {0x0554,0x0584,0x0554},  {0x0555,0x0585,0x0555},
  {0x0556,0x0586,0x0556},  {0x0557,0x0557,0x0557},
  {0x0558,0x0558,0x0558},  {0x0559,0x0559,0x0559},
  {0x055A,0x055A,0x055A},  {0x055B,0x055B,0x055B},
  {0x055C,0x055C,0x055C},  {0x055D,0x055D,0x055D},
  {0x055E,0x055E,0x055E},  {0x055F,0x055F,0x055F},
  {0x0560,0x0560,0x0560},  {0x0531,0x0561,0x0531},
  {0x0532,0x0562,0x0532},  {0x0533,0x0563,0x0533},
  {0x0534,0x0564,0x0534},  {0x0535,0x0565,0x0535},
  {0x0536,0x0566,0x0536},  {0x0537,0x0567,0x0537},
  {0x0538,0x0568,0x0538},  {0x0539,0x0569,0x0539},
  {0x053A,0x056A,0x053A},  {0x053B,0x056B,0x053B},
  {0x053C,0x056C,0x053C},  {0x053D,0x056D,0x053D},
  {0x053E,0x056E,0x053E},  {0x053F,0x056F,0x053F},
  {0x0540,0x0570,0x0540},  {0x0541,0x0571,0x0541},
  {0x0542,0x0572,0x0542},  {0x0543,0x0573,0x0543},
  {0x0544,0x0574,0x0544},  {0x0545,0x0575,0x0545},
  {0x0546,0x0576,0x0546},  {0x0547,0x0577,0x0547},
  {0x0548,0x0578,0x0548},  {0x0549,0x0579,0x0549},
  {0x054A,0x057A,0x054A},  {0x054B,0x057B,0x054B},
  {0x054C,0x057C,0x054C},  {0x054D,0x057D,0x054D},
  {0x054E,0x057E,0x054E},  {0x054F,0x057F,0x054F},
  {0x0550,0x0580,0x0550},  {0x0551,0x0581,0x0551},
  {0x0552,0x0582,0x0552},  {0x0553,0x0583,0x0553},
  {0x0554,0x0584,0x0554},  {0x0555,0x0585,0x0555},
  {0x0556,0x0586,0x0556},  {0x0587,0x0587,0x0587},
  {0x0588,0x0588,0x0588},  {0x0589,0x0589,0x0589},
  {0x058A,0x058A,0x058A},  {0x058B,0x058B,0x058B},
  {0x058C,0x058C,0x058C},  {0x058D,0x058D,0x058D},
  {0x058E,0x058E,0x058E},  {0x058F,0x058F,0x058F},
  {0x0590,0x0590,0x0590},  {0x0591,0x0591,0x0591},
  {0x0592,0x0592,0x0592},  {0x0593,0x0593,0x0593},
  {0x0594,0x0594,0x0594},  {0x0595,0x0595,0x0595},
  {0x0596,0x0596,0x0596},  {0x0597,0x0597,0x0597},
  {0x0598,0x0598,0x0598},  {0x0599,0x0599,0x0599},
  {0x059A,0x059A,0x059A},  {0x059B,0x059B,0x059B},
  {0x059C,0x059C,0x059C},  {0x059D,0x059D,0x059D},
  {0x059E,0x059E,0x059E},  {0x059F,0x059F,0x059F},
  {0x05A0,0x05A0,0x05A0},  {0x05A1,0x05A1,0x05A1},
  {0x05A2,0x05A2,0x05A2},  {0x05A3,0x05A3,0x05A3},
  {0x05A4,0x05A4,0x05A4},  {0x05A5,0x05A5,0x05A5},
  {0x05A6,0x05A6,0x05A6},  {0x05A7,0x05A7,0x05A7},
  {0x05A8,0x05A8,0x05A8},  {0x05A9,0x05A9,0x05A9},
  {0x05AA,0x05AA,0x05AA},  {0x05AB,0x05AB,0x05AB},
  {0x05AC,0x05AC,0x05AC},  {0x05AD,0x05AD,0x05AD},
  {0x05AE,0x05AE,0x05AE},  {0x05AF,0x05AF,0x05AF},
  {0x05B0,0x05B0,0x05B0},  {0x05B1,0x05B1,0x05B1},
  {0x05B2,0x05B2,0x05B2},  {0x05B3,0x05B3,0x05B3},
  {0x05B4,0x05B4,0x05B4},  {0x05B5,0x05B5,0x05B5},
  {0x05B6,0x05B6,0x05B6},  {0x05B7,0x05B7,0x05B7},
  {0x05B8,0x05B8,0x05B8},  {0x05B9,0x05B9,0x05B9},
  {0x05BA,0x05BA,0x05BA},  {0x05BB,0x05BB,0x05BB},
  {0x05BC,0x05BC,0x05BC},  {0x05BD,0x05BD,0x05BD},
  {0x05BE,0x05BE,0x05BE},  {0x05BF,0x05BF,0x05BF},
  {0x05C0,0x05C0,0x05C0},  {0x05C1,0x05C1,0x05C1},
  {0x05C2,0x05C2,0x05C2},  {0x05C3,0x05C3,0x05C3},
  {0x05C4,0x05C4,0x05C4},  {0x05C5,0x05C5,0x05C5},
  {0x05C6,0x05C6,0x05C6},  {0x05C7,0x05C7,0x05C7},
  {0x05C8,0x05C8,0x05C8},  {0x05C9,0x05C9,0x05C9},
  {0x05CA,0x05CA,0x05CA},  {0x05CB,0x05CB,0x05CB},
  {0x05CC,0x05CC,0x05CC},  {0x05CD,0x05CD,0x05CD},
  {0x05CE,0x05CE,0x05CE},  {0x05CF,0x05CF,0x05CF},
  {0x05D0,0x05D0,0x05D0},  {0x05D1,0x05D1,0x05D1},
  {0x05D2,0x05D2,0x05D2},  {0x05D3,0x05D3,0x05D3},
  {0x05D4,0x05D4,0x05D4},  {0x05D5,0x05D5,0x05D5},
  {0x05D6,0x05D6,0x05D6},  {0x05D7,0x05D7,0x05D7},
  {0x05D8,0x05D8,0x05D8},  {0x05D9,0x05D9,0x05D9},
  {0x05DA,0x05DA,0x05DA},  {0x05DB,0x05DB,0x05DB},
  {0x05DC,0x05DC,0x05DC},  {0x05DD,0x05DD,0x05DD},
  {0x05DE,0x05DE,0x05DE},  {0x05DF,0x05DF,0x05DF},
  {0x05E0,0x05E0,0x05E0},  {0x05E1,0x05E1,0x05E1},
  {0x05E2,0x05E2,0x05E2},  {0x05E3,0x05E3,0x05E3},
  {0x05E4,0x05E4,0x05E4},  {0x05E5,0x05E5,0x05E5},
  {0x05E6,0x05E6,0x05E6},  {0x05E7,0x05E7,0x05E7},
  {0x05E8,0x05E8,0x05E8},  {0x05E9,0x05E9,0x05E9},
  {0x05EA,0x05EA,0x05EA},  {0x05EB,0x05EB,0x05EB},
  {0x05EC,0x05EC,0x05EC},  {0x05ED,0x05ED,0x05ED},
  {0x05EE,0x05EE,0x05EE},  {0x05EF,0x05EF,0x05EF},
  {0x05F0,0x05F0,0x05F0},  {0x05F1,0x05F1,0x05F1},
  {0x05F2,0x05F2,0x05F2},  {0x05F3,0x05F3,0x05F3},
  {0x05F4,0x05F4,0x05F4},  {0x05F5,0x05F5,0x05F5},
  {0x05F6,0x05F6,0x05F6},  {0x05F7,0x05F7,0x05F7},
  {0x05F8,0x05F8,0x05F8},  {0x05F9,0x05F9,0x05F9},
  {0x05FA,0x05FA,0x05FA},  {0x05FB,0x05FB,0x05FB},
  {0x05FC,0x05FC,0x05FC},  {0x05FD,0x05FD,0x05FD},
  {0x05FE,0x05FE,0x05FE},  {0x05FF,0x05FF,0x05FF}
};

static MY_UNICASE_CHARACTER plane1E[]={
  {0x1E00,0x1E01,0x0041},  {0x1E00,0x1E01,0x0041},
  {0x1E02,0x1E03,0x0042},  {0x1E02,0x1E03,0x0042},
  {0x1E04,0x1E05,0x0042},  {0x1E04,0x1E05,0x0042},
  {0x1E06,0x1E07,0x0042},  {0x1E06,0x1E07,0x0042},
  {0x1E08,0x1E09,0x0043},  {0x1E08,0x1E09,0x0043},
  {0x1E0A,0x1E0B,0x0044},  {0x1E0A,0x1E0B,0x0044},
  {0x1E0C,0x1E0D,0x0044},  {0x1E0C,0x1E0D,0x0044},
  {0x1E0E,0x1E0F,0x0044},  {0x1E0E,0x1E0F,0x0044},
  {0x1E10,0x1E11,0x0044},  {0x1E10,0x1E11,0x0044},
  {0x1E12,0x1E13,0x0044},  {0x1E12,0x1E13,0x0044},
  {0x1E14,0x1E15,0x0045},  {0x1E14,0x1E15,0x0045},
  {0x1E16,0x1E17,0x0045},  {0x1E16,0x1E17,0x0045},
  {0x1E18,0x1E19,0x0045},  {0x1E18,0x1E19,0x0045},
  {0x1E1A,0x1E1B,0x0045},  {0x1E1A,0x1E1B,0x0045},
  {0x1E1C,0x1E1D,0x0045},  {0x1E1C,0x1E1D,0x0045},
  {0x1E1E,0x1E1F,0x0046},  {0x1E1E,0x1E1F,0x0046},
  {0x1E20,0x1E21,0x0047},  {0x1E20,0x1E21,0x0047},
  {0x1E22,0x1E23,0x0048},  {0x1E22,0x1E23,0x0048},
  {0x1E24,0x1E25,0x0048},  {0x1E24,0x1E25,0x0048},
  {0x1E26,0x1E27,0x0048},  {0x1E26,0x1E27,0x0048},
  {0x1E28,0x1E29,0x0048},  {0x1E28,0x1E29,0x0048},
  {0x1E2A,0x1E2B,0x0048},  {0x1E2A,0x1E2B,0x0048},
  {0x1E2C,0x1E2D,0x0049},  {0x1E2C,0x1E2D,0x0049},
  {0x1E2E,0x1E2F,0x0049},  {0x1E2E,0x1E2F,0x0049},
  {0x1E30,0x1E31,0x004B},  {0x1E30,0x1E31,0x004B},
  {0x1E32,0x1E33,0x004B},  {0x1E32,0x1E33,0x004B},
  {0x1E34,0x1E35,0x004B},  {0x1E34,0x1E35,0x004B},
  {0x1E36,0x1E37,0x004C},  {0x1E36,0x1E37,0x004C},
  {0x1E38,0x1E39,0x004C},  {0x1E38,0x1E39,0x004C},
  {0x1E3A,0x1E3B,0x004C},  {0x1E3A,0x1E3B,0x004C},
  {0x1E3C,0x1E3D,0x004C},  {0x1E3C,0x1E3D,0x004C},
  {0x1E3E,0x1E3F,0x004D},  {0x1E3E,0x1E3F,0x004D},
  {0x1E40,0x1E41,0x004D},  {0x1E40,0x1E41,0x004D},
  {0x1E42,0x1E43,0x004D},  {0x1E42,0x1E43,0x004D},
  {0x1E44,0x1E45,0x004E},  {0x1E44,0x1E45,0x004E},
  {0x1E46,0x1E47,0x004E},  {0x1E46,0x1E47,0x004E},
  {0x1E48,0x1E49,0x004E},  {0x1E48,0x1E49,0x004E},
  {0x1E4A,0x1E4B,0x004E},  {0x1E4A,0x1E4B,0x004E},
  {0x1E4C,0x1E4D,0x004F},  {0x1E4C,0x1E4D,0x004F},
  {0x1E4E,0x1E4F,0x004F},  {0x1E4E,0x1E4F,0x004F},
  {0x1E50,0x1E51,0x004F},  {0x1E50,0x1E51,0x004F},
  {0x1E52,0x1E53,0x004F},  {0x1E52,0x1E53,0x004F},
  {0x1E54,0x1E55,0x0050},  {0x1E54,0x1E55,0x0050},
  {0x1E56,0x1E57,0x0050},  {0x1E56,0x1E57,0x0050},
  {0x1E58,0x1E59,0x0052},  {0x1E58,0x1E59,0x0052},
  {0x1E5A,0x1E5B,0x0052},  {0x1E5A,0x1E5B,0x0052},
  {0x1E5C,0x1E5D,0x0052},  {0x1E5C,0x1E5D,0x0052},
  {0x1E5E,0x1E5F,0x0052},  {0x1E5E,0x1E5F,0x0052},
  {0x1E60,0x1E61,0x0053},  {0x1E60,0x1E61,0x0053},
  {0x1E62,0x1E63,0x0053},  {0x1E62,0x1E63,0x0053},
  {0x1E64,0x1E65,0x0053},  {0x1E64,0x1E65,0x0053},
  {0x1E66,0x1E67,0x0053},  {0x1E66,0x1E67,0x0053},
  {0x1E68,0x1E69,0x0053},  {0x1E68,0x1E69,0x0053},
  {0x1E6A,0x1E6B,0x0054},  {0x1E6A,0x1E6B,0x0054},
  {0x1E6C,0x1E6D,0x0054},  {0x1E6C,0x1E6D,0x0054},
  {0x1E6E,0x1E6F,0x0054},  {0x1E6E,0x1E6F,0x0054},
  {0x1E70,0x1E71,0x0054},  {0x1E70,0x1E71,0x0054},
  {0x1E72,0x1E73,0x0055},  {0x1E72,0x1E73,0x0055},
  {0x1E74,0x1E75,0x0055},  {0x1E74,0x1E75,0x0055},
  {0x1E76,0x1E77,0x0055},  {0x1E76,0x1E77,0x0055},
  {0x1E78,0x1E79,0x0055},  {0x1E78,0x1E79,0x0055},
  {0x1E7A,0x1E7B,0x0055},  {0x1E7A,0x1E7B,0x0055},
  {0x1E7C,0x1E7D,0x0056},  {0x1E7C,0x1E7D,0x0056},
  {0x1E7E,0x1E7F,0x0056},  {0x1E7E,0x1E7F,0x0056},
  {0x1E80,0x1E81,0x0057},  {0x1E80,0x1E81,0x0057},
  {0x1E82,0x1E83,0x0057},  {0x1E82,0x1E83,0x0057},
  {0x1E84,0x1E85,0x0057},  {0x1E84,0x1E85,0x0057},
  {0x1E86,0x1E87,0x0057},  {0x1E86,0x1E87,0x0057},
  {0x1E88,0x1E89,0x0057},  {0x1E88,0x1E89,0x0057},
  {0x1E8A,0x1E8B,0x0058},  {0x1E8A,0x1E8B,0x0058},
  {0x1E8C,0x1E8D,0x0058},  {0x1E8C,0x1E8D,0x0058},
  {0x1E8E,0x1E8F,0x0059},  {0x1E8E,0x1E8F,0x0059},
  {0x1E90,0x1E91,0x005A},  {0x1E90,0x1E91,0x005A},
  {0x1E92,0x1E93,0x005A},  {0x1E92,0x1E93,0x005A},
  {0x1E94,0x1E95,0x005A},  {0x1E94,0x1E95,0x005A},
  {0x1E96,0x1E96,0x0048},  {0x1E97,0x1E97,0x0054},
  {0x1E98,0x1E98,0x0057},  {0x1E99,0x1E99,0x0059},
  {0x1E9A,0x1E9A,0x1E9A},  {0x1E60,0x1E9B,0x0053},
  {0x1E9C,0x1E9C,0x1E9C},  {0x1E9D,0x1E9D,0x1E9D},
  {0x1E9E,0x1E9E,0x1E9E},  {0x1E9F,0x1E9F,0x1E9F},
  {0x1EA0,0x1EA1,0x0041},  {0x1EA0,0x1EA1,0x0041},
  {0x1EA2,0x1EA3,0x0041},  {0x1EA2,0x1EA3,0x0041},
  {0x1EA4,0x1EA5,0x0041},  {0x1EA4,0x1EA5,0x0041},
  {0x1EA6,0x1EA7,0x0041},  {0x1EA6,0x1EA7,0x0041},
  {0x1EA8,0x1EA9,0x0041},  {0x1EA8,0x1EA9,0x0041},
  {0x1EAA,0x1EAB,0x0041},  {0x1EAA,0x1EAB,0x0041},
  {0x1EAC,0x1EAD,0x0041},  {0x1EAC,0x1EAD,0x0041},
  {0x1EAE,0x1EAF,0x0041},  {0x1EAE,0x1EAF,0x0041},
  {0x1EB0,0x1EB1,0x0041},  {0x1EB0,0x1EB1,0x0041},
  {0x1EB2,0x1EB3,0x0041},  {0x1EB2,0x1EB3,0x0041},
  {0x1EB4,0x1EB5,0x0041},  {0x1EB4,0x1EB5,0x0041},
  {0x1EB6,0x1EB7,0x0041},  {0x1EB6,0x1EB7,0x0041},
  {0x1EB8,0x1EB9,0x0045},  {0x1EB8,0x1EB9,0x0045},
  {0x1EBA,0x1EBB,0x0045},  {0x1EBA,0x1EBB,0x0045},
  {0x1EBC,0x1EBD,0x0045},  {0x1EBC,0x1EBD,0x0045},
  {0x1EBE,0x1EBF,0x0045},  {0x1EBE,0x1EBF,0x0045},
  {0x1EC0,0x1EC1,0x0045},  {0x1EC0,0x1EC1,0x0045},
  {0x1EC2,0x1EC3,0x0045},  {0x1EC2,0x1EC3,0x0045},
  {0x1EC4,0x1EC5,0x0045},  {0x1EC4,0x1EC5,0x0045},
  {0x1EC6,0x1EC7,0x0045},  {0x1EC6,0x1EC7,0x0045},
  {0x1EC8,0x1EC9,0x0049},  {0x1EC8,0x1EC9,0x0049},
  {0x1ECA,0x1ECB,0x0049},  {0x1ECA,0x1ECB,0x0049},
  {0x1ECC,0x1ECD,0x004F},  {0x1ECC,0x1ECD,0x004F},
  {0x1ECE,0x1ECF,0x004F},  {0x1ECE,0x1ECF,0x004F},
  {0x1ED0,0x1ED1,0x004F},  {0x1ED0,0x1ED1,0x004F},
  {0x1ED2,0x1ED3,0x004F},  {0x1ED2,0x1ED3,0x004F},
  {0x1ED4,0x1ED5,0x004F},  {0x1ED4,0x1ED5,0x004F},
  {0x1ED6,0x1ED7,0x004F},  {0x1ED6,0x1ED7,0x004F},
  {0x1ED8,0x1ED9,0x004F},  {0x1ED8,0x1ED9,0x004F},
  {0x1EDA,0x1EDB,0x004F},  {0x1EDA,0x1EDB,0x004F},
  {0x1EDC,0x1EDD,0x004F},  {0x1EDC,0x1EDD,0x004F},
  {0x1EDE,0x1EDF,0x004F},  {0x1EDE,0x1EDF,0x004F},
  {0x1EE0,0x1EE1,0x004F},  {0x1EE0,0x1EE1,0x004F},
  {0x1EE2,0x1EE3,0x004F},  {0x1EE2,0x1EE3,0x004F},
  {0x1EE4,0x1EE5,0x0055},  {0x1EE4,0x1EE5,0x0055},
  {0x1EE6,0x1EE7,0x0055},  {0x1EE6,0x1EE7,0x0055},
  {0x1EE8,0x1EE9,0x0055},  {0x1EE8,0x1EE9,0x0055},
  {0x1EEA,0x1EEB,0x0055},  {0x1EEA,0x1EEB,0x0055},
  {0x1EEC,0x1EED,0x0055},  {0x1EEC,0x1EED,0x0055},
  {0x1EEE,0x1EEF,0x0055},  {0x1EEE,0x1EEF,0x0055},
  {0x1EF0,0x1EF1,0x0055},  {0x1EF0,0x1EF1,0x0055},
  {0x1EF2,0x1EF3,0x0059},  {0x1EF2,0x1EF3,0x0059},
  {0x1EF4,0x1EF5,0x0059},  {0x1EF4,0x1EF5,0x0059},
  {0x1EF6,0x1EF7,0x0059},  {0x1EF6,0x1EF7,0x0059},
  {0x1EF8,0x1EF9,0x0059},  {0x1EF8,0x1EF9,0x0059},
  {0x1EFA,0x1EFA,0x1EFA},  {0x1EFB,0x1EFB,0x1EFB},
  {0x1EFC,0x1EFC,0x1EFC},  {0x1EFD,0x1EFD,0x1EFD},
  {0x1EFE,0x1EFE,0x1EFE},  {0x1EFF,0x1EFF,0x1EFF}
};

static MY_UNICASE_CHARACTER plane1F[]={
  {0x1F08,0x1F00,0x0391},  {0x1F09,0x1F01,0x0391},
  {0x1F0A,0x1F02,0x0391},  {0x1F0B,0x1F03,0x0391},
  {0x1F0C,0x1F04,0x0391},  {0x1F0D,0x1F05,0x0391},
  {0x1F0E,0x1F06,0x0391},  {0x1F0F,0x1F07,0x0391},
  {0x1F08,0x1F00,0x0391},  {0x1F09,0x1F01,0x0391},
  {0x1F0A,0x1F02,0x0391},  {0x1F0B,0x1F03,0x0391},
  {0x1F0C,0x1F04,0x0391},  {0x1F0D,0x1F05,0x0391},
  {0x1F0E,0x1F06,0x0391},  {0x1F0F,0x1F07,0x0391},
  {0x1F18,0x1F10,0x0395},  {0x1F19,0x1F11,0x0395},
  {0x1F1A,0x1F12,0x0395},  {0x1F1B,0x1F13,0x0395},
  {0x1F1C,0x1F14,0x0395},  {0x1F1D,0x1F15,0x0395},
  {0x1F16,0x1F16,0x1F16},  {0x1F17,0x1F17,0x1F17},
  {0x1F18,0x1F10,0x0395},  {0x1F19,0x1F11,0x0395},
  {0x1F1A,0x1F12,0x0395},  {0x1F1B,0x1F13,0x0395},
  {0x1F1C,0x1F14,0x0395},  {0x1F1D,0x1F15,0x0395},
  {0x1F1E,0x1F1E,0x1F1E},  {0x1F1F,0x1F1F,0x1F1F},
  {0x1F28,0x1F20,0x0397},  {0x1F29,0x1F21,0x0397},
  {0x1F2A,0x1F22,0x0397},  {0x1F2B,0x1F23,0x0397},
  {0x1F2C,0x1F24,0x0397},  {0x1F2D,0x1F25,0x0397},
  {0x1F2E,0x1F26,0x0397},  {0x1F2F,0x1F27,0x0397},
  {0x1F28,0x1F20,0x0397},  {0x1F29,0x1F21,0x0397},
  {0x1F2A,0x1F22,0x0397},  {0x1F2B,0x1F23,0x0397},
  {0x1F2C,0x1F24,0x0397},  {0x1F2D,0x1F25,0x0397},
  {0x1F2E,0x1F26,0x0397},  {0x1F2F,0x1F27,0x0397},
  {0x1F38,0x1F30,0x0399},  {0x1F39,0x1F31,0x0399},
  {0x1F3A,0x1F32,0x0399},  {0x1F3B,0x1F33,0x0399},
  {0x1F3C,0x1F34,0x0399},  {0x1F3D,0x1F35,0x0399},
  {0x1F3E,0x1F36,0x0399},  {0x1F3F,0x1F37,0x0399},
  {0x1F38,0x1F30,0x0399},  {0x1F39,0x1F31,0x0399},
  {0x1F3A,0x1F32,0x0399},  {0x1F3B,0x1F33,0x0399},
  {0x1F3C,0x1F34,0x0399},  {0x1F3D,0x1F35,0x0399},
  {0x1F3E,0x1F36,0x0399},  {0x1F3F,0x1F37,0x0399},
  {0x1F48,0x1F40,0x039F},  {0x1F49,0x1F41,0x039F},
  {0x1F4A,0x1F42,0x039F},  {0x1F4B,0x1F43,0x039F},
  {0x1F4C,0x1F44,0x039F},  {0x1F4D,0x1F45,0x039F},
  {0x1F46,0x1F46,0x1F46},  {0x1F47,0x1F47,0x1F47},
  {0x1F48,0x1F40,0x039F},  {0x1F49,0x1F41,0x039F},
  {0x1F4A,0x1F42,0x039F},  {0x1F4B,0x1F43,0x039F},
  {0x1F4C,0x1F44,0x039F},  {0x1F4D,0x1F45,0x039F},
  {0x1F4E,0x1F4E,0x1F4E},  {0x1F4F,0x1F4F,0x1F4F},
  {0x1F50,0x1F50,0x03A5},  {0x1F59,0x1F51,0x03A5},
  {0x1F52,0x1F52,0x03A5},  {0x1F5B,0x1F53,0x03A5},
  {0x1F54,0x1F54,0x03A5},  {0x1F5D,0x1F55,0x03A5},
  {0x1F56,0x1F56,0x03A5},  {0x1F5F,0x1F57,0x03A5},
  {0x1F58,0x1F58,0x1F58},  {0x1F59,0x1F51,0x03A5},
  {0x1F5A,0x1F5A,0x1F5A},  {0x1F5B,0x1F53,0x03A5},
  {0x1F5C,0x1F5C,0x1F5C},  {0x1F5D,0x1F55,0x03A5},
  {0x1F5E,0x1F5E,0x1F5E},  {0x1F5F,0x1F57,0x03A5},
  {0x1F68,0x1F60,0x03A9},  {0x1F69,0x1F61,0x03A9},
  {0x1F6A,0x1F62,0x03A9},  {0x1F6B,0x1F63,0x03A9},
  {0x1F6C,0x1F64,0x03A9},  {0x1F6D,0x1F65,0x03A9},
  {0x1F6E,0x1F66,0x03A9},  {0x1F6F,0x1F67,0x03A9},
  {0x1F68,0x1F60,0x03A9},  {0x1F69,0x1F61,0x03A9},
  {0x1F6A,0x1F62,0x03A9},  {0x1F6B,0x1F63,0x03A9},
  {0x1F6C,0x1F64,0x03A9},  {0x1F6D,0x1F65,0x03A9},
  {0x1F6E,0x1F66,0x03A9},  {0x1F6F,0x1F67,0x03A9},
  {0x1FBA,0x1F70,0x0391},  {0x1FBB,0x1F71,0x1FBB},
  {0x1FC8,0x1F72,0x0395},  {0x1FC9,0x1F73,0x1FC9},
  {0x1FCA,0x1F74,0x0397},  {0x1FCB,0x1F75,0x1FCB},
  {0x1FDA,0x1F76,0x0399},  {0x1FDB,0x1F77,0x1FDB},
  {0x1FF8,0x1F78,0x039F},  {0x1FF9,0x1F79,0x1FF9},
  {0x1FEA,0x1F7A,0x03A5},  {0x1FEB,0x1F7B,0x1FEB},
  {0x1FFA,0x1F7C,0x03A9},  {0x1FFB,0x1F7D,0x1FFB},
  {0x1F7E,0x1F7E,0x1F7E},  {0x1F7F,0x1F7F,0x1F7F},
  {0x1F88,0x1F80,0x0391},  {0x1F89,0x1F81,0x0391},
  {0x1F8A,0x1F82,0x0391},  {0x1F8B,0x1F83,0x0391},
  {0x1F8C,0x1F84,0x0391},  {0x1F8D,0x1F85,0x0391},
  {0x1F8E,0x1F86,0x0391},  {0x1F8F,0x1F87,0x0391},
  {0x1F88,0x1F80,0x0391},  {0x1F89,0x1F81,0x0391},
  {0x1F8A,0x1F82,0x0391},  {0x1F8B,0x1F83,0x0391},
  {0x1F8C,0x1F84,0x0391},  {0x1F8D,0x1F85,0x0391},
  {0x1F8E,0x1F86,0x0391},  {0x1F8F,0x1F87,0x0391},
  {0x1F98,0x1F90,0x0397},  {0x1F99,0x1F91,0x0397},
  {0x1F9A,0x1F92,0x0397},  {0x1F9B,0x1F93,0x0397},
  {0x1F9C,0x1F94,0x0397},  {0x1F9D,0x1F95,0x0397},
  {0x1F9E,0x1F96,0x0397},  {0x1F9F,0x1F97,0x0397},
  {0x1F98,0x1F90,0x0397},  {0x1F99,0x1F91,0x0397},
  {0x1F9A,0x1F92,0x0397},  {0x1F9B,0x1F93,0x0397},
  {0x1F9C,0x1F94,0x0397},  {0x1F9D,0x1F95,0x0397},
  {0x1F9E,0x1F96,0x0397},  {0x1F9F,0x1F97,0x0397},
  {0x1FA8,0x1FA0,0x03A9},  {0x1FA9,0x1FA1,0x03A9},
  {0x1FAA,0x1FA2,0x03A9},  {0x1FAB,0x1FA3,0x03A9},
  {0x1FAC,0x1FA4,0x03A9},  {0x1FAD,0x1FA5,0x03A9},
  {0x1FAE,0x1FA6,0x03A9},  {0x1FAF,0x1FA7,0x03A9},
  {0x1FA8,0x1FA0,0x03A9},  {0x1FA9,0x1FA1,0x03A9},
  {0x1FAA,0x1FA2,0x03A9},  {0x1FAB,0x1FA3,0x03A9},
  {0x1FAC,0x1FA4,0x03A9},  {0x1FAD,0x1FA5,0x03A9},
  {0x1FAE,0x1FA6,0x03A9},  {0x1FAF,0x1FA7,0x03A9},
  {0x1FB8,0x1FB0,0x0391},  {0x1FB9,0x1FB1,0x0391},
  {0x1FB2,0x1FB2,0x0391},  {0x1FBC,0x1FB3,0x0391},
  {0x1FB4,0x1FB4,0x0391},  {0x1FB5,0x1FB5,0x1FB5},
  {0x1FB6,0x1FB6,0x0391},  {0x1FB7,0x1FB7,0x0391},
  {0x1FB8,0x1FB0,0x0391},  {0x1FB9,0x1FB1,0x0391},
  {0x1FBA,0x1F70,0x0391},  {0x1FBB,0x1F71,0x1FBB},
  {0x1FBC,0x1FB3,0x0391},  {0x1FBD,0x1FBD,0x1FBD},
  {0x0399,0x1FBE,0x0399},  {0x1FBF,0x1FBF,0x1FBF},
  {0x1FC0,0x1FC0,0x1FC0},  {0x1FC1,0x1FC1,0x1FC1},
  {0x1FC2,0x1FC2,0x0397},  {0x1FCC,0x1FC3,0x0397},
  {0x1FC4,0x1FC4,0x0397},  {0x1FC5,0x1FC5,0x1FC5},
  {0x1FC6,0x1FC6,0x0397},  {0x1FC7,0x1FC7,0x0397},
  {0x1FC8,0x1F72,0x0395},  {0x1FC9,0x1F73,0x1FC9},
  {0x1FCA,0x1F74,0x0397},  {0x1FCB,0x1F75,0x1FCB},
  {0x1FCC,0x1FC3,0x0397},  {0x1FCD,0x1FCD,0x1FCD},
  {0x1FCE,0x1FCE,0x1FCE},  {0x1FCF,0x1FCF,0x1FCF},
  {0x1FD8,0x1FD0,0x0399},  {0x1FD9,0x1FD1,0x0399},
  {0x1FD2,0x1FD2,0x0399},  {0x1FD3,0x1FD3,0x1FD3},
  {0x1FD4,0x1FD4,0x1FD4},  {0x1FD5,0x1FD5,0x1FD5},
  {0x1FD6,0x1FD6,0x0399},  {0x1FD7,0x1FD7,0x0399},
  {0x1FD8,0x1FD0,0x0399},  {0x1FD9,0x1FD1,0x0399},
  {0x1FDA,0x1F76,0x0399},  {0x1FDB,0x1F77,0x1FDB},
  {0x1FDC,0x1FDC,0x1FDC},  {0x1FDD,0x1FDD,0x1FDD},
  {0x1FDE,0x1FDE,0x1FDE},  {0x1FDF,0x1FDF,0x1FDF},
  {0x1FE8,0x1FE0,0x03A5},  {0x1FE9,0x1FE1,0x03A5},
  {0x1FE2,0x1FE2,0x03A5},  {0x1FE3,0x1FE3,0x1FE3},
  {0x1FE4,0x1FE4,0x03A1},  {0x1FEC,0x1FE5,0x03A1},
  {0x1FE6,0x1FE6,0x03A5},  {0x1FE7,0x1FE7,0x03A5},
  {0x1FE8,0x1FE0,0x03A5},  {0x1FE9,0x1FE1,0x03A5},
  {0x1FEA,0x1F7A,0x03A5},  {0x1FEB,0x1F7B,0x1FEB},
  {0x1FEC,0x1FE5,0x03A1},  {0x1FED,0x1FED,0x1FED},
  {0x1FEE,0x1FEE,0x1FEE},  {0x1FEF,0x1FEF,0x1FEF},
  {0x1FF0,0x1FF0,0x1FF0},  {0x1FF1,0x1FF1,0x1FF1},
  {0x1FF2,0x1FF2,0x03A9},  {0x1FFC,0x1FF3,0x03A9},
  {0x1FF4,0x1FF4,0x03A9},  {0x1FF5,0x1FF5,0x1FF5},
  {0x1FF6,0x1FF6,0x03A9},  {0x1FF7,0x1FF7,0x03A9},
  {0x1FF8,0x1F78,0x039F},  {0x1FF9,0x1F79,0x1FF9},
  {0x1FFA,0x1F7C,0x03A9},  {0x1FFB,0x1F7D,0x1FFB},
  {0x1FFC,0x1FF3,0x03A9},  {0x1FFD,0x1FFD,0x1FFD},
  {0x1FFE,0x1FFE,0x1FFE},  {0x1FFF,0x1FFF,0x1FFF}
};

static MY_UNICASE_CHARACTER plane21[]={
  {0x2100,0x2100,0x2100},  {0x2101,0x2101,0x2101},
  {0x2102,0x2102,0x2102},  {0x2103,0x2103,0x2103},
  {0x2104,0x2104,0x2104},  {0x2105,0x2105,0x2105},
  {0x2106,0x2106,0x2106},  {0x2107,0x2107,0x2107},
  {0x2108,0x2108,0x2108},  {0x2109,0x2109,0x2109},
  {0x210A,0x210A,0x210A},  {0x210B,0x210B,0x210B},
  {0x210C,0x210C,0x210C},  {0x210D,0x210D,0x210D},
  {0x210E,0x210E,0x210E},  {0x210F,0x210F,0x210F},
  {0x2110,0x2110,0x2110},  {0x2111,0x2111,0x2111},
  {0x2112,0x2112,0x2112},  {0x2113,0x2113,0x2113},
  {0x2114,0x2114,0x2114},  {0x2115,0x2115,0x2115},
  {0x2116,0x2116,0x2116},  {0x2117,0x2117,0x2117},
  {0x2118,0x2118,0x2118},  {0x2119,0x2119,0x2119},
  {0x211A,0x211A,0x211A},  {0x211B,0x211B,0x211B},
  {0x211C,0x211C,0x211C},  {0x211D,0x211D,0x211D},
  {0x211E,0x211E,0x211E},  {0x211F,0x211F,0x211F},
  {0x2120,0x2120,0x2120},  {0x2121,0x2121,0x2121},
  {0x2122,0x2122,0x2122},  {0x2123,0x2123,0x2123},
  {0x2124,0x2124,0x2124},  {0x2125,0x2125,0x2125},
  {0x2126,0x03C9,0x2126},  {0x2127,0x2127,0x2127},
  {0x2128,0x2128,0x2128},  {0x2129,0x2129,0x2129},
  {0x212A,0x006B,0x212A},  {0x212B,0x00E5,0x212B},
  {0x212C,0x212C,0x212C},  {0x212D,0x212D,0x212D},
  {0x212E,0x212E,0x212E},  {0x212F,0x212F,0x212F},
  {0x2130,0x2130,0x2130},  {0x2131,0x2131,0x2131},
  {0x2132,0x2132,0x2132},  {0x2133,0x2133,0x2133},
  {0x2134,0x2134,0x2134},  {0x2135,0x2135,0x2135},
  {0x2136,0x2136,0x2136},  {0x2137,0x2137,0x2137},
  {0x2138,0x2138,0x2138},  {0x2139,0x2139,0x2139},
  {0x213A,0x213A,0x213A},  {0x213B,0x213B,0x213B},
  {0x213C,0x213C,0x213C},  {0x213D,0x213D,0x213D},
  {0x213E,0x213E,0x213E},  {0x213F,0x213F,0x213F},
  {0x2140,0x2140,0x2140},  {0x2141,0x2141,0x2141},
  {0x2142,0x2142,0x2142},  {0x2143,0x2143,0x2143},
  {0x2144,0x2144,0x2144},  {0x2145,0x2145,0x2145},
  {0x2146,0x2146,0x2146},  {0x2147,0x2147,0x2147},
  {0x2148,0x2148,0x2148},  {0x2149,0x2149,0x2149},
  {0x214A,0x214A,0x214A},  {0x214B,0x214B,0x214B},
  {0x214C,0x214C,0x214C},  {0x214D,0x214D,0x214D},
  {0x214E,0x214E,0x214E},  {0x214F,0x214F,0x214F},
  {0x2150,0x2150,0x2150},  {0x2151,0x2151,0x2151},
  {0x2152,0x2152,0x2152},  {0x2153,0x2153,0x2153},
  {0x2154,0x2154,0x2154},  {0x2155,0x2155,0x2155},
  {0x2156,0x2156,0x2156},  {0x2157,0x2157,0x2157},
  {0x2158,0x2158,0x2158},  {0x2159,0x2159,0x2159},
  {0x215A,0x215A,0x215A},  {0x215B,0x215B,0x215B},
  {0x215C,0x215C,0x215C},  {0x215D,0x215D,0x215D},
  {0x215E,0x215E,0x215E},  {0x215F,0x215F,0x215F},
  {0x2160,0x2170,0x2160},  {0x2161,0x2171,0x2161},
  {0x2162,0x2172,0x2162},  {0x2163,0x2173,0x2163},
  {0x2164,0x2174,0x2164},  {0x2165,0x2175,0x2165},
  {0x2166,0x2176,0x2166},  {0x2167,0x2177,0x2167},
  {0x2168,0x2178,0x2168},  {0x2169,0x2179,0x2169},
  {0x216A,0x217A,0x216A},  {0x216B,0x217B,0x216B},
  {0x216C,0x217C,0x216C},  {0x216D,0x217D,0x216D},
  {0x216E,0x217E,0x216E},  {0x216F,0x217F,0x216F},
  {0x2160,0x2170,0x2160},  {0x2161,0x2171,0x2161},
  {0x2162,0x2172,0x2162},  {0x2163,0x2173,0x2163},
  {0x2164,0x2174,0x2164},  {0x2165,0x2175,0x2165},
  {0x2166,0x2176,0x2166},  {0x2167,0x2177,0x2167},
  {0x2168,0x2178,0x2168},  {0x2169,0x2179,0x2169},
  {0x216A,0x217A,0x216A},  {0x216B,0x217B,0x216B},
  {0x216C,0x217C,0x216C},  {0x216D,0x217D,0x216D},
  {0x216E,0x217E,0x216E},  {0x216F,0x217F,0x216F},
  {0x2180,0x2180,0x2180},  {0x2181,0x2181,0x2181},
  {0x2182,0x2182,0x2182},  {0x2183,0x2183,0x2183},
  {0x2184,0x2184,0x2184},  {0x2185,0x2185,0x2185},
  {0x2186,0x2186,0x2186},  {0x2187,0x2187,0x2187},
  {0x2188,0x2188,0x2188},  {0x2189,0x2189,0x2189},
  {0x218A,0x218A,0x218A},  {0x218B,0x218B,0x218B},
  {0x218C,0x218C,0x218C},  {0x218D,0x218D,0x218D},
  {0x218E,0x218E,0x218E},  {0x218F,0x218F,0x218F},
  {0x2190,0x2190,0x2190},  {0x2191,0x2191,0x2191},
  {0x2192,0x2192,0x2192},  {0x2193,0x2193,0x2193},
  {0x2194,0x2194,0x2194},  {0x2195,0x2195,0x2195},
  {0x2196,0x2196,0x2196},  {0x2197,0x2197,0x2197},
  {0x2198,0x2198,0x2198},  {0x2199,0x2199,0x2199},
  {0x219A,0x219A,0x219A},  {0x219B,0x219B,0x219B},
  {0x219C,0x219C,0x219C},  {0x219D,0x219D,0x219D},
  {0x219E,0x219E,0x219E},  {0x219F,0x219F,0x219F},
  {0x21A0,0x21A0,0x21A0},  {0x21A1,0x21A1,0x21A1},
  {0x21A2,0x21A2,0x21A2},  {0x21A3,0x21A3,0x21A3},
  {0x21A4,0x21A4,0x21A4},  {0x21A5,0x21A5,0x21A5},
  {0x21A6,0x21A6,0x21A6},  {0x21A7,0x21A7,0x21A7},
  {0x21A8,0x21A8,0x21A8},  {0x21A9,0x21A9,0x21A9},
  {0x21AA,0x21AA,0x21AA},  {0x21AB,0x21AB,0x21AB},
  {0x21AC,0x21AC,0x21AC},  {0x21AD,0x21AD,0x21AD},
  {0x21AE,0x21AE,0x21AE},  {0x21AF,0x21AF,0x21AF},
  {0x21B0,0x21B0,0x21B0},  {0x21B1,0x21B1,0x21B1},
  {0x21B2,0x21B2,0x21B2},  {0x21B3,0x21B3,0x21B3},
  {0x21B4,0x21B4,0x21B4},  {0x21B5,0x21B5,0x21B5},
  {0x21B6,0x21B6,0x21B6},  {0x21B7,0x21B7,0x21B7},
  {0x21B8,0x21B8,0x21B8},  {0x21B9,0x21B9,0x21B9},
  {0x21BA,0x21BA,0x21BA},  {0x21BB,0x21BB,0x21BB},
  {0x21BC,0x21BC,0x21BC},  {0x21BD,0x21BD,0x21BD},
  {0x21BE,0x21BE,0x21BE},  {0x21BF,0x21BF,0x21BF},
  {0x21C0,0x21C0,0x21C0},  {0x21C1,0x21C1,0x21C1},
  {0x21C2,0x21C2,0x21C2},  {0x21C3,0x21C3,0x21C3},
  {0x21C4,0x21C4,0x21C4},  {0x21C5,0x21C5,0x21C5},
  {0x21C6,0x21C6,0x21C6},  {0x21C7,0x21C7,0x21C7},
  {0x21C8,0x21C8,0x21C8},  {0x21C9,0x21C9,0x21C9},
  {0x21CA,0x21CA,0x21CA},  {0x21CB,0x21CB,0x21CB},
  {0x21CC,0x21CC,0x21CC},  {0x21CD,0x21CD,0x21CD},
  {0x21CE,0x21CE,0x21CE},  {0x21CF,0x21CF,0x21CF},
  {0x21D0,0x21D0,0x21D0},  {0x21D1,0x21D1,0x21D1},
  {0x21D2,0x21D2,0x21D2},  {0x21D3,0x21D3,0x21D3},
  {0x21D4,0x21D4,0x21D4},  {0x21D5,0x21D5,0x21D5},
  {0x21D6,0x21D6,0x21D6},  {0x21D7,0x21D7,0x21D7},
  {0x21D8,0x21D8,0x21D8},  {0x21D9,0x21D9,0x21D9},
  {0x21DA,0x21DA,0x21DA},  {0x21DB,0x21DB,0x21DB},
  {0x21DC,0x21DC,0x21DC},  {0x21DD,0x21DD,0x21DD},
  {0x21DE,0x21DE,0x21DE},  {0x21DF,0x21DF,0x21DF},
  {0x21E0,0x21E0,0x21E0},  {0x21E1,0x21E1,0x21E1},
  {0x21E2,0x21E2,0x21E2},  {0x21E3,0x21E3,0x21E3},
  {0x21E4,0x21E4,0x21E4},  {0x21E5,0x21E5,0x21E5},
  {0x21E6,0x21E6,0x21E6},  {0x21E7,0x21E7,0x21E7},
  {0x21E8,0x21E8,0x21E8},  {0x21E9,0x21E9,0x21E9},
  {0x21EA,0x21EA,0x21EA},  {0x21EB,0x21EB,0x21EB},
  {0x21EC,0x21EC,0x21EC},  {0x21ED,0x21ED,0x21ED},
  {0x21EE,0x21EE,0x21EE},  {0x21EF,0x21EF,0x21EF},
  {0x21F0,0x21F0,0x21F0},  {0x21F1,0x21F1,0x21F1},
  {0x21F2,0x21F2,0x21F2},  {0x21F3,0x21F3,0x21F3},
  {0x21F4,0x21F4,0x21F4},  {0x21F5,0x21F5,0x21F5},
  {0x21F6,0x21F6,0x21F6},  {0x21F7,0x21F7,0x21F7},
  {0x21F8,0x21F8,0x21F8},  {0x21F9,0x21F9,0x21F9},
  {0x21FA,0x21FA,0x21FA},  {0x21FB,0x21FB,0x21FB},
  {0x21FC,0x21FC,0x21FC},  {0x21FD,0x21FD,0x21FD},
  {0x21FE,0x21FE,0x21FE},  {0x21FF,0x21FF,0x21FF}
};

static MY_UNICASE_CHARACTER plane24[]={
  {0x2400,0x2400,0x2400},  {0x2401,0x2401,0x2401},
  {0x2402,0x2402,0x2402},  {0x2403,0x2403,0x2403},
  {0x2404,0x2404,0x2404},  {0x2405,0x2405,0x2405},
  {0x2406,0x2406,0x2406},  {0x2407,0x2407,0x2407},
  {0x2408,0x2408,0x2408},  {0x2409,0x2409,0x2409},
  {0x240A,0x240A,0x240A},  {0x240B,0x240B,0x240B},
  {0x240C,0x240C,0x240C},  {0x240D,0x240D,0x240D},
  {0x240E,0x240E,0x240E},  {0x240F,0x240F,0x240F},
  {0x2410,0x2410,0x2410},  {0x2411,0x2411,0x2411},
  {0x2412,0x2412,0x2412},  {0x2413,0x2413,0x2413},
  {0x2414,0x2414,0x2414},  {0x2415,0x2415,0x2415},
  {0x2416,0x2416,0x2416},  {0x2417,0x2417,0x2417},
  {0x2418,0x2418,0x2418},  {0x2419,0x2419,0x2419},
  {0x241A,0x241A,0x241A},  {0x241B,0x241B,0x241B},
  {0x241C,0x241C,0x241C},  {0x241D,0x241D,0x241D},
  {0x241E,0x241E,0x241E},  {0x241F,0x241F,0x241F},
  {0x2420,0x2420,0x2420},  {0x2421,0x2421,0x2421},
  {0x2422,0x2422,0x2422},  {0x2423,0x2423,0x2423},
  {0x2424,0x2424,0x2424},  {0x2425,0x2425,0x2425},
  {0x2426,0x2426,0x2426},  {0x2427,0x2427,0x2427},
  {0x2428,0x2428,0x2428},  {0x2429,0x2429,0x2429},
  {0x242A,0x242A,0x242A},  {0x242B,0x242B,0x242B},
  {0x242C,0x242C,0x242C},  {0x242D,0x242D,0x242D},
  {0x242E,0x242E,0x242E},  {0x242F,0x242F,0x242F},
  {0x2430,0x2430,0x2430},  {0x2431,0x2431,0x2431},
  {0x2432,0x2432,0x2432},  {0x2433,0x2433,0x2433},
  {0x2434,0x2434,0x2434},  {0x2435,0x2435,0x2435},
  {0x2436,0x2436,0x2436},  {0x2437,0x2437,0x2437},
  {0x2438,0x2438,0x2438},  {0x2439,0x2439,0x2439},
  {0x243A,0x243A,0x243A},  {0x243B,0x243B,0x243B},
  {0x243C,0x243C,0x243C},  {0x243D,0x243D,0x243D},
  {0x243E,0x243E,0x243E},  {0x243F,0x243F,0x243F},
  {0x2440,0x2440,0x2440},  {0x2441,0x2441,0x2441},
  {0x2442,0x2442,0x2442},  {0x2443,0x2443,0x2443},
  {0x2444,0x2444,0x2444},  {0x2445,0x2445,0x2445},
  {0x2446,0x2446,0x2446},  {0x2447,0x2447,0x2447},
  {0x2448,0x2448,0x2448},  {0x2449,0x2449,0x2449},
  {0x244A,0x244A,0x244A},  {0x244B,0x244B,0x244B},
  {0x244C,0x244C,0x244C},  {0x244D,0x244D,0x244D},
  {0x244E,0x244E,0x244E},  {0x244F,0x244F,0x244F},
  {0x2450,0x2450,0x2450},  {0x2451,0x2451,0x2451},
  {0x2452,0x2452,0x2452},  {0x2453,0x2453,0x2453},
  {0x2454,0x2454,0x2454},  {0x2455,0x2455,0x2455},
  {0x2456,0x2456,0x2456},  {0x2457,0x2457,0x2457},
  {0x2458,0x2458,0x2458},  {0x2459,0x2459,0x2459},
  {0x245A,0x245A,0x245A},  {0x245B,0x245B,0x245B},
  {0x245C,0x245C,0x245C},  {0x245D,0x245D,0x245D},
  {0x245E,0x245E,0x245E},  {0x245F,0x245F,0x245F},
  {0x2460,0x2460,0x2460},  {0x2461,0x2461,0x2461},
  {0x2462,0x2462,0x2462},  {0x2463,0x2463,0x2463},
  {0x2464,0x2464,0x2464},  {0x2465,0x2465,0x2465},
  {0x2466,0x2466,0x2466},  {0x2467,0x2467,0x2467},
  {0x2468,0x2468,0x2468},  {0x2469,0x2469,0x2469},
  {0x246A,0x246A,0x246A},  {0x246B,0x246B,0x246B},
  {0x246C,0x246C,0x246C},  {0x246D,0x246D,0x246D},
  {0x246E,0x246E,0x246E},  {0x246F,0x246F,0x246F},
  {0x2470,0x2470,0x2470},  {0x2471,0x2471,0x2471},
  {0x2472,0x2472,0x2472},  {0x2473,0x2473,0x2473},
  {0x2474,0x2474,0x2474},  {0x2475,0x2475,0x2475},
  {0x2476,0x2476,0x2476},  {0x2477,0x2477,0x2477},
  {0x2478,0x2478,0x2478},  {0x2479,0x2479,0x2479},
  {0x247A,0x247A,0x247A},  {0x247B,0x247B,0x247B},
  {0x247C,0x247C,0x247C},  {0x247D,0x247D,0x247D},
  {0x247E,0x247E,0x247E},  {0x247F,0x247F,0x247F},
  {0x2480,0x2480,0x2480},  {0x2481,0x2481,0x2481},
  {0x2482,0x2482,0x2482},  {0x2483,0x2483,0x2483},
  {0x2484,0x2484,0x2484},  {0x2485,0x2485,0x2485},
  {0x2486,0x2486,0x2486},  {0x2487,0x2487,0x2487},
  {0x2488,0x2488,0x2488},  {0x2489,0x2489,0x2489},
  {0x248A,0x248A,0x248A},  {0x248B,0x248B,0x248B},
  {0x248C,0x248C,0x248C},  {0x248D,0x248D,0x248D},
  {0x248E,0x248E,0x248E},  {0x248F,0x248F,0x248F},
  {0x2490,0x2490,0x2490},  {0x2491,0x2491,0x2491},
  {0x2492,0x2492,0x2492},  {0x2493,0x2493,0x2493},
  {0x2494,0x2494,0x2494},  {0x2495,0x2495,0x2495},
  {0x2496,0x2496,0x2496},  {0x2497,0x2497,0x2497},
  {0x2498,0x2498,0x2498},  {0x2499,0x2499,0x2499},
  {0x249A,0x249A,0x249A},  {0x249B,0x249B,0x249B},
  {0x249C,0x249C,0x249C},  {0x249D,0x249D,0x249D},
  {0x249E,0x249E,0x249E},  {0x249F,0x249F,0x249F},
  {0x24A0,0x24A0,0x24A0},  {0x24A1,0x24A1,0x24A1},
  {0x24A2,0x24A2,0x24A2},  {0x24A3,0x24A3,0x24A3},
  {0x24A4,0x24A4,0x24A4},  {0x24A5,0x24A5,0x24A5},
  {0x24A6,0x24A6,0x24A6},  {0x24A7,0x24A7,0x24A7},
  {0x24A8,0x24A8,0x24A8},  {0x24A9,0x24A9,0x24A9},
  {0x24AA,0x24AA,0x24AA},  {0x24AB,0x24AB,0x24AB},
  {0x24AC,0x24AC,0x24AC},  {0x24AD,0x24AD,0x24AD},
  {0x24AE,0x24AE,0x24AE},  {0x24AF,0x24AF,0x24AF},
  {0x24B0,0x24B0,0x24B0},  {0x24B1,0x24B1,0x24B1},
  {0x24B2,0x24B2,0x24B2},  {0x24B3,0x24B3,0x24B3},
  {0x24B4,0x24B4,0x24B4},  {0x24B5,0x24B5,0x24B5},
  {0x24B6,0x24D0,0x24B6},  {0x24B7,0x24D1,0x24B7},
  {0x24B8,0x24D2,0x24B8},  {0x24B9,0x24D3,0x24B9},
  {0x24BA,0x24D4,0x24BA},  {0x24BB,0x24D5,0x24BB},
  {0x24BC,0x24D6,0x24BC},  {0x24BD,0x24D7,0x24BD},
  {0x24BE,0x24D8,0x24BE},  {0x24BF,0x24D9,0x24BF},
  {0x24C0,0x24DA,0x24C0},  {0x24C1,0x24DB,0x24C1},
  {0x24C2,0x24DC,0x24C2},  {0x24C3,0x24DD,0x24C3},
  {0x24C4,0x24DE,0x24C4},  {0x24C5,0x24DF,0x24C5},
  {0x24C6,0x24E0,0x24C6},  {0x24C7,0x24E1,0x24C7},
  {0x24C8,0x24E2,0x24C8},  {0x24C9,0x24E3,0x24C9},
  {0x24CA,0x24E4,0x24CA},  {0x24CB,0x24E5,0x24CB},
  {0x24CC,0x24E6,0x24CC},  {0x24CD,0x24E7,0x24CD},
  {0x24CE,0x24E8,0x24CE},  {0x24CF,0x24E9,0x24CF},
  {0x24B6,0x24D0,0x24B6},  {0x24B7,0x24D1,0x24B7},
  {0x24B8,0x24D2,0x24B8},  {0x24B9,0x24D3,0x24B9},
  {0x24BA,0x24D4,0x24BA},  {0x24BB,0x24D5,0x24BB},
  {0x24BC,0x24D6,0x24BC},  {0x24BD,0x24D7,0x24BD},
  {0x24BE,0x24D8,0x24BE},  {0x24BF,0x24D9,0x24BF},
  {0x24C0,0x24DA,0x24C0},  {0x24C1,0x24DB,0x24C1},
  {0x24C2,0x24DC,0x24C2},  {0x24C3,0x24DD,0x24C3},
  {0x24C4,0x24DE,0x24C4},  {0x24C5,0x24DF,0x24C5},
  {0x24C6,0x24E0,0x24C6},  {0x24C7,0x24E1,0x24C7},
  {0x24C8,0x24E2,0x24C8},  {0x24C9,0x24E3,0x24C9},
  {0x24CA,0x24E4,0x24CA},  {0x24CB,0x24E5,0x24CB},
  {0x24CC,0x24E6,0x24CC},  {0x24CD,0x24E7,0x24CD},
  {0x24CE,0x24E8,0x24CE},  {0x24CF,0x24E9,0x24CF},
  {0x24EA,0x24EA,0x24EA},  {0x24EB,0x24EB,0x24EB},
  {0x24EC,0x24EC,0x24EC},  {0x24ED,0x24ED,0x24ED},
  {0x24EE,0x24EE,0x24EE},  {0x24EF,0x24EF,0x24EF},
  {0x24F0,0x24F0,0x24F0},  {0x24F1,0x24F1,0x24F1},
  {0x24F2,0x24F2,0x24F2},  {0x24F3,0x24F3,0x24F3},
  {0x24F4,0x24F4,0x24F4},  {0x24F5,0x24F5,0x24F5},
  {0x24F6,0x24F6,0x24F6},  {0x24F7,0x24F7,0x24F7},
  {0x24F8,0x24F8,0x24F8},  {0x24F9,0x24F9,0x24F9},
  {0x24FA,0x24FA,0x24FA},  {0x24FB,0x24FB,0x24FB},
  {0x24FC,0x24FC,0x24FC},  {0x24FD,0x24FD,0x24FD},
  {0x24FE,0x24FE,0x24FE},  {0x24FF,0x24FF,0x24FF}
};

static MY_UNICASE_CHARACTER planeFF[]={
  {0xFF00,0xFF00,0xFF00},  {0xFF01,0xFF01,0xFF01},
  {0xFF02,0xFF02,0xFF02},  {0xFF03,0xFF03,0xFF03},
  {0xFF04,0xFF04,0xFF04},  {0xFF05,0xFF05,0xFF05},
  {0xFF06,0xFF06,0xFF06},  {0xFF07,0xFF07,0xFF07},
  {0xFF08,0xFF08,0xFF08},  {0xFF09,0xFF09,0xFF09},
  {0xFF0A,0xFF0A,0xFF0A},  {0xFF0B,0xFF0B,0xFF0B},
  {0xFF0C,0xFF0C,0xFF0C},  {0xFF0D,0xFF0D,0xFF0D},
  {0xFF0E,0xFF0E,0xFF0E},  {0xFF0F,0xFF0F,0xFF0F},
  {0xFF10,0xFF10,0xFF10},  {0xFF11,0xFF11,0xFF11},
  {0xFF12,0xFF12,0xFF12},  {0xFF13,0xFF13,0xFF13},
  {0xFF14,0xFF14,0xFF14},  {0xFF15,0xFF15,0xFF15},
  {0xFF16,0xFF16,0xFF16},  {0xFF17,0xFF17,0xFF17},
  {0xFF18,0xFF18,0xFF18},  {0xFF19,0xFF19,0xFF19},
  {0xFF1A,0xFF1A,0xFF1A},  {0xFF1B,0xFF1B,0xFF1B},
  {0xFF1C,0xFF1C,0xFF1C},  {0xFF1D,0xFF1D,0xFF1D},
  {0xFF1E,0xFF1E,0xFF1E},  {0xFF1F,0xFF1F,0xFF1F},
  {0xFF20,0xFF20,0xFF20},  {0xFF21,0xFF41,0xFF21},
  {0xFF22,0xFF42,0xFF22},  {0xFF23,0xFF43,0xFF23},
  {0xFF24,0xFF44,0xFF24},  {0xFF25,0xFF45,0xFF25},
  {0xFF26,0xFF46,0xFF26},  {0xFF27,0xFF47,0xFF27},
  {0xFF28,0xFF48,0xFF28},  {0xFF29,0xFF49,0xFF29},
  {0xFF2A,0xFF4A,0xFF2A},  {0xFF2B,0xFF4B,0xFF2B},
  {0xFF2C,0xFF4C,0xFF2C},  {0xFF2D,0xFF4D,0xFF2D},
  {0xFF2E,0xFF4E,0xFF2E},  {0xFF2F,0xFF4F,0xFF2F},
  {0xFF30,0xFF50,0xFF30},  {0xFF31,0xFF51,0xFF31},
  {0xFF32,0xFF52,0xFF32},  {0xFF33,0xFF53,0xFF33},
  {0xFF34,0xFF54,0xFF34},  {0xFF35,0xFF55,0xFF35},
  {0xFF36,0xFF56,0xFF36},  {0xFF37,0xFF57,0xFF37},
  {0xFF38,0xFF58,0xFF38},  {0xFF39,0xFF59,0xFF39},
  {0xFF3A,0xFF5A,0xFF3A},  {0xFF3B,0xFF3B,0xFF3B},
  {0xFF3C,0xFF3C,0xFF3C},  {0xFF3D,0xFF3D,0xFF3D},
  {0xFF3E,0xFF3E,0xFF3E},  {0xFF3F,0xFF3F,0xFF3F},
  {0xFF40,0xFF40,0xFF40},  {0xFF21,0xFF41,0xFF21},
  {0xFF22,0xFF42,0xFF22},  {0xFF23,0xFF43,0xFF23},
  {0xFF24,0xFF44,0xFF24},  {0xFF25,0xFF45,0xFF25},
  {0xFF26,0xFF46,0xFF26},  {0xFF27,0xFF47,0xFF27},
  {0xFF28,0xFF48,0xFF28},  {0xFF29,0xFF49,0xFF29},
  {0xFF2A,0xFF4A,0xFF2A},  {0xFF2B,0xFF4B,0xFF2B},
  {0xFF2C,0xFF4C,0xFF2C},  {0xFF2D,0xFF4D,0xFF2D},
  {0xFF2E,0xFF4E,0xFF2E},  {0xFF2F,0xFF4F,0xFF2F},
  {0xFF30,0xFF50,0xFF30},  {0xFF31,0xFF51,0xFF31},
  {0xFF32,0xFF52,0xFF32},  {0xFF33,0xFF53,0xFF33},
  {0xFF34,0xFF54,0xFF34},  {0xFF35,0xFF55,0xFF35},
  {0xFF36,0xFF56,0xFF36},  {0xFF37,0xFF57,0xFF37},
  {0xFF38,0xFF58,0xFF38},  {0xFF39,0xFF59,0xFF39},
  {0xFF3A,0xFF5A,0xFF3A},  {0xFF5B,0xFF5B,0xFF5B},
  {0xFF5C,0xFF5C,0xFF5C},  {0xFF5D,0xFF5D,0xFF5D},
  {0xFF5E,0xFF5E,0xFF5E},  {0xFF5F,0xFF5F,0xFF5F},
  {0xFF60,0xFF60,0xFF60},  {0xFF61,0xFF61,0xFF61},
  {0xFF62,0xFF62,0xFF62},  {0xFF63,0xFF63,0xFF63},
  {0xFF64,0xFF64,0xFF64},  {0xFF65,0xFF65,0xFF65},
  {0xFF66,0xFF66,0xFF66},  {0xFF67,0xFF67,0xFF67},
  {0xFF68,0xFF68,0xFF68},  {0xFF69,0xFF69,0xFF69},
  {0xFF6A,0xFF6A,0xFF6A},  {0xFF6B,0xFF6B,0xFF6B},
  {0xFF6C,0xFF6C,0xFF6C},  {0xFF6D,0xFF6D,0xFF6D},
  {0xFF6E,0xFF6E,0xFF6E},  {0xFF6F,0xFF6F,0xFF6F},
  {0xFF70,0xFF70,0xFF70},  {0xFF71,0xFF71,0xFF71},
  {0xFF72,0xFF72,0xFF72},  {0xFF73,0xFF73,0xFF73},
  {0xFF74,0xFF74,0xFF74},  {0xFF75,0xFF75,0xFF75},
  {0xFF76,0xFF76,0xFF76},  {0xFF77,0xFF77,0xFF77},
  {0xFF78,0xFF78,0xFF78},  {0xFF79,0xFF79,0xFF79},
  {0xFF7A,0xFF7A,0xFF7A},  {0xFF7B,0xFF7B,0xFF7B},
  {0xFF7C,0xFF7C,0xFF7C},  {0xFF7D,0xFF7D,0xFF7D},
  {0xFF7E,0xFF7E,0xFF7E},  {0xFF7F,0xFF7F,0xFF7F},
  {0xFF80,0xFF80,0xFF80},  {0xFF81,0xFF81,0xFF81},
  {0xFF82,0xFF82,0xFF82},  {0xFF83,0xFF83,0xFF83},
  {0xFF84,0xFF84,0xFF84},  {0xFF85,0xFF85,0xFF85},
  {0xFF86,0xFF86,0xFF86},  {0xFF87,0xFF87,0xFF87},
  {0xFF88,0xFF88,0xFF88},  {0xFF89,0xFF89,0xFF89},
  {0xFF8A,0xFF8A,0xFF8A},  {0xFF8B,0xFF8B,0xFF8B},
  {0xFF8C,0xFF8C,0xFF8C},  {0xFF8D,0xFF8D,0xFF8D},
  {0xFF8E,0xFF8E,0xFF8E},  {0xFF8F,0xFF8F,0xFF8F},
  {0xFF90,0xFF90,0xFF90},  {0xFF91,0xFF91,0xFF91},
  {0xFF92,0xFF92,0xFF92},  {0xFF93,0xFF93,0xFF93},
  {0xFF94,0xFF94,0xFF94},  {0xFF95,0xFF95,0xFF95},
  {0xFF96,0xFF96,0xFF96},  {0xFF97,0xFF97,0xFF97},
  {0xFF98,0xFF98,0xFF98},  {0xFF99,0xFF99,0xFF99},
  {0xFF9A,0xFF9A,0xFF9A},  {0xFF9B,0xFF9B,0xFF9B},
  {0xFF9C,0xFF9C,0xFF9C},  {0xFF9D,0xFF9D,0xFF9D},
  {0xFF9E,0xFF9E,0xFF9E},  {0xFF9F,0xFF9F,0xFF9F},
  {0xFFA0,0xFFA0,0xFFA0},  {0xFFA1,0xFFA1,0xFFA1},
  {0xFFA2,0xFFA2,0xFFA2},  {0xFFA3,0xFFA3,0xFFA3},
  {0xFFA4,0xFFA4,0xFFA4},  {0xFFA5,0xFFA5,0xFFA5},
  {0xFFA6,0xFFA6,0xFFA6},  {0xFFA7,0xFFA7,0xFFA7},
  {0xFFA8,0xFFA8,0xFFA8},  {0xFFA9,0xFFA9,0xFFA9},
  {0xFFAA,0xFFAA,0xFFAA},  {0xFFAB,0xFFAB,0xFFAB},
  {0xFFAC,0xFFAC,0xFFAC},  {0xFFAD,0xFFAD,0xFFAD},
  {0xFFAE,0xFFAE,0xFFAE},  {0xFFAF,0xFFAF,0xFFAF},
  {0xFFB0,0xFFB0,0xFFB0},  {0xFFB1,0xFFB1,0xFFB1},
  {0xFFB2,0xFFB2,0xFFB2},  {0xFFB3,0xFFB3,0xFFB3},
  {0xFFB4,0xFFB4,0xFFB4},  {0xFFB5,0xFFB5,0xFFB5},
  {0xFFB6,0xFFB6,0xFFB6},  {0xFFB7,0xFFB7,0xFFB7},
  {0xFFB8,0xFFB8,0xFFB8},  {0xFFB9,0xFFB9,0xFFB9},
  {0xFFBA,0xFFBA,0xFFBA},  {0xFFBB,0xFFBB,0xFFBB},
  {0xFFBC,0xFFBC,0xFFBC},  {0xFFBD,0xFFBD,0xFFBD},
  {0xFFBE,0xFFBE,0xFFBE},  {0xFFBF,0xFFBF,0xFFBF},
  {0xFFC0,0xFFC0,0xFFC0},  {0xFFC1,0xFFC1,0xFFC1},
  {0xFFC2,0xFFC2,0xFFC2},  {0xFFC3,0xFFC3,0xFFC3},
  {0xFFC4,0xFFC4,0xFFC4},  {0xFFC5,0xFFC5,0xFFC5},
  {0xFFC6,0xFFC6,0xFFC6},  {0xFFC7,0xFFC7,0xFFC7},
  {0xFFC8,0xFFC8,0xFFC8},  {0xFFC9,0xFFC9,0xFFC9},
  {0xFFCA,0xFFCA,0xFFCA},  {0xFFCB,0xFFCB,0xFFCB},
  {0xFFCC,0xFFCC,0xFFCC},  {0xFFCD,0xFFCD,0xFFCD},
  {0xFFCE,0xFFCE,0xFFCE},  {0xFFCF,0xFFCF,0xFFCF},
  {0xFFD0,0xFFD0,0xFFD0},  {0xFFD1,0xFFD1,0xFFD1},
  {0xFFD2,0xFFD2,0xFFD2},  {0xFFD3,0xFFD3,0xFFD3},
  {0xFFD4,0xFFD4,0xFFD4},  {0xFFD5,0xFFD5,0xFFD5},
  {0xFFD6,0xFFD6,0xFFD6},  {0xFFD7,0xFFD7,0xFFD7},
  {0xFFD8,0xFFD8,0xFFD8},  {0xFFD9,0xFFD9,0xFFD9},
  {0xFFDA,0xFFDA,0xFFDA},  {0xFFDB,0xFFDB,0xFFDB},
  {0xFFDC,0xFFDC,0xFFDC},  {0xFFDD,0xFFDD,0xFFDD},
  {0xFFDE,0xFFDE,0xFFDE},  {0xFFDF,0xFFDF,0xFFDF},
  {0xFFE0,0xFFE0,0xFFE0},  {0xFFE1,0xFFE1,0xFFE1},
  {0xFFE2,0xFFE2,0xFFE2},  {0xFFE3,0xFFE3,0xFFE3},
  {0xFFE4,0xFFE4,0xFFE4},  {0xFFE5,0xFFE5,0xFFE5},
  {0xFFE6,0xFFE6,0xFFE6},  {0xFFE7,0xFFE7,0xFFE7},
  {0xFFE8,0xFFE8,0xFFE8},  {0xFFE9,0xFFE9,0xFFE9},
  {0xFFEA,0xFFEA,0xFFEA},  {0xFFEB,0xFFEB,0xFFEB},
  {0xFFEC,0xFFEC,0xFFEC},  {0xFFED,0xFFED,0xFFED},
  {0xFFEE,0xFFEE,0xFFEE},  {0xFFEF,0xFFEF,0xFFEF},
  {0xFFF0,0xFFF0,0xFFF0},  {0xFFF1,0xFFF1,0xFFF1},
  {0xFFF2,0xFFF2,0xFFF2},  {0xFFF3,0xFFF3,0xFFF3},
  {0xFFF4,0xFFF4,0xFFF4},  {0xFFF5,0xFFF5,0xFFF5},
  {0xFFF6,0xFFF6,0xFFF6},  {0xFFF7,0xFFF7,0xFFF7},
  {0xFFF8,0xFFF8,0xFFF8},  {0xFFF9,0xFFF9,0xFFF9},
  {0xFFFA,0xFFFA,0xFFFA},  {0xFFFB,0xFFFB,0xFFFB},
  {0xFFFC,0xFFFC,0xFFFC},  {0xFFFD,0xFFFD,0xFFFD},
  {0xFFFE,0xFFFE,0xFFFE},  {0xFFFF,0xFFFF,0xFFFF}
};


MY_UNICASE_CHARACTER *my_unicase_default_pages[256]=
{
    my_unicase_default_page00,
             plane01, plane02, plane03, plane04, plane05, NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, plane1E, plane1F,
    NULL, plane21,    NULL,    NULL, plane24,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, planeFF
};


MY_UNICASE_INFO my_unicase_default=
{
  MY_UNICASE_INFO_DEFAULT_MAXCHAR,
  my_unicase_default_pages
};


/*
  Reproduce old utf8_general_ci behaviour before we fixed Bug#27877.
*/
MY_UNICASE_CHARACTER *my_unicase_pages_mysql500[256]={
 plane00_mysql500,
          plane01, plane02, plane03, plane04, plane05,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, plane1E, plane1F,
    NULL, plane21,    NULL,    NULL, plane24,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, planeFF

};


MY_UNICASE_INFO my_unicase_mysql500=
{
  0xFFFF,
  my_unicase_pages_mysql500
};


/*
  Turkish lower/upper mapping:
  1. LOWER(0x0049 LATIN CAPITAL LETTER I) -> 
           0x0131 LATIN SMALL   LETTER DOTLESS I
  2. UPPER(0x0069 LATIN SMALL   LETTER I) -> 
           0x0130 LATIN CAPITAL LETTER I WITH DOT ABOVE
*/

static MY_UNICASE_CHARACTER turk00[]=
{
  {0x0000,0x0000,0x0000},  {0x0001,0x0001,0x0001},
  {0x0002,0x0002,0x0002},  {0x0003,0x0003,0x0003},
  {0x0004,0x0004,0x0004},  {0x0005,0x0005,0x0005},
  {0x0006,0x0006,0x0006},  {0x0007,0x0007,0x0007},
  {0x0008,0x0008,0x0008},  {0x0009,0x0009,0x0009},
  {0x000A,0x000A,0x000A},  {0x000B,0x000B,0x000B},
  {0x000C,0x000C,0x000C},  {0x000D,0x000D,0x000D},
  {0x000E,0x000E,0x000E},  {0x000F,0x000F,0x000F},
  {0x0010,0x0010,0x0010},  {0x0011,0x0011,0x0011},
  {0x0012,0x0012,0x0012},  {0x0013,0x0013,0x0013},
  {0x0014,0x0014,0x0014},  {0x0015,0x0015,0x0015},
  {0x0016,0x0016,0x0016},  {0x0017,0x0017,0x0017},
  {0x0018,0x0018,0x0018},  {0x0019,0x0019,0x0019},
  {0x001A,0x001A,0x001A},  {0x001B,0x001B,0x001B},
  {0x001C,0x001C,0x001C},  {0x001D,0x001D,0x001D},
  {0x001E,0x001E,0x001E},  {0x001F,0x001F,0x001F},
  {0x0020,0x0020,0x0020},  {0x0021,0x0021,0x0021},
  {0x0022,0x0022,0x0022},  {0x0023,0x0023,0x0023},
  {0x0024,0x0024,0x0024},  {0x0025,0x0025,0x0025},
  {0x0026,0x0026,0x0026},  {0x0027,0x0027,0x0027},
  {0x0028,0x0028,0x0028},  {0x0029,0x0029,0x0029},
  {0x002A,0x002A,0x002A},  {0x002B,0x002B,0x002B},
  {0x002C,0x002C,0x002C},  {0x002D,0x002D,0x002D},
  {0x002E,0x002E,0x002E},  {0x002F,0x002F,0x002F},
  {0x0030,0x0030,0x0030},  {0x0031,0x0031,0x0031},
  {0x0032,0x0032,0x0032},  {0x0033,0x0033,0x0033},
  {0x0034,0x0034,0x0034},  {0x0035,0x0035,0x0035},
  {0x0036,0x0036,0x0036},  {0x0037,0x0037,0x0037},
  {0x0038,0x0038,0x0038},  {0x0039,0x0039,0x0039},
  {0x003A,0x003A,0x003A},  {0x003B,0x003B,0x003B},
  {0x003C,0x003C,0x003C},  {0x003D,0x003D,0x003D},
  {0x003E,0x003E,0x003E},  {0x003F,0x003F,0x003F},
  {0x0040,0x0040,0x0040},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0049,0x0131,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x005B,0x005B,0x005B},
  {0x005C,0x005C,0x005C},  {0x005D,0x005D,0x005D},
  {0x005E,0x005E,0x005E},  {0x005F,0x005F,0x005F},
  {0x0060,0x0060,0x0060},  {0x0041,0x0061,0x0041},
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043},
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045},
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047},
  {0x0048,0x0068,0x0048},  {0x0130,0x0069,0x0049},
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B},
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D},
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F},
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051},
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053},
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055},
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057},
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059},
  {0x005A,0x007A,0x005A},  {0x007B,0x007B,0x007B},
  {0x007C,0x007C,0x007C},  {0x007D,0x007D,0x007D},
  {0x007E,0x007E,0x007E},  {0x007F,0x007F,0x007F},
  {0x0080,0x0080,0x0080},  {0x0081,0x0081,0x0081},
  {0x0082,0x0082,0x0082},  {0x0083,0x0083,0x0083},
  {0x0084,0x0084,0x0084},  {0x0085,0x0085,0x0085},
  {0x0086,0x0086,0x0086},  {0x0087,0x0087,0x0087},
  {0x0088,0x0088,0x0088},  {0x0089,0x0089,0x0089},
  {0x008A,0x008A,0x008A},  {0x008B,0x008B,0x008B},
  {0x008C,0x008C,0x008C},  {0x008D,0x008D,0x008D},
  {0x008E,0x008E,0x008E},  {0x008F,0x008F,0x008F},
  {0x0090,0x0090,0x0090},  {0x0091,0x0091,0x0091},
  {0x0092,0x0092,0x0092},  {0x0093,0x0093,0x0093},
  {0x0094,0x0094,0x0094},  {0x0095,0x0095,0x0095},
  {0x0096,0x0096,0x0096},  {0x0097,0x0097,0x0097},
  {0x0098,0x0098,0x0098},  {0x0099,0x0099,0x0099},
  {0x009A,0x009A,0x009A},  {0x009B,0x009B,0x009B},
  {0x009C,0x009C,0x009C},  {0x009D,0x009D,0x009D},
  {0x009E,0x009E,0x009E},  {0x009F,0x009F,0x009F},
  {0x00A0,0x00A0,0x00A0},  {0x00A1,0x00A1,0x00A1},
  {0x00A2,0x00A2,0x00A2},  {0x00A3,0x00A3,0x00A3},
  {0x00A4,0x00A4,0x00A4},  {0x00A5,0x00A5,0x00A5},
  {0x00A6,0x00A6,0x00A6},  {0x00A7,0x00A7,0x00A7},
  {0x00A8,0x00A8,0x00A8},  {0x00A9,0x00A9,0x00A9},
  {0x00AA,0x00AA,0x00AA},  {0x00AB,0x00AB,0x00AB},
  {0x00AC,0x00AC,0x00AC},  {0x00AD,0x00AD,0x00AD},
  {0x00AE,0x00AE,0x00AE},  {0x00AF,0x00AF,0x00AF},
  {0x00B0,0x00B0,0x00B0},  {0x00B1,0x00B1,0x00B1},
  {0x00B2,0x00B2,0x00B2},  {0x00B3,0x00B3,0x00B3},
  {0x00B4,0x00B4,0x00B4},  {0x039C,0x00B5,0x039C},
  {0x00B6,0x00B6,0x00B6},  {0x00B7,0x00B7,0x00B7},
  {0x00B8,0x00B8,0x00B8},  {0x00B9,0x00B9,0x00B9},
  {0x00BA,0x00BA,0x00BA},  {0x00BB,0x00BB,0x00BB},
  {0x00BC,0x00BC,0x00BC},  {0x00BD,0x00BD,0x00BD},
  {0x00BE,0x00BE,0x00BE},  {0x00BF,0x00BF,0x00BF},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00D7,0x00D7,0x00D7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x00DF,0x00DF,0x00DF},
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041},
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041},
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041},
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043},
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045},
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045},
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049},
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049},
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E},
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F},
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F},
  {0x00D6,0x00F6,0x004F},  {0x00F7,0x00F7,0x00F7},
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055},
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055},
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059},
  {0x00DE,0x00FE,0x00DE},  {0x0178,0x00FF,0x0059}
};



static MY_UNICASE_CHARACTER *my_unicase_pages_turkish[256]=
{
  turk00, plane01, plane02, plane03, plane04, plane05,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, plane1E, plane1F,
    NULL, plane21,    NULL,    NULL, plane24,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, planeFF
};


MY_UNICASE_INFO my_unicase_turkish=
{
  0xFFFF,
  my_unicase_pages_turkish
};


/* Unicode-5.2.0 case folding information */
static MY_UNICASE_CHARACTER u520p00[]={
  {0x0000,0x0000,0x0000},  {0x0001,0x0001,0x0001}, /* 0000 */
  {0x0002,0x0002,0x0002},  {0x0003,0x0003,0x0003}, /* 0002 */
  {0x0004,0x0004,0x0004},  {0x0005,0x0005,0x0005}, /* 0004 */
  {0x0006,0x0006,0x0006},  {0x0007,0x0007,0x0007}, /* 0006 */
  {0x0008,0x0008,0x0008},  {0x0009,0x0009,0x0009}, /* 0008 */
  {0x000A,0x000A,0x000A},  {0x000B,0x000B,0x000B}, /* 000A */
  {0x000C,0x000C,0x000C},  {0x000D,0x000D,0x000D}, /* 000C */
  {0x000E,0x000E,0x000E},  {0x000F,0x000F,0x000F}, /* 000E */
  {0x0010,0x0010,0x0010},  {0x0011,0x0011,0x0011}, /* 0010 */
  {0x0012,0x0012,0x0012},  {0x0013,0x0013,0x0013}, /* 0012 */
  {0x0014,0x0014,0x0014},  {0x0015,0x0015,0x0015}, /* 0014 */
  {0x0016,0x0016,0x0016},  {0x0017,0x0017,0x0017}, /* 0016 */
  {0x0018,0x0018,0x0018},  {0x0019,0x0019,0x0019}, /* 0018 */
  {0x001A,0x001A,0x001A},  {0x001B,0x001B,0x001B}, /* 001A */
  {0x001C,0x001C,0x001C},  {0x001D,0x001D,0x001D}, /* 001C */
  {0x001E,0x001E,0x001E},  {0x001F,0x001F,0x001F}, /* 001E */
  {0x0020,0x0020,0x0020},  {0x0021,0x0021,0x0021}, /* 0020 */
  {0x0022,0x0022,0x0022},  {0x0023,0x0023,0x0023}, /* 0022 */
  {0x0024,0x0024,0x0024},  {0x0025,0x0025,0x0025}, /* 0024 */
  {0x0026,0x0026,0x0026},  {0x0027,0x0027,0x0027}, /* 0026 */
  {0x0028,0x0028,0x0028},  {0x0029,0x0029,0x0029}, /* 0028 */
  {0x002A,0x002A,0x002A},  {0x002B,0x002B,0x002B}, /* 002A */
  {0x002C,0x002C,0x002C},  {0x002D,0x002D,0x002D}, /* 002C */
  {0x002E,0x002E,0x002E},  {0x002F,0x002F,0x002F}, /* 002E */
  {0x0030,0x0030,0x0030},  {0x0031,0x0031,0x0031}, /* 0030 */
  {0x0032,0x0032,0x0032},  {0x0033,0x0033,0x0033}, /* 0032 */
  {0x0034,0x0034,0x0034},  {0x0035,0x0035,0x0035}, /* 0034 */
  {0x0036,0x0036,0x0036},  {0x0037,0x0037,0x0037}, /* 0036 */
  {0x0038,0x0038,0x0038},  {0x0039,0x0039,0x0039}, /* 0038 */
  {0x003A,0x003A,0x003A},  {0x003B,0x003B,0x003B}, /* 003A */
  {0x003C,0x003C,0x003C},  {0x003D,0x003D,0x003D}, /* 003C */
  {0x003E,0x003E,0x003E},  {0x003F,0x003F,0x003F}, /* 003E */
  {0x0040,0x0040,0x0040},  {0x0041,0x0061,0x0041}, /* 0040 */
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043}, /* 0042 */
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045}, /* 0044 */
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047}, /* 0046 */
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049}, /* 0048 */
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B}, /* 004A */
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D}, /* 004C */
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F}, /* 004E */
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051}, /* 0050 */
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053}, /* 0052 */
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055}, /* 0054 */
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057}, /* 0056 */
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059}, /* 0058 */
  {0x005A,0x007A,0x005A},  {0x005B,0x005B,0x005B}, /* 005A */
  {0x005C,0x005C,0x005C},  {0x005D,0x005D,0x005D}, /* 005C */
  {0x005E,0x005E,0x005E},  {0x005F,0x005F,0x005F}, /* 005E */
  {0x0060,0x0060,0x0060},  {0x0041,0x0061,0x0041}, /* 0060 */
  {0x0042,0x0062,0x0042},  {0x0043,0x0063,0x0043}, /* 0062 */
  {0x0044,0x0064,0x0044},  {0x0045,0x0065,0x0045}, /* 0064 */
  {0x0046,0x0066,0x0046},  {0x0047,0x0067,0x0047}, /* 0066 */
  {0x0048,0x0068,0x0048},  {0x0049,0x0069,0x0049}, /* 0068 */
  {0x004A,0x006A,0x004A},  {0x004B,0x006B,0x004B}, /* 006A */
  {0x004C,0x006C,0x004C},  {0x004D,0x006D,0x004D}, /* 006C */
  {0x004E,0x006E,0x004E},  {0x004F,0x006F,0x004F}, /* 006E */
  {0x0050,0x0070,0x0050},  {0x0051,0x0071,0x0051}, /* 0070 */
  {0x0052,0x0072,0x0052},  {0x0053,0x0073,0x0053}, /* 0072 */
  {0x0054,0x0074,0x0054},  {0x0055,0x0075,0x0055}, /* 0074 */
  {0x0056,0x0076,0x0056},  {0x0057,0x0077,0x0057}, /* 0076 */
  {0x0058,0x0078,0x0058},  {0x0059,0x0079,0x0059}, /* 0078 */
  {0x005A,0x007A,0x005A},  {0x007B,0x007B,0x007B}, /* 007A */
  {0x007C,0x007C,0x007C},  {0x007D,0x007D,0x007D}, /* 007C */
  {0x007E,0x007E,0x007E},  {0x007F,0x007F,0x007F}, /* 007E */
  {0x0080,0x0080,0x0080},  {0x0081,0x0081,0x0081}, /* 0080 */
  {0x0082,0x0082,0x0082},  {0x0083,0x0083,0x0083}, /* 0082 */
  {0x0084,0x0084,0x0084},  {0x0085,0x0085,0x0085}, /* 0084 */
  {0x0086,0x0086,0x0086},  {0x0087,0x0087,0x0087}, /* 0086 */
  {0x0088,0x0088,0x0088},  {0x0089,0x0089,0x0089}, /* 0088 */
  {0x008A,0x008A,0x008A},  {0x008B,0x008B,0x008B}, /* 008A */
  {0x008C,0x008C,0x008C},  {0x008D,0x008D,0x008D}, /* 008C */
  {0x008E,0x008E,0x008E},  {0x008F,0x008F,0x008F}, /* 008E */
  {0x0090,0x0090,0x0090},  {0x0091,0x0091,0x0091}, /* 0090 */
  {0x0092,0x0092,0x0092},  {0x0093,0x0093,0x0093}, /* 0092 */
  {0x0094,0x0094,0x0094},  {0x0095,0x0095,0x0095}, /* 0094 */
  {0x0096,0x0096,0x0096},  {0x0097,0x0097,0x0097}, /* 0096 */
  {0x0098,0x0098,0x0098},  {0x0099,0x0099,0x0099}, /* 0098 */
  {0x009A,0x009A,0x009A},  {0x009B,0x009B,0x009B}, /* 009A */
  {0x009C,0x009C,0x009C},  {0x009D,0x009D,0x009D}, /* 009C */
  {0x009E,0x009E,0x009E},  {0x009F,0x009F,0x009F}, /* 009E */
  {0x00A0,0x00A0,0x00A0},  {0x00A1,0x00A1,0x00A1}, /* 00A0 */
  {0x00A2,0x00A2,0x00A2},  {0x00A3,0x00A3,0x00A3}, /* 00A2 */
  {0x00A4,0x00A4,0x00A4},  {0x00A5,0x00A5,0x00A5}, /* 00A4 */
  {0x00A6,0x00A6,0x00A6},  {0x00A7,0x00A7,0x00A7}, /* 00A6 */
  {0x00A8,0x00A8,0x00A8},  {0x00A9,0x00A9,0x00A9}, /* 00A8 */
  {0x00AA,0x00AA,0x00AA},  {0x00AB,0x00AB,0x00AB}, /* 00AA */
  {0x00AC,0x00AC,0x00AC},  {0x00AD,0x00AD,0x00AD}, /* 00AC */
  {0x00AE,0x00AE,0x00AE},  {0x00AF,0x00AF,0x00AF}, /* 00AE */
  {0x00B0,0x00B0,0x00B0},  {0x00B1,0x00B1,0x00B1}, /* 00B0 */
  {0x00B2,0x00B2,0x00B2},  {0x00B3,0x00B3,0x00B3}, /* 00B2 */
  {0x00B4,0x00B4,0x00B4},  {0x039C,0x00B5,0x039C}, /* 00B4 */
  {0x00B6,0x00B6,0x00B6},  {0x00B7,0x00B7,0x00B7}, /* 00B6 */
  {0x00B8,0x00B8,0x00B8},  {0x00B9,0x00B9,0x00B9}, /* 00B8 */
  {0x00BA,0x00BA,0x00BA},  {0x00BB,0x00BB,0x00BB}, /* 00BA */
  {0x00BC,0x00BC,0x00BC},  {0x00BD,0x00BD,0x00BD}, /* 00BC */
  {0x00BE,0x00BE,0x00BE},  {0x00BF,0x00BF,0x00BF}, /* 00BE */
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041}, /* 00C0 */
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041}, /* 00C2 */
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041}, /* 00C4 */
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043}, /* 00C6 */
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045}, /* 00C8 */
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045}, /* 00CA */
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049}, /* 00CC */
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049}, /* 00CE */
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E}, /* 00D0 */
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F}, /* 00D2 */
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F}, /* 00D4 */
  {0x00D6,0x00F6,0x004F},  {0x00D7,0x00D7,0x00D7}, /* 00D6 */
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055}, /* 00D8 */
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055}, /* 00DA */
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059}, /* 00DC */
  {0x00DE,0x00FE,0x00DE},  {0x00DF,0x00DF,0x0053}, /* 00DE */
  {0x00C0,0x00E0,0x0041},  {0x00C1,0x00E1,0x0041}, /* 00E0 */
  {0x00C2,0x00E2,0x0041},  {0x00C3,0x00E3,0x0041}, /* 00E2 */
  {0x00C4,0x00E4,0x0041},  {0x00C5,0x00E5,0x0041}, /* 00E4 */
  {0x00C6,0x00E6,0x00C6},  {0x00C7,0x00E7,0x0043}, /* 00E6 */
  {0x00C8,0x00E8,0x0045},  {0x00C9,0x00E9,0x0045}, /* 00E8 */
  {0x00CA,0x00EA,0x0045},  {0x00CB,0x00EB,0x0045}, /* 00EA */
  {0x00CC,0x00EC,0x0049},  {0x00CD,0x00ED,0x0049}, /* 00EC */
  {0x00CE,0x00EE,0x0049},  {0x00CF,0x00EF,0x0049}, /* 00EE */
  {0x00D0,0x00F0,0x00D0},  {0x00D1,0x00F1,0x004E}, /* 00F0 */
  {0x00D2,0x00F2,0x004F},  {0x00D3,0x00F3,0x004F}, /* 00F2 */
  {0x00D4,0x00F4,0x004F},  {0x00D5,0x00F5,0x004F}, /* 00F4 */
  {0x00D6,0x00F6,0x004F},  {0x00F7,0x00F7,0x00F7}, /* 00F6 */
  {0x00D8,0x00F8,0x00D8},  {0x00D9,0x00F9,0x0055}, /* 00F8 */
  {0x00DA,0x00FA,0x0055},  {0x00DB,0x00FB,0x0055}, /* 00FA */
  {0x00DC,0x00FC,0x0055},  {0x00DD,0x00FD,0x0059}, /* 00FC */
  {0x00DE,0x00FE,0x00DE},  {0x0178,0x00FF,0x0059} /* 00FE */
};

static MY_UNICASE_CHARACTER u520p01[]={
  {0x0100,0x0101,0x0041},  {0x0100,0x0101,0x0041}, /* 0100 */
  {0x0102,0x0103,0x0041},  {0x0102,0x0103,0x0041}, /* 0102 */
  {0x0104,0x0105,0x0041},  {0x0104,0x0105,0x0041}, /* 0104 */
  {0x0106,0x0107,0x0043},  {0x0106,0x0107,0x0043}, /* 0106 */
  {0x0108,0x0109,0x0043},  {0x0108,0x0109,0x0043}, /* 0108 */
  {0x010A,0x010B,0x0043},  {0x010A,0x010B,0x0043}, /* 010A */
  {0x010C,0x010D,0x0043},  {0x010C,0x010D,0x0043}, /* 010C */
  {0x010E,0x010F,0x0044},  {0x010E,0x010F,0x0044}, /* 010E */
  {0x0110,0x0111,0x0110},  {0x0110,0x0111,0x0110}, /* 0110 */
  {0x0112,0x0113,0x0045},  {0x0112,0x0113,0x0045}, /* 0112 */
  {0x0114,0x0115,0x0045},  {0x0114,0x0115,0x0045}, /* 0114 */
  {0x0116,0x0117,0x0045},  {0x0116,0x0117,0x0045}, /* 0116 */
  {0x0118,0x0119,0x0045},  {0x0118,0x0119,0x0045}, /* 0118 */
  {0x011A,0x011B,0x0045},  {0x011A,0x011B,0x0045}, /* 011A */
  {0x011C,0x011D,0x0047},  {0x011C,0x011D,0x0047}, /* 011C */
  {0x011E,0x011F,0x0047},  {0x011E,0x011F,0x0047}, /* 011E */
  {0x0120,0x0121,0x0047},  {0x0120,0x0121,0x0047}, /* 0120 */
  {0x0122,0x0123,0x0047},  {0x0122,0x0123,0x0047}, /* 0122 */
  {0x0124,0x0125,0x0048},  {0x0124,0x0125,0x0048}, /* 0124 */
  {0x0126,0x0127,0x0126},  {0x0126,0x0127,0x0126}, /* 0126 */
  {0x0128,0x0129,0x0049},  {0x0128,0x0129,0x0049}, /* 0128 */
  {0x012A,0x012B,0x0049},  {0x012A,0x012B,0x0049}, /* 012A */
  {0x012C,0x012D,0x0049},  {0x012C,0x012D,0x0049}, /* 012C */
  {0x012E,0x012F,0x0049},  {0x012E,0x012F,0x0049}, /* 012E */
  {0x0130,0x0069,0x0049},  {0x0049,0x0131,0x0049}, /* 0130 */
  {0x0132,0x0133,0x0132},  {0x0132,0x0133,0x0132}, /* 0132 */
  {0x0134,0x0135,0x004A},  {0x0134,0x0135,0x004A}, /* 0134 */
  {0x0136,0x0137,0x004B},  {0x0136,0x0137,0x004B}, /* 0136 */
  {0x0138,0x0138,0x0138},  {0x0139,0x013A,0x004C}, /* 0138 */
  {0x0139,0x013A,0x004C},  {0x013B,0x013C,0x004C}, /* 013A */
  {0x013B,0x013C,0x004C},  {0x013D,0x013E,0x004C}, /* 013C */
  {0x013D,0x013E,0x004C},  {0x013F,0x0140,0x013F}, /* 013E */
  {0x013F,0x0140,0x013F},  {0x0141,0x0142,0x0141}, /* 0140 */
  {0x0141,0x0142,0x0141},  {0x0143,0x0144,0x004E}, /* 0142 */
  {0x0143,0x0144,0x004E},  {0x0145,0x0146,0x004E}, /* 0144 */
  {0x0145,0x0146,0x004E},  {0x0147,0x0148,0x004E}, /* 0146 */
  {0x0147,0x0148,0x004E},  {0x0149,0x0149,0x0149}, /* 0148 */
  {0x014A,0x014B,0x014A},  {0x014A,0x014B,0x014A}, /* 014A */
  {0x014C,0x014D,0x004F},  {0x014C,0x014D,0x004F}, /* 014C */
  {0x014E,0x014F,0x004F},  {0x014E,0x014F,0x004F}, /* 014E */
  {0x0150,0x0151,0x004F},  {0x0150,0x0151,0x004F}, /* 0150 */
  {0x0152,0x0153,0x0152},  {0x0152,0x0153,0x0152}, /* 0152 */
  {0x0154,0x0155,0x0052},  {0x0154,0x0155,0x0052}, /* 0154 */
  {0x0156,0x0157,0x0052},  {0x0156,0x0157,0x0052}, /* 0156 */
  {0x0158,0x0159,0x0052},  {0x0158,0x0159,0x0052}, /* 0158 */
  {0x015A,0x015B,0x0053},  {0x015A,0x015B,0x0053}, /* 015A */
  {0x015C,0x015D,0x0053},  {0x015C,0x015D,0x0053}, /* 015C */
  {0x015E,0x015F,0x0053},  {0x015E,0x015F,0x0053}, /* 015E */
  {0x0160,0x0161,0x0053},  {0x0160,0x0161,0x0053}, /* 0160 */
  {0x0162,0x0163,0x0054},  {0x0162,0x0163,0x0054}, /* 0162 */
  {0x0164,0x0165,0x0054},  {0x0164,0x0165,0x0054}, /* 0164 */
  {0x0166,0x0167,0x0166},  {0x0166,0x0167,0x0166}, /* 0166 */
  {0x0168,0x0169,0x0055},  {0x0168,0x0169,0x0055}, /* 0168 */
  {0x016A,0x016B,0x0055},  {0x016A,0x016B,0x0055}, /* 016A */
  {0x016C,0x016D,0x0055},  {0x016C,0x016D,0x0055}, /* 016C */
  {0x016E,0x016F,0x0055},  {0x016E,0x016F,0x0055}, /* 016E */
  {0x0170,0x0171,0x0055},  {0x0170,0x0171,0x0055}, /* 0170 */
  {0x0172,0x0173,0x0055},  {0x0172,0x0173,0x0055}, /* 0172 */
  {0x0174,0x0175,0x0057},  {0x0174,0x0175,0x0057}, /* 0174 */
  {0x0176,0x0177,0x0059},  {0x0176,0x0177,0x0059}, /* 0176 */
  {0x0178,0x00FF,0x0059},  {0x0179,0x017A,0x005A}, /* 0178 */
  {0x0179,0x017A,0x005A},  {0x017B,0x017C,0x005A}, /* 017A */
  {0x017B,0x017C,0x005A},  {0x017D,0x017E,0x005A}, /* 017C */
  {0x017D,0x017E,0x005A},  {0x0053,0x017F,0x0053}, /* 017E */
  {0x0243,0x0180,0x0243},  {0x0181,0x0253,0x0181}, /* 0180 */
  {0x0182,0x0183,0x0182},  {0x0182,0x0183,0x0182}, /* 0182 */
  {0x0184,0x0185,0x0184},  {0x0184,0x0185,0x0184}, /* 0184 */
  {0x0186,0x0254,0x0186},  {0x0187,0x0188,0x0187}, /* 0186 */
  {0x0187,0x0188,0x0187},  {0x0189,0x0256,0x0189}, /* 0188 */
  {0x018A,0x0257,0x018A},  {0x018B,0x018C,0x018B}, /* 018A */
  {0x018B,0x018C,0x018B},  {0x018D,0x018D,0x018D}, /* 018C */
  {0x018E,0x01DD,0x018E},  {0x018F,0x0259,0x018F}, /* 018E */
  {0x0190,0x025B,0x0190},  {0x0191,0x0192,0x0191}, /* 0190 */
  {0x0191,0x0192,0x0191},  {0x0193,0x0260,0x0193}, /* 0192 */
  {0x0194,0x0263,0x0194},  {0x01F6,0x0195,0x01F6}, /* 0194 */
  {0x0196,0x0269,0x0196},  {0x0197,0x0268,0x0197}, /* 0196 */
  {0x0198,0x0199,0x0198},  {0x0198,0x0199,0x0198}, /* 0198 */
  {0x023D,0x019A,0x023D},  {0x019B,0x019B,0x019B}, /* 019A */
  {0x019C,0x026F,0x019C},  {0x019D,0x0272,0x019D}, /* 019C */
  {0x0220,0x019E,0x0220},  {0x019F,0x0275,0x019F}, /* 019E */
  {0x01A0,0x01A1,0x004F},  {0x01A0,0x01A1,0x004F}, /* 01A0 */
  {0x01A2,0x01A3,0x01A2},  {0x01A2,0x01A3,0x01A2}, /* 01A2 */
  {0x01A4,0x01A5,0x01A4},  {0x01A4,0x01A5,0x01A4}, /* 01A4 */
  {0x01A6,0x0280,0x01A6},  {0x01A7,0x01A8,0x01A7}, /* 01A6 */
  {0x01A7,0x01A8,0x01A7},  {0x01A9,0x0283,0x01A9}, /* 01A8 */
  {0x01AA,0x01AA,0x01AA},  {0x01AB,0x01AB,0x01AB}, /* 01AA */
  {0x01AC,0x01AD,0x01AC},  {0x01AC,0x01AD,0x01AC}, /* 01AC */
  {0x01AE,0x0288,0x01AE},  {0x01AF,0x01B0,0x0055}, /* 01AE */
  {0x01AF,0x01B0,0x0055},  {0x01B1,0x028A,0x01B1}, /* 01B0 */
  {0x01B2,0x028B,0x01B2},  {0x01B3,0x01B4,0x01B3}, /* 01B2 */
  {0x01B3,0x01B4,0x01B3},  {0x01B5,0x01B6,0x01B5}, /* 01B4 */
  {0x01B5,0x01B6,0x01B5},  {0x01B7,0x0292,0x01B7}, /* 01B6 */
  {0x01B8,0x01B9,0x01B8},  {0x01B8,0x01B9,0x01B8}, /* 01B8 */
  {0x01BA,0x01BA,0x01BA},  {0x01BB,0x01BB,0x01BB}, /* 01BA */
  {0x01BC,0x01BD,0x01BC},  {0x01BC,0x01BD,0x01BC}, /* 01BC */
  {0x01BE,0x01BE,0x01BE},  {0x01F7,0x01BF,0x01F7}, /* 01BE */
  {0x01C0,0x01C0,0x01C0},  {0x01C1,0x01C1,0x01C1}, /* 01C0 */
  {0x01C2,0x01C2,0x01C2},  {0x01C3,0x01C3,0x01C3}, /* 01C2 */
  {0x01C4,0x01C6,0x01C4},  {0x01C4,0x01C6,0x01C4}, /* 01C4 */
  {0x01C4,0x01C6,0x01C4},  {0x01C7,0x01C9,0x01C7}, /* 01C6 */
  {0x01C7,0x01C9,0x01C7},  {0x01C7,0x01C9,0x01C7}, /* 01C8 */
  {0x01CA,0x01CC,0x01CA},  {0x01CA,0x01CC,0x01CA}, /* 01CA */
  {0x01CA,0x01CC,0x01CA},  {0x01CD,0x01CE,0x0041}, /* 01CC */
  {0x01CD,0x01CE,0x0041},  {0x01CF,0x01D0,0x0049}, /* 01CE */
  {0x01CF,0x01D0,0x0049},  {0x01D1,0x01D2,0x004F}, /* 01D0 */
  {0x01D1,0x01D2,0x004F},  {0x01D3,0x01D4,0x0055}, /* 01D2 */
  {0x01D3,0x01D4,0x0055},  {0x01D5,0x01D6,0x0055}, /* 01D4 */
  {0x01D5,0x01D6,0x0055},  {0x01D7,0x01D8,0x0055}, /* 01D6 */
  {0x01D7,0x01D8,0x0055},  {0x01D9,0x01DA,0x0055}, /* 01D8 */
  {0x01D9,0x01DA,0x0055},  {0x01DB,0x01DC,0x0055}, /* 01DA */
  {0x01DB,0x01DC,0x0055},  {0x018E,0x01DD,0x018E}, /* 01DC */
  {0x01DE,0x01DF,0x0041},  {0x01DE,0x01DF,0x0041}, /* 01DE */
  {0x01E0,0x01E1,0x0041},  {0x01E0,0x01E1,0x0041}, /* 01E0 */
  {0x01E2,0x01E3,0x00C6},  {0x01E2,0x01E3,0x00C6}, /* 01E2 */
  {0x01E4,0x01E5,0x01E4},  {0x01E4,0x01E5,0x01E4}, /* 01E4 */
  {0x01E6,0x01E7,0x0047},  {0x01E6,0x01E7,0x0047}, /* 01E6 */
  {0x01E8,0x01E9,0x004B},  {0x01E8,0x01E9,0x004B}, /* 01E8 */
  {0x01EA,0x01EB,0x004F},  {0x01EA,0x01EB,0x004F}, /* 01EA */
  {0x01EC,0x01ED,0x004F},  {0x01EC,0x01ED,0x004F}, /* 01EC */
  {0x01EE,0x01EF,0x01B7},  {0x01EE,0x01EF,0x01B7}, /* 01EE */
  {0x01F0,0x01F0,0x004A},  {0x01F1,0x01F3,0x01F1}, /* 01F0 */
  {0x01F1,0x01F3,0x01F1},  {0x01F1,0x01F3,0x01F1}, /* 01F2 */
  {0x01F4,0x01F5,0x0047},  {0x01F4,0x01F5,0x0047}, /* 01F4 */
  {0x01F6,0x0195,0x01F6},  {0x01F7,0x01BF,0x01F7}, /* 01F6 */
  {0x01F8,0x01F9,0x004E},  {0x01F8,0x01F9,0x004E}, /* 01F8 */
  {0x01FA,0x01FB,0x0041},  {0x01FA,0x01FB,0x0041}, /* 01FA */
  {0x01FC,0x01FD,0x00C6},  {0x01FC,0x01FD,0x00C6}, /* 01FC */
  {0x01FE,0x01FF,0x00D8},  {0x01FE,0x01FF,0x00D8} /* 01FE */
};

static MY_UNICASE_CHARACTER u520p02[]={
  {0x0200,0x0201,0x0041},  {0x0200,0x0201,0x0041}, /* 0200 */
  {0x0202,0x0203,0x0041},  {0x0202,0x0203,0x0041}, /* 0202 */
  {0x0204,0x0205,0x0045},  {0x0204,0x0205,0x0045}, /* 0204 */
  {0x0206,0x0207,0x0045},  {0x0206,0x0207,0x0045}, /* 0206 */
  {0x0208,0x0209,0x0049},  {0x0208,0x0209,0x0049}, /* 0208 */
  {0x020A,0x020B,0x0049},  {0x020A,0x020B,0x0049}, /* 020A */
  {0x020C,0x020D,0x004F},  {0x020C,0x020D,0x004F}, /* 020C */
  {0x020E,0x020F,0x004F},  {0x020E,0x020F,0x004F}, /* 020E */
  {0x0210,0x0211,0x0052},  {0x0210,0x0211,0x0052}, /* 0210 */
  {0x0212,0x0213,0x0052},  {0x0212,0x0213,0x0052}, /* 0212 */
  {0x0214,0x0215,0x0055},  {0x0214,0x0215,0x0055}, /* 0214 */
  {0x0216,0x0217,0x0055},  {0x0216,0x0217,0x0055}, /* 0216 */
  {0x0218,0x0219,0x0053},  {0x0218,0x0219,0x0053}, /* 0218 */
  {0x021A,0x021B,0x0054},  {0x021A,0x021B,0x0054}, /* 021A */
  {0x021C,0x021D,0x021C},  {0x021C,0x021D,0x021C}, /* 021C */
  {0x021E,0x021F,0x0048},  {0x021E,0x021F,0x0048}, /* 021E */
  {0x0220,0x019E,0x0220},  {0x0221,0x0221,0x0221}, /* 0220 */
  {0x0222,0x0223,0x0222},  {0x0222,0x0223,0x0222}, /* 0222 */
  {0x0224,0x0225,0x0224},  {0x0224,0x0225,0x0224}, /* 0224 */
  {0x0226,0x0227,0x0041},  {0x0226,0x0227,0x0041}, /* 0226 */
  {0x0228,0x0229,0x0045},  {0x0228,0x0229,0x0045}, /* 0228 */
  {0x022A,0x022B,0x004F},  {0x022A,0x022B,0x004F}, /* 022A */
  {0x022C,0x022D,0x004F},  {0x022C,0x022D,0x004F}, /* 022C */
  {0x022E,0x022F,0x004F},  {0x022E,0x022F,0x004F}, /* 022E */
  {0x0230,0x0231,0x004F},  {0x0230,0x0231,0x004F}, /* 0230 */
  {0x0232,0x0233,0x0059},  {0x0232,0x0233,0x0059}, /* 0232 */
  {0x0234,0x0234,0x0234},  {0x0235,0x0235,0x0235}, /* 0234 */
  {0x0236,0x0236,0x0236},  {0x0237,0x0237,0x0237}, /* 0236 */
  {0x0238,0x0238,0x0238},  {0x0239,0x0239,0x0239}, /* 0238 */
  {0x023A,0x2C65,0x023A},  {0x023B,0x023C,0x023B}, /* 023A */
  {0x023B,0x023C,0x023B},  {0x023D,0x019A,0x023D}, /* 023C */
  {0x023E,0x2C66,0x023E},  {0x2C7E,0x023F,0x2C7E}, /* 023E */
  {0x2C7F,0x0240,0x2C7F},  {0x0241,0x0242,0x0241}, /* 0240 */
  {0x0241,0x0242,0x0241},  {0x0243,0x0180,0x0243}, /* 0242 */
  {0x0244,0x0289,0x0244},  {0x0245,0x028C,0x0245}, /* 0244 */
  {0x0246,0x0247,0x0246},  {0x0246,0x0247,0x0246}, /* 0246 */
  {0x0248,0x0249,0x0248},  {0x0248,0x0249,0x0248}, /* 0248 */
  {0x024A,0x024B,0x024A},  {0x024A,0x024B,0x024A}, /* 024A */
  {0x024C,0x024D,0x024C},  {0x024C,0x024D,0x024C}, /* 024C */
  {0x024E,0x024F,0x024E},  {0x024E,0x024F,0x024E}, /* 024E */
  {0x2C6F,0x0250,0x2C6F},  {0x2C6D,0x0251,0x2C6D}, /* 0250 */
  {0x2C70,0x0252,0x2C70},  {0x0181,0x0253,0x0181}, /* 0252 */
  {0x0186,0x0254,0x0186},  {0x0255,0x0255,0x0255}, /* 0254 */
  {0x0189,0x0256,0x0189},  {0x018A,0x0257,0x018A}, /* 0256 */
  {0x0258,0x0258,0x0258},  {0x018F,0x0259,0x018F}, /* 0258 */
  {0x025A,0x025A,0x025A},  {0x0190,0x025B,0x0190}, /* 025A */
  {0x025C,0x025C,0x025C},  {0x025D,0x025D,0x025D}, /* 025C */
  {0x025E,0x025E,0x025E},  {0x025F,0x025F,0x025F}, /* 025E */
  {0x0193,0x0260,0x0193},  {0x0261,0x0261,0x0261}, /* 0260 */
  {0x0262,0x0262,0x0262},  {0x0194,0x0263,0x0194}, /* 0262 */
  {0x0264,0x0264,0x0264},  {0x0265,0x0265,0x0265}, /* 0264 */
  {0x0266,0x0266,0x0266},  {0x0267,0x0267,0x0267}, /* 0266 */
  {0x0197,0x0268,0x0197},  {0x0196,0x0269,0x0196}, /* 0268 */
  {0x026A,0x026A,0x026A},  {0x2C62,0x026B,0x2C62}, /* 026A */
  {0x026C,0x026C,0x026C},  {0x026D,0x026D,0x026D}, /* 026C */
  {0x026E,0x026E,0x026E},  {0x019C,0x026F,0x019C}, /* 026E */
  {0x0270,0x0270,0x0270},  {0x2C6E,0x0271,0x2C6E}, /* 0270 */
  {0x019D,0x0272,0x019D},  {0x0273,0x0273,0x0273}, /* 0272 */
  {0x0274,0x0274,0x0274},  {0x019F,0x0275,0x019F}, /* 0274 */
  {0x0276,0x0276,0x0276},  {0x0277,0x0277,0x0277}, /* 0276 */
  {0x0278,0x0278,0x0278},  {0x0279,0x0279,0x0279}, /* 0278 */
  {0x027A,0x027A,0x027A},  {0x027B,0x027B,0x027B}, /* 027A */
  {0x027C,0x027C,0x027C},  {0x2C64,0x027D,0x2C64}, /* 027C */
  {0x027E,0x027E,0x027E},  {0x027F,0x027F,0x027F}, /* 027E */
  {0x01A6,0x0280,0x01A6},  {0x0281,0x0281,0x0281}, /* 0280 */
  {0x0282,0x0282,0x0282},  {0x01A9,0x0283,0x01A9}, /* 0282 */
  {0x0284,0x0284,0x0284},  {0x0285,0x0285,0x0285}, /* 0284 */
  {0x0286,0x0286,0x0286},  {0x0287,0x0287,0x0287}, /* 0286 */
  {0x01AE,0x0288,0x01AE},  {0x0244,0x0289,0x0244}, /* 0288 */
  {0x01B1,0x028A,0x01B1},  {0x01B2,0x028B,0x01B2}, /* 028A */
  {0x0245,0x028C,0x0245},  {0x028D,0x028D,0x028D}, /* 028C */
  {0x028E,0x028E,0x028E},  {0x028F,0x028F,0x028F}, /* 028E */
  {0x0290,0x0290,0x0290},  {0x0291,0x0291,0x0291}, /* 0290 */
  {0x01B7,0x0292,0x01B7},  {0x0293,0x0293,0x0293}, /* 0292 */
  {0x0294,0x0294,0x0294},  {0x0295,0x0295,0x0295}, /* 0294 */
  {0x0296,0x0296,0x0296},  {0x0297,0x0297,0x0297}, /* 0296 */
  {0x0298,0x0298,0x0298},  {0x0299,0x0299,0x0299}, /* 0298 */
  {0x029A,0x029A,0x029A},  {0x029B,0x029B,0x029B}, /* 029A */
  {0x029C,0x029C,0x029C},  {0x029D,0x029D,0x029D}, /* 029C */
  {0x029E,0x029E,0x029E},  {0x029F,0x029F,0x029F}, /* 029E */
  {0x02A0,0x02A0,0x02A0},  {0x02A1,0x02A1,0x02A1}, /* 02A0 */
  {0x02A2,0x02A2,0x02A2},  {0x02A3,0x02A3,0x02A3}, /* 02A2 */
  {0x02A4,0x02A4,0x02A4},  {0x02A5,0x02A5,0x02A5}, /* 02A4 */
  {0x02A6,0x02A6,0x02A6},  {0x02A7,0x02A7,0x02A7}, /* 02A6 */
  {0x02A8,0x02A8,0x02A8},  {0x02A9,0x02A9,0x02A9}, /* 02A8 */
  {0x02AA,0x02AA,0x02AA},  {0x02AB,0x02AB,0x02AB}, /* 02AA */
  {0x02AC,0x02AC,0x02AC},  {0x02AD,0x02AD,0x02AD}, /* 02AC */
  {0x02AE,0x02AE,0x02AE},  {0x02AF,0x02AF,0x02AF}, /* 02AE */
  {0x02B0,0x02B0,0x02B0},  {0x02B1,0x02B1,0x02B1}, /* 02B0 */
  {0x02B2,0x02B2,0x02B2},  {0x02B3,0x02B3,0x02B3}, /* 02B2 */
  {0x02B4,0x02B4,0x02B4},  {0x02B5,0x02B5,0x02B5}, /* 02B4 */
  {0x02B6,0x02B6,0x02B6},  {0x02B7,0x02B7,0x02B7}, /* 02B6 */
  {0x02B8,0x02B8,0x02B8},  {0x02B9,0x02B9,0x02B9}, /* 02B8 */
  {0x02BA,0x02BA,0x02BA},  {0x02BB,0x02BB,0x02BB}, /* 02BA */
  {0x02BC,0x02BC,0x02BC},  {0x02BD,0x02BD,0x02BD}, /* 02BC */
  {0x02BE,0x02BE,0x02BE},  {0x02BF,0x02BF,0x02BF}, /* 02BE */
  {0x02C0,0x02C0,0x02C0},  {0x02C1,0x02C1,0x02C1}, /* 02C0 */
  {0x02C2,0x02C2,0x02C2},  {0x02C3,0x02C3,0x02C3}, /* 02C2 */
  {0x02C4,0x02C4,0x02C4},  {0x02C5,0x02C5,0x02C5}, /* 02C4 */
  {0x02C6,0x02C6,0x02C6},  {0x02C7,0x02C7,0x02C7}, /* 02C6 */
  {0x02C8,0x02C8,0x02C8},  {0x02C9,0x02C9,0x02C9}, /* 02C8 */
  {0x02CA,0x02CA,0x02CA},  {0x02CB,0x02CB,0x02CB}, /* 02CA */
  {0x02CC,0x02CC,0x02CC},  {0x02CD,0x02CD,0x02CD}, /* 02CC */
  {0x02CE,0x02CE,0x02CE},  {0x02CF,0x02CF,0x02CF}, /* 02CE */
  {0x02D0,0x02D0,0x02D0},  {0x02D1,0x02D1,0x02D1}, /* 02D0 */
  {0x02D2,0x02D2,0x02D2},  {0x02D3,0x02D3,0x02D3}, /* 02D2 */
  {0x02D4,0x02D4,0x02D4},  {0x02D5,0x02D5,0x02D5}, /* 02D4 */
  {0x02D6,0x02D6,0x02D6},  {0x02D7,0x02D7,0x02D7}, /* 02D6 */
  {0x02D8,0x02D8,0x02D8},  {0x02D9,0x02D9,0x02D9}, /* 02D8 */
  {0x02DA,0x02DA,0x02DA},  {0x02DB,0x02DB,0x02DB}, /* 02DA */
  {0x02DC,0x02DC,0x02DC},  {0x02DD,0x02DD,0x02DD}, /* 02DC */
  {0x02DE,0x02DE,0x02DE},  {0x02DF,0x02DF,0x02DF}, /* 02DE */
  {0x02E0,0x02E0,0x02E0},  {0x02E1,0x02E1,0x02E1}, /* 02E0 */
  {0x02E2,0x02E2,0x02E2},  {0x02E3,0x02E3,0x02E3}, /* 02E2 */
  {0x02E4,0x02E4,0x02E4},  {0x02E5,0x02E5,0x02E5}, /* 02E4 */
  {0x02E6,0x02E6,0x02E6},  {0x02E7,0x02E7,0x02E7}, /* 02E6 */
  {0x02E8,0x02E8,0x02E8},  {0x02E9,0x02E9,0x02E9}, /* 02E8 */
  {0x02EA,0x02EA,0x02EA},  {0x02EB,0x02EB,0x02EB}, /* 02EA */
  {0x02EC,0x02EC,0x02EC},  {0x02ED,0x02ED,0x02ED}, /* 02EC */
  {0x02EE,0x02EE,0x02EE},  {0x02EF,0x02EF,0x02EF}, /* 02EE */
  {0x02F0,0x02F0,0x02F0},  {0x02F1,0x02F1,0x02F1}, /* 02F0 */
  {0x02F2,0x02F2,0x02F2},  {0x02F3,0x02F3,0x02F3}, /* 02F2 */
  {0x02F4,0x02F4,0x02F4},  {0x02F5,0x02F5,0x02F5}, /* 02F4 */
  {0x02F6,0x02F6,0x02F6},  {0x02F7,0x02F7,0x02F7}, /* 02F6 */
  {0x02F8,0x02F8,0x02F8},  {0x02F9,0x02F9,0x02F9}, /* 02F8 */
  {0x02FA,0x02FA,0x02FA},  {0x02FB,0x02FB,0x02FB}, /* 02FA */
  {0x02FC,0x02FC,0x02FC},  {0x02FD,0x02FD,0x02FD}, /* 02FC */
  {0x02FE,0x02FE,0x02FE},  {0x02FF,0x02FF,0x02FF} /* 02FE */
};

static MY_UNICASE_CHARACTER u520p03[]={
  {0x0300,0x0300,0x0300},  {0x0301,0x0301,0x0301}, /* 0300 */
  {0x0302,0x0302,0x0302},  {0x0303,0x0303,0x0303}, /* 0302 */
  {0x0304,0x0304,0x0304},  {0x0305,0x0305,0x0305}, /* 0304 */
  {0x0306,0x0306,0x0306},  {0x0307,0x0307,0x0307}, /* 0306 */
  {0x0308,0x0308,0x0308},  {0x0309,0x0309,0x0309}, /* 0308 */
  {0x030A,0x030A,0x030A},  {0x030B,0x030B,0x030B}, /* 030A */
  {0x030C,0x030C,0x030C},  {0x030D,0x030D,0x030D}, /* 030C */
  {0x030E,0x030E,0x030E},  {0x030F,0x030F,0x030F}, /* 030E */
  {0x0310,0x0310,0x0310},  {0x0311,0x0311,0x0311}, /* 0310 */
  {0x0312,0x0312,0x0312},  {0x0313,0x0313,0x0313}, /* 0312 */
  {0x0314,0x0314,0x0314},  {0x0315,0x0315,0x0315}, /* 0314 */
  {0x0316,0x0316,0x0316},  {0x0317,0x0317,0x0317}, /* 0316 */
  {0x0318,0x0318,0x0318},  {0x0319,0x0319,0x0319}, /* 0318 */
  {0x031A,0x031A,0x031A},  {0x031B,0x031B,0x031B}, /* 031A */
  {0x031C,0x031C,0x031C},  {0x031D,0x031D,0x031D}, /* 031C */
  {0x031E,0x031E,0x031E},  {0x031F,0x031F,0x031F}, /* 031E */
  {0x0320,0x0320,0x0320},  {0x0321,0x0321,0x0321}, /* 0320 */
  {0x0322,0x0322,0x0322},  {0x0323,0x0323,0x0323}, /* 0322 */
  {0x0324,0x0324,0x0324},  {0x0325,0x0325,0x0325}, /* 0324 */
  {0x0326,0x0326,0x0326},  {0x0327,0x0327,0x0327}, /* 0326 */
  {0x0328,0x0328,0x0328},  {0x0329,0x0329,0x0329}, /* 0328 */
  {0x032A,0x032A,0x032A},  {0x032B,0x032B,0x032B}, /* 032A */
  {0x032C,0x032C,0x032C},  {0x032D,0x032D,0x032D}, /* 032C */
  {0x032E,0x032E,0x032E},  {0x032F,0x032F,0x032F}, /* 032E */
  {0x0330,0x0330,0x0330},  {0x0331,0x0331,0x0331}, /* 0330 */
  {0x0332,0x0332,0x0332},  {0x0333,0x0333,0x0333}, /* 0332 */
  {0x0334,0x0334,0x0334},  {0x0335,0x0335,0x0335}, /* 0334 */
  {0x0336,0x0336,0x0336},  {0x0337,0x0337,0x0337}, /* 0336 */
  {0x0338,0x0338,0x0338},  {0x0339,0x0339,0x0339}, /* 0338 */
  {0x033A,0x033A,0x033A},  {0x033B,0x033B,0x033B}, /* 033A */
  {0x033C,0x033C,0x033C},  {0x033D,0x033D,0x033D}, /* 033C */
  {0x033E,0x033E,0x033E},  {0x033F,0x033F,0x033F}, /* 033E */
  {0x0340,0x0340,0x0340},  {0x0341,0x0341,0x0341}, /* 0340 */
  {0x0342,0x0342,0x0342},  {0x0343,0x0343,0x0343}, /* 0342 */
  {0x0344,0x0344,0x0344},  {0x0399,0x0345,0x0399}, /* 0344 */
  {0x0346,0x0346,0x0346},  {0x0347,0x0347,0x0347}, /* 0346 */
  {0x0348,0x0348,0x0348},  {0x0349,0x0349,0x0349}, /* 0348 */
  {0x034A,0x034A,0x034A},  {0x034B,0x034B,0x034B}, /* 034A */
  {0x034C,0x034C,0x034C},  {0x034D,0x034D,0x034D}, /* 034C */
  {0x034E,0x034E,0x034E},  {0x034F,0x034F,0x034F}, /* 034E */
  {0x0350,0x0350,0x0350},  {0x0351,0x0351,0x0351}, /* 0350 */
  {0x0352,0x0352,0x0352},  {0x0353,0x0353,0x0353}, /* 0352 */
  {0x0354,0x0354,0x0354},  {0x0355,0x0355,0x0355}, /* 0354 */
  {0x0356,0x0356,0x0356},  {0x0357,0x0357,0x0357}, /* 0356 */
  {0x0358,0x0358,0x0358},  {0x0359,0x0359,0x0359}, /* 0358 */
  {0x035A,0x035A,0x035A},  {0x035B,0x035B,0x035B}, /* 035A */
  {0x035C,0x035C,0x035C},  {0x035D,0x035D,0x035D}, /* 035C */
  {0x035E,0x035E,0x035E},  {0x035F,0x035F,0x035F}, /* 035E */
  {0x0360,0x0360,0x0360},  {0x0361,0x0361,0x0361}, /* 0360 */
  {0x0362,0x0362,0x0362},  {0x0363,0x0363,0x0363}, /* 0362 */
  {0x0364,0x0364,0x0364},  {0x0365,0x0365,0x0365}, /* 0364 */
  {0x0366,0x0366,0x0366},  {0x0367,0x0367,0x0367}, /* 0366 */
  {0x0368,0x0368,0x0368},  {0x0369,0x0369,0x0369}, /* 0368 */
  {0x036A,0x036A,0x036A},  {0x036B,0x036B,0x036B}, /* 036A */
  {0x036C,0x036C,0x036C},  {0x036D,0x036D,0x036D}, /* 036C */
  {0x036E,0x036E,0x036E},  {0x036F,0x036F,0x036F}, /* 036E */
  {0x0370,0x0371,0x0370},  {0x0370,0x0371,0x0370}, /* 0370 */
  {0x0372,0x0373,0x0372},  {0x0372,0x0373,0x0372}, /* 0372 */
  {0x0374,0x0374,0x0374},  {0x0375,0x0375,0x0375}, /* 0374 */
  {0x0376,0x0377,0x0376},  {0x0376,0x0377,0x0376}, /* 0376 */
  {0x0378,0x0378,0x0378},  {0x0379,0x0379,0x0379}, /* 0378 */
  {0x037A,0x037A,0x037A},  {0x03FD,0x037B,0x03FD}, /* 037A */
  {0x03FE,0x037C,0x03FE},  {0x03FF,0x037D,0x03FF}, /* 037C */
  {0x037E,0x037E,0x037E},  {0x037F,0x037F,0x037F}, /* 037E */
  {0x0380,0x0380,0x0380},  {0x0381,0x0381,0x0381}, /* 0380 */
  {0x0382,0x0382,0x0382},  {0x0383,0x0383,0x0383}, /* 0382 */
  {0x0384,0x0384,0x0384},  {0x0385,0x0385,0x0385}, /* 0384 */
  {0x0386,0x03AC,0x0391},  {0x0387,0x0387,0x0387}, /* 0386 */
  {0x0388,0x03AD,0x0395},  {0x0389,0x03AE,0x0397}, /* 0388 */
  {0x038A,0x03AF,0x0399},  {0x038B,0x038B,0x038B}, /* 038A */
  {0x038C,0x03CC,0x039F},  {0x038D,0x038D,0x038D}, /* 038C */
  {0x038E,0x03CD,0x03A5},  {0x038F,0x03CE,0x03A9}, /* 038E */
  {0x0390,0x0390,0x0399},  {0x0391,0x03B1,0x0391}, /* 0390 */
  {0x0392,0x03B2,0x0392},  {0x0393,0x03B3,0x0393}, /* 0392 */
  {0x0394,0x03B4,0x0394},  {0x0395,0x03B5,0x0395}, /* 0394 */
  {0x0396,0x03B6,0x0396},  {0x0397,0x03B7,0x0397}, /* 0396 */
  {0x0398,0x03B8,0x0398},  {0x0399,0x03B9,0x0399}, /* 0398 */
  {0x039A,0x03BA,0x039A},  {0x039B,0x03BB,0x039B}, /* 039A */
  {0x039C,0x03BC,0x039C},  {0x039D,0x03BD,0x039D}, /* 039C */
  {0x039E,0x03BE,0x039E},  {0x039F,0x03BF,0x039F}, /* 039E */
  {0x03A0,0x03C0,0x03A0},  {0x03A1,0x03C1,0x03A1}, /* 03A0 */
  {0x03A2,0x03A2,0x03A2},  {0x03A3,0x03C3,0x03A3}, /* 03A2 */
  {0x03A4,0x03C4,0x03A4},  {0x03A5,0x03C5,0x03A5}, /* 03A4 */
  {0x03A6,0x03C6,0x03A6},  {0x03A7,0x03C7,0x03A7}, /* 03A6 */
  {0x03A8,0x03C8,0x03A8},  {0x03A9,0x03C9,0x03A9}, /* 03A8 */
  {0x03AA,0x03CA,0x0399},  {0x03AB,0x03CB,0x03A5}, /* 03AA */
  {0x0386,0x03AC,0x0391},  {0x0388,0x03AD,0x0395}, /* 03AC */
  {0x0389,0x03AE,0x0397},  {0x038A,0x03AF,0x0399}, /* 03AE */
  {0x03B0,0x03B0,0x03A5},  {0x0391,0x03B1,0x0391}, /* 03B0 */
  {0x0392,0x03B2,0x0392},  {0x0393,0x03B3,0x0393}, /* 03B2 */
  {0x0394,0x03B4,0x0394},  {0x0395,0x03B5,0x0395}, /* 03B4 */
  {0x0396,0x03B6,0x0396},  {0x0397,0x03B7,0x0397}, /* 03B6 */
  {0x0398,0x03B8,0x0398},  {0x0399,0x03B9,0x0399}, /* 03B8 */
  {0x039A,0x03BA,0x039A},  {0x039B,0x03BB,0x039B}, /* 03BA */
  {0x039C,0x03BC,0x039C},  {0x039D,0x03BD,0x039D}, /* 03BC */
  {0x039E,0x03BE,0x039E},  {0x039F,0x03BF,0x039F}, /* 03BE */
  {0x03A0,0x03C0,0x03A0},  {0x03A1,0x03C1,0x03A1}, /* 03C0 */
  {0x03A3,0x03C2,0x03A3},  {0x03A3,0x03C3,0x03A3}, /* 03C2 */
  {0x03A4,0x03C4,0x03A4},  {0x03A5,0x03C5,0x03A5}, /* 03C4 */
  {0x03A6,0x03C6,0x03A6},  {0x03A7,0x03C7,0x03A7}, /* 03C6 */
  {0x03A8,0x03C8,0x03A8},  {0x03A9,0x03C9,0x03A9}, /* 03C8 */
  {0x03AA,0x03CA,0x0399},  {0x03AB,0x03CB,0x03A5}, /* 03CA */
  {0x038C,0x03CC,0x039F},  {0x038E,0x03CD,0x03A5}, /* 03CC */
  {0x038F,0x03CE,0x03A9},  {0x03CF,0x03D7,0x03CF}, /* 03CE */
  {0x0392,0x03D0,0x0392},  {0x0398,0x03D1,0x0398}, /* 03D0 */
  {0x03D2,0x03D2,0x03D2},  {0x03D3,0x03D3,0x03D2}, /* 03D2 */
  {0x03D4,0x03D4,0x03D2},  {0x03A6,0x03D5,0x03A6}, /* 03D4 */
  {0x03A0,0x03D6,0x03A0},  {0x03CF,0x03D7,0x03CF}, /* 03D6 */
  {0x03D8,0x03D9,0x03D8},  {0x03D8,0x03D9,0x03D8}, /* 03D8 */
  {0x03DA,0x03DB,0x03DA},  {0x03DA,0x03DB,0x03DA}, /* 03DA */
  {0x03DC,0x03DD,0x03DC},  {0x03DC,0x03DD,0x03DC}, /* 03DC */
  {0x03DE,0x03DF,0x03DE},  {0x03DE,0x03DF,0x03DE}, /* 03DE */
  {0x03E0,0x03E1,0x03E0},  {0x03E0,0x03E1,0x03E0}, /* 03E0 */
  {0x03E2,0x03E3,0x03E2},  {0x03E2,0x03E3,0x03E2}, /* 03E2 */
  {0x03E4,0x03E5,0x03E4},  {0x03E4,0x03E5,0x03E4}, /* 03E4 */
  {0x03E6,0x03E7,0x03E6},  {0x03E6,0x03E7,0x03E6}, /* 03E6 */
  {0x03E8,0x03E9,0x03E8},  {0x03E8,0x03E9,0x03E8}, /* 03E8 */
  {0x03EA,0x03EB,0x03EA},  {0x03EA,0x03EB,0x03EA}, /* 03EA */
  {0x03EC,0x03ED,0x03EC},  {0x03EC,0x03ED,0x03EC}, /* 03EC */
  {0x03EE,0x03EF,0x03EE},  {0x03EE,0x03EF,0x03EE}, /* 03EE */
  {0x039A,0x03F0,0x039A},  {0x03A1,0x03F1,0x03A1}, /* 03F0 */
  {0x03F9,0x03F2,0x03F9},  {0x03F3,0x03F3,0x03F3}, /* 03F2 */
  {0x03F4,0x03B8,0x03F4},  {0x0395,0x03F5,0x0395}, /* 03F4 */
  {0x03F6,0x03F6,0x03F6},  {0x03F7,0x03F8,0x03F7}, /* 03F6 */
  {0x03F7,0x03F8,0x03F7},  {0x03F9,0x03F2,0x03F9}, /* 03F8 */
  {0x03FA,0x03FB,0x03FA},  {0x03FA,0x03FB,0x03FA}, /* 03FA */
  {0x03FC,0x03FC,0x03FC},  {0x03FD,0x037B,0x03FD}, /* 03FC */
  {0x03FE,0x037C,0x03FE},  {0x03FF,0x037D,0x03FF} /* 03FE */
};

static MY_UNICASE_CHARACTER u520p04[]={
  {0x0400,0x0450,0x0415},  {0x0401,0x0451,0x0415}, /* 0400 */
  {0x0402,0x0452,0x0402},  {0x0403,0x0453,0x0413}, /* 0402 */
  {0x0404,0x0454,0x0404},  {0x0405,0x0455,0x0405}, /* 0404 */
  {0x0406,0x0456,0x0406},  {0x0407,0x0457,0x0406}, /* 0406 */
  {0x0408,0x0458,0x0408},  {0x0409,0x0459,0x0409}, /* 0408 */
  {0x040A,0x045A,0x040A},  {0x040B,0x045B,0x040B}, /* 040A */
  {0x040C,0x045C,0x041A},  {0x040D,0x045D,0x0418}, /* 040C */
  {0x040E,0x045E,0x0423},  {0x040F,0x045F,0x040F}, /* 040E */
  {0x0410,0x0430,0x0410},  {0x0411,0x0431,0x0411}, /* 0410 */
  {0x0412,0x0432,0x0412},  {0x0413,0x0433,0x0413}, /* 0412 */
  {0x0414,0x0434,0x0414},  {0x0415,0x0435,0x0415}, /* 0414 */
  {0x0416,0x0436,0x0416},  {0x0417,0x0437,0x0417}, /* 0416 */
  {0x0418,0x0438,0x0418},  {0x0419,0x0439,0x0419}, /* 0418 */
  {0x041A,0x043A,0x041A},  {0x041B,0x043B,0x041B}, /* 041A */
  {0x041C,0x043C,0x041C},  {0x041D,0x043D,0x041D}, /* 041C */
  {0x041E,0x043E,0x041E},  {0x041F,0x043F,0x041F}, /* 041E */
  {0x0420,0x0440,0x0420},  {0x0421,0x0441,0x0421}, /* 0420 */
  {0x0422,0x0442,0x0422},  {0x0423,0x0443,0x0423}, /* 0422 */
  {0x0424,0x0444,0x0424},  {0x0425,0x0445,0x0425}, /* 0424 */
  {0x0426,0x0446,0x0426},  {0x0427,0x0447,0x0427}, /* 0426 */
  {0x0428,0x0448,0x0428},  {0x0429,0x0449,0x0429}, /* 0428 */
  {0x042A,0x044A,0x042A},  {0x042B,0x044B,0x042B}, /* 042A */
  {0x042C,0x044C,0x042C},  {0x042D,0x044D,0x042D}, /* 042C */
  {0x042E,0x044E,0x042E},  {0x042F,0x044F,0x042F}, /* 042E */
  {0x0410,0x0430,0x0410},  {0x0411,0x0431,0x0411}, /* 0430 */
  {0x0412,0x0432,0x0412},  {0x0413,0x0433,0x0413}, /* 0432 */
  {0x0414,0x0434,0x0414},  {0x0415,0x0435,0x0415}, /* 0434 */
  {0x0416,0x0436,0x0416},  {0x0417,0x0437,0x0417}, /* 0436 */
  {0x0418,0x0438,0x0418},  {0x0419,0x0439,0x0419}, /* 0438 */
  {0x041A,0x043A,0x041A},  {0x041B,0x043B,0x041B}, /* 043A */
  {0x041C,0x043C,0x041C},  {0x041D,0x043D,0x041D}, /* 043C */
  {0x041E,0x043E,0x041E},  {0x041F,0x043F,0x041F}, /* 043E */
  {0x0420,0x0440,0x0420},  {0x0421,0x0441,0x0421}, /* 0440 */
  {0x0422,0x0442,0x0422},  {0x0423,0x0443,0x0423}, /* 0442 */
  {0x0424,0x0444,0x0424},  {0x0425,0x0445,0x0425}, /* 0444 */
  {0x0426,0x0446,0x0426},  {0x0427,0x0447,0x0427}, /* 0446 */
  {0x0428,0x0448,0x0428},  {0x0429,0x0449,0x0429}, /* 0448 */
  {0x042A,0x044A,0x042A},  {0x042B,0x044B,0x042B}, /* 044A */
  {0x042C,0x044C,0x042C},  {0x042D,0x044D,0x042D}, /* 044C */
  {0x042E,0x044E,0x042E},  {0x042F,0x044F,0x042F}, /* 044E */
  {0x0400,0x0450,0x0415},  {0x0401,0x0451,0x0415}, /* 0450 */
  {0x0402,0x0452,0x0402},  {0x0403,0x0453,0x0413}, /* 0452 */
  {0x0404,0x0454,0x0404},  {0x0405,0x0455,0x0405}, /* 0454 */
  {0x0406,0x0456,0x0406},  {0x0407,0x0457,0x0406}, /* 0456 */
  {0x0408,0x0458,0x0408},  {0x0409,0x0459,0x0409}, /* 0458 */
  {0x040A,0x045A,0x040A},  {0x040B,0x045B,0x040B}, /* 045A */
  {0x040C,0x045C,0x041A},  {0x040D,0x045D,0x0418}, /* 045C */
  {0x040E,0x045E,0x0423},  {0x040F,0x045F,0x040F}, /* 045E */
  {0x0460,0x0461,0x0460},  {0x0460,0x0461,0x0460}, /* 0460 */
  {0x0462,0x0463,0x0462},  {0x0462,0x0463,0x0462}, /* 0462 */
  {0x0464,0x0465,0x0464},  {0x0464,0x0465,0x0464}, /* 0464 */
  {0x0466,0x0467,0x0466},  {0x0466,0x0467,0x0466}, /* 0466 */
  {0x0468,0x0469,0x0468},  {0x0468,0x0469,0x0468}, /* 0468 */
  {0x046A,0x046B,0x046A},  {0x046A,0x046B,0x046A}, /* 046A */
  {0x046C,0x046D,0x046C},  {0x046C,0x046D,0x046C}, /* 046C */
  {0x046E,0x046F,0x046E},  {0x046E,0x046F,0x046E}, /* 046E */
  {0x0470,0x0471,0x0470},  {0x0470,0x0471,0x0470}, /* 0470 */
  {0x0472,0x0473,0x0472},  {0x0472,0x0473,0x0472}, /* 0472 */
  {0x0474,0x0475,0x0474},  {0x0474,0x0475,0x0474}, /* 0474 */
  {0x0476,0x0477,0x0474},  {0x0476,0x0477,0x0474}, /* 0476 */
  {0x0478,0x0479,0x0478},  {0x0478,0x0479,0x0478}, /* 0478 */
  {0x047A,0x047B,0x047A},  {0x047A,0x047B,0x047A}, /* 047A */
  {0x047C,0x047D,0x047C},  {0x047C,0x047D,0x047C}, /* 047C */
  {0x047E,0x047F,0x047E},  {0x047E,0x047F,0x047E}, /* 047E */
  {0x0480,0x0481,0x0480},  {0x0480,0x0481,0x0480}, /* 0480 */
  {0x0482,0x0482,0x0482},  {0x0483,0x0483,0x0483}, /* 0482 */
  {0x0484,0x0484,0x0484},  {0x0485,0x0485,0x0485}, /* 0484 */
  {0x0486,0x0486,0x0486},  {0x0487,0x0487,0x0487}, /* 0486 */
  {0x0488,0x0488,0x0488},  {0x0489,0x0489,0x0489}, /* 0488 */
  {0x048A,0x048B,0x048A},  {0x048A,0x048B,0x048A}, /* 048A */
  {0x048C,0x048D,0x048C},  {0x048C,0x048D,0x048C}, /* 048C */
  {0x048E,0x048F,0x048E},  {0x048E,0x048F,0x048E}, /* 048E */
  {0x0490,0x0491,0x0490},  {0x0490,0x0491,0x0490}, /* 0490 */
  {0x0492,0x0493,0x0492},  {0x0492,0x0493,0x0492}, /* 0492 */
  {0x0494,0x0495,0x0494},  {0x0494,0x0495,0x0494}, /* 0494 */
  {0x0496,0x0497,0x0496},  {0x0496,0x0497,0x0496}, /* 0496 */
  {0x0498,0x0499,0x0498},  {0x0498,0x0499,0x0498}, /* 0498 */
  {0x049A,0x049B,0x049A},  {0x049A,0x049B,0x049A}, /* 049A */
  {0x049C,0x049D,0x049C},  {0x049C,0x049D,0x049C}, /* 049C */
  {0x049E,0x049F,0x049E},  {0x049E,0x049F,0x049E}, /* 049E */
  {0x04A0,0x04A1,0x04A0},  {0x04A0,0x04A1,0x04A0}, /* 04A0 */
  {0x04A2,0x04A3,0x04A2},  {0x04A2,0x04A3,0x04A2}, /* 04A2 */
  {0x04A4,0x04A5,0x04A4},  {0x04A4,0x04A5,0x04A4}, /* 04A4 */
  {0x04A6,0x04A7,0x04A6},  {0x04A6,0x04A7,0x04A6}, /* 04A6 */
  {0x04A8,0x04A9,0x04A8},  {0x04A8,0x04A9,0x04A8}, /* 04A8 */
  {0x04AA,0x04AB,0x04AA},  {0x04AA,0x04AB,0x04AA}, /* 04AA */
  {0x04AC,0x04AD,0x04AC},  {0x04AC,0x04AD,0x04AC}, /* 04AC */
  {0x04AE,0x04AF,0x04AE},  {0x04AE,0x04AF,0x04AE}, /* 04AE */
  {0x04B0,0x04B1,0x04B0},  {0x04B0,0x04B1,0x04B0}, /* 04B0 */
  {0x04B2,0x04B3,0x04B2},  {0x04B2,0x04B3,0x04B2}, /* 04B2 */
  {0x04B4,0x04B5,0x04B4},  {0x04B4,0x04B5,0x04B4}, /* 04B4 */
  {0x04B6,0x04B7,0x04B6},  {0x04B6,0x04B7,0x04B6}, /* 04B6 */
  {0x04B8,0x04B9,0x04B8},  {0x04B8,0x04B9,0x04B8}, /* 04B8 */
  {0x04BA,0x04BB,0x04BA},  {0x04BA,0x04BB,0x04BA}, /* 04BA */
  {0x04BC,0x04BD,0x04BC},  {0x04BC,0x04BD,0x04BC}, /* 04BC */
  {0x04BE,0x04BF,0x04BE},  {0x04BE,0x04BF,0x04BE}, /* 04BE */
  {0x04C0,0x04CF,0x04C0},  {0x04C1,0x04C2,0x0416}, /* 04C0 */
  {0x04C1,0x04C2,0x0416},  {0x04C3,0x04C4,0x04C3}, /* 04C2 */
  {0x04C3,0x04C4,0x04C3},  {0x04C5,0x04C6,0x04C5}, /* 04C4 */
  {0x04C5,0x04C6,0x04C5},  {0x04C7,0x04C8,0x04C7}, /* 04C6 */
  {0x04C7,0x04C8,0x04C7},  {0x04C9,0x04CA,0x04C9}, /* 04C8 */
  {0x04C9,0x04CA,0x04C9},  {0x04CB,0x04CC,0x04CB}, /* 04CA */
  {0x04CB,0x04CC,0x04CB},  {0x04CD,0x04CE,0x04CD}, /* 04CC */
  {0x04CD,0x04CE,0x04CD},  {0x04C0,0x04CF,0x04C0}, /* 04CE */
  {0x04D0,0x04D1,0x0410},  {0x04D0,0x04D1,0x0410}, /* 04D0 */
  {0x04D2,0x04D3,0x0410},  {0x04D2,0x04D3,0x0410}, /* 04D2 */
  {0x04D4,0x04D5,0x04D4},  {0x04D4,0x04D5,0x04D4}, /* 04D4 */
  {0x04D6,0x04D7,0x0415},  {0x04D6,0x04D7,0x0415}, /* 04D6 */
  {0x04D8,0x04D9,0x04D8},  {0x04D8,0x04D9,0x04D8}, /* 04D8 */
  {0x04DA,0x04DB,0x04D8},  {0x04DA,0x04DB,0x04D8}, /* 04DA */
  {0x04DC,0x04DD,0x0416},  {0x04DC,0x04DD,0x0416}, /* 04DC */
  {0x04DE,0x04DF,0x0417},  {0x04DE,0x04DF,0x0417}, /* 04DE */
  {0x04E0,0x04E1,0x04E0},  {0x04E0,0x04E1,0x04E0}, /* 04E0 */
  {0x04E2,0x04E3,0x0418},  {0x04E2,0x04E3,0x0418}, /* 04E2 */
  {0x04E4,0x04E5,0x0418},  {0x04E4,0x04E5,0x0418}, /* 04E4 */
  {0x04E6,0x04E7,0x041E},  {0x04E6,0x04E7,0x041E}, /* 04E6 */
  {0x04E8,0x04E9,0x04E8},  {0x04E8,0x04E9,0x04E8}, /* 04E8 */
  {0x04EA,0x04EB,0x04E8},  {0x04EA,0x04EB,0x04E8}, /* 04EA */
  {0x04EC,0x04ED,0x042D},  {0x04EC,0x04ED,0x042D}, /* 04EC */
  {0x04EE,0x04EF,0x0423},  {0x04EE,0x04EF,0x0423}, /* 04EE */
  {0x04F0,0x04F1,0x0423},  {0x04F0,0x04F1,0x0423}, /* 04F0 */
  {0x04F2,0x04F3,0x0423},  {0x04F2,0x04F3,0x0423}, /* 04F2 */
  {0x04F4,0x04F5,0x0427},  {0x04F4,0x04F5,0x0427}, /* 04F4 */
  {0x04F6,0x04F7,0x04F6},  {0x04F6,0x04F7,0x04F6}, /* 04F6 */
  {0x04F8,0x04F9,0x042B},  {0x04F8,0x04F9,0x042B}, /* 04F8 */
  {0x04FA,0x04FB,0x04FA},  {0x04FA,0x04FB,0x04FA}, /* 04FA */
  {0x04FC,0x04FD,0x04FC},  {0x04FC,0x04FD,0x04FC}, /* 04FC */
  {0x04FE,0x04FF,0x04FE},  {0x04FE,0x04FF,0x04FE} /* 04FE */
};

static MY_UNICASE_CHARACTER u520p05[]={
  {0x0500,0x0501,0x0500},  {0x0500,0x0501,0x0500}, /* 0500 */
  {0x0502,0x0503,0x0502},  {0x0502,0x0503,0x0502}, /* 0502 */
  {0x0504,0x0505,0x0504},  {0x0504,0x0505,0x0504}, /* 0504 */
  {0x0506,0x0507,0x0506},  {0x0506,0x0507,0x0506}, /* 0506 */
  {0x0508,0x0509,0x0508},  {0x0508,0x0509,0x0508}, /* 0508 */
  {0x050A,0x050B,0x050A},  {0x050A,0x050B,0x050A}, /* 050A */
  {0x050C,0x050D,0x050C},  {0x050C,0x050D,0x050C}, /* 050C */
  {0x050E,0x050F,0x050E},  {0x050E,0x050F,0x050E}, /* 050E */
  {0x0510,0x0511,0x0510},  {0x0510,0x0511,0x0510}, /* 0510 */
  {0x0512,0x0513,0x0512},  {0x0512,0x0513,0x0512}, /* 0512 */
  {0x0514,0x0515,0x0514},  {0x0514,0x0515,0x0514}, /* 0514 */
  {0x0516,0x0517,0x0516},  {0x0516,0x0517,0x0516}, /* 0516 */
  {0x0518,0x0519,0x0518},  {0x0518,0x0519,0x0518}, /* 0518 */
  {0x051A,0x051B,0x051A},  {0x051A,0x051B,0x051A}, /* 051A */
  {0x051C,0x051D,0x051C},  {0x051C,0x051D,0x051C}, /* 051C */
  {0x051E,0x051F,0x051E},  {0x051E,0x051F,0x051E}, /* 051E */
  {0x0520,0x0521,0x0520},  {0x0520,0x0521,0x0520}, /* 0520 */
  {0x0522,0x0523,0x0522},  {0x0522,0x0523,0x0522}, /* 0522 */
  {0x0524,0x0525,0x0524},  {0x0524,0x0525,0x0524}, /* 0524 */
  {0x0526,0x0526,0x0526},  {0x0527,0x0527,0x0527}, /* 0526 */
  {0x0528,0x0528,0x0528},  {0x0529,0x0529,0x0529}, /* 0528 */
  {0x052A,0x052A,0x052A},  {0x052B,0x052B,0x052B}, /* 052A */
  {0x052C,0x052C,0x052C},  {0x052D,0x052D,0x052D}, /* 052C */
  {0x052E,0x052E,0x052E},  {0x052F,0x052F,0x052F}, /* 052E */
  {0x0530,0x0530,0x0530},  {0x0531,0x0561,0x0531}, /* 0530 */
  {0x0532,0x0562,0x0532},  {0x0533,0x0563,0x0533}, /* 0532 */
  {0x0534,0x0564,0x0534},  {0x0535,0x0565,0x0535}, /* 0534 */
  {0x0536,0x0566,0x0536},  {0x0537,0x0567,0x0537}, /* 0536 */
  {0x0538,0x0568,0x0538},  {0x0539,0x0569,0x0539}, /* 0538 */
  {0x053A,0x056A,0x053A},  {0x053B,0x056B,0x053B}, /* 053A */
  {0x053C,0x056C,0x053C},  {0x053D,0x056D,0x053D}, /* 053C */
  {0x053E,0x056E,0x053E},  {0x053F,0x056F,0x053F}, /* 053E */
  {0x0540,0x0570,0x0540},  {0x0541,0x0571,0x0541}, /* 0540 */
  {0x0542,0x0572,0x0542},  {0x0543,0x0573,0x0543}, /* 0542 */
  {0x0544,0x0574,0x0544},  {0x0545,0x0575,0x0545}, /* 0544 */
  {0x0546,0x0576,0x0546},  {0x0547,0x0577,0x0547}, /* 0546 */
  {0x0548,0x0578,0x0548},  {0x0549,0x0579,0x0549}, /* 0548 */
  {0x054A,0x057A,0x054A},  {0x054B,0x057B,0x054B}, /* 054A */
  {0x054C,0x057C,0x054C},  {0x054D,0x057D,0x054D}, /* 054C */
  {0x054E,0x057E,0x054E},  {0x054F,0x057F,0x054F}, /* 054E */
  {0x0550,0x0580,0x0550},  {0x0551,0x0581,0x0551}, /* 0550 */
  {0x0552,0x0582,0x0552},  {0x0553,0x0583,0x0553}, /* 0552 */
  {0x0554,0x0584,0x0554},  {0x0555,0x0585,0x0555}, /* 0554 */
  {0x0556,0x0586,0x0556},  {0x0557,0x0557,0x0557}, /* 0556 */
  {0x0558,0x0558,0x0558},  {0x0559,0x0559,0x0559}, /* 0558 */
  {0x055A,0x055A,0x055A},  {0x055B,0x055B,0x055B}, /* 055A */
  {0x055C,0x055C,0x055C},  {0x055D,0x055D,0x055D}, /* 055C */
  {0x055E,0x055E,0x055E},  {0x055F,0x055F,0x055F}, /* 055E */
  {0x0560,0x0560,0x0560},  {0x0531,0x0561,0x0531}, /* 0560 */
  {0x0532,0x0562,0x0532},  {0x0533,0x0563,0x0533}, /* 0562 */
  {0x0534,0x0564,0x0534},  {0x0535,0x0565,0x0535}, /* 0564 */
  {0x0536,0x0566,0x0536},  {0x0537,0x0567,0x0537}, /* 0566 */
  {0x0538,0x0568,0x0538},  {0x0539,0x0569,0x0539}, /* 0568 */
  {0x053A,0x056A,0x053A},  {0x053B,0x056B,0x053B}, /* 056A */
  {0x053C,0x056C,0x053C},  {0x053D,0x056D,0x053D}, /* 056C */
  {0x053E,0x056E,0x053E},  {0x053F,0x056F,0x053F}, /* 056E */
  {0x0540,0x0570,0x0540},  {0x0541,0x0571,0x0541}, /* 0570 */
  {0x0542,0x0572,0x0542},  {0x0543,0x0573,0x0543}, /* 0572 */
  {0x0544,0x0574,0x0544},  {0x0545,0x0575,0x0545}, /* 0574 */
  {0x0546,0x0576,0x0546},  {0x0547,0x0577,0x0547}, /* 0576 */
  {0x0548,0x0578,0x0548},  {0x0549,0x0579,0x0549}, /* 0578 */
  {0x054A,0x057A,0x054A},  {0x054B,0x057B,0x054B}, /* 057A */
  {0x054C,0x057C,0x054C},  {0x054D,0x057D,0x054D}, /* 057C */
  {0x054E,0x057E,0x054E},  {0x054F,0x057F,0x054F}, /* 057E */
  {0x0550,0x0580,0x0550},  {0x0551,0x0581,0x0551}, /* 0580 */
  {0x0552,0x0582,0x0552},  {0x0553,0x0583,0x0553}, /* 0582 */
  {0x0554,0x0584,0x0554},  {0x0555,0x0585,0x0555}, /* 0584 */
  {0x0556,0x0586,0x0556},  {0x0587,0x0587,0x0587}, /* 0586 */
  {0x0588,0x0588,0x0588},  {0x0589,0x0589,0x0589}, /* 0588 */
  {0x058A,0x058A,0x058A},  {0x058B,0x058B,0x058B}, /* 058A */
  {0x058C,0x058C,0x058C},  {0x058D,0x058D,0x058D}, /* 058C */
  {0x058E,0x058E,0x058E},  {0x058F,0x058F,0x058F}, /* 058E */
  {0x0590,0x0590,0x0590},  {0x0591,0x0591,0x0591}, /* 0590 */
  {0x0592,0x0592,0x0592},  {0x0593,0x0593,0x0593}, /* 0592 */
  {0x0594,0x0594,0x0594},  {0x0595,0x0595,0x0595}, /* 0594 */
  {0x0596,0x0596,0x0596},  {0x0597,0x0597,0x0597}, /* 0596 */
  {0x0598,0x0598,0x0598},  {0x0599,0x0599,0x0599}, /* 0598 */
  {0x059A,0x059A,0x059A},  {0x059B,0x059B,0x059B}, /* 059A */
  {0x059C,0x059C,0x059C},  {0x059D,0x059D,0x059D}, /* 059C */
  {0x059E,0x059E,0x059E},  {0x059F,0x059F,0x059F}, /* 059E */
  {0x05A0,0x05A0,0x05A0},  {0x05A1,0x05A1,0x05A1}, /* 05A0 */
  {0x05A2,0x05A2,0x05A2},  {0x05A3,0x05A3,0x05A3}, /* 05A2 */
  {0x05A4,0x05A4,0x05A4},  {0x05A5,0x05A5,0x05A5}, /* 05A4 */
  {0x05A6,0x05A6,0x05A6},  {0x05A7,0x05A7,0x05A7}, /* 05A6 */
  {0x05A8,0x05A8,0x05A8},  {0x05A9,0x05A9,0x05A9}, /* 05A8 */
  {0x05AA,0x05AA,0x05AA},  {0x05AB,0x05AB,0x05AB}, /* 05AA */
  {0x05AC,0x05AC,0x05AC},  {0x05AD,0x05AD,0x05AD}, /* 05AC */
  {0x05AE,0x05AE,0x05AE},  {0x05AF,0x05AF,0x05AF}, /* 05AE */
  {0x05B0,0x05B0,0x05B0},  {0x05B1,0x05B1,0x05B1}, /* 05B0 */
  {0x05B2,0x05B2,0x05B2},  {0x05B3,0x05B3,0x05B3}, /* 05B2 */
  {0x05B4,0x05B4,0x05B4},  {0x05B5,0x05B5,0x05B5}, /* 05B4 */
  {0x05B6,0x05B6,0x05B6},  {0x05B7,0x05B7,0x05B7}, /* 05B6 */
  {0x05B8,0x05B8,0x05B8},  {0x05B9,0x05B9,0x05B9}, /* 05B8 */
  {0x05BA,0x05BA,0x05BA},  {0x05BB,0x05BB,0x05BB}, /* 05BA */
  {0x05BC,0x05BC,0x05BC},  {0x05BD,0x05BD,0x05BD}, /* 05BC */
  {0x05BE,0x05BE,0x05BE},  {0x05BF,0x05BF,0x05BF}, /* 05BE */
  {0x05C0,0x05C0,0x05C0},  {0x05C1,0x05C1,0x05C1}, /* 05C0 */
  {0x05C2,0x05C2,0x05C2},  {0x05C3,0x05C3,0x05C3}, /* 05C2 */
  {0x05C4,0x05C4,0x05C4},  {0x05C5,0x05C5,0x05C5}, /* 05C4 */
  {0x05C6,0x05C6,0x05C6},  {0x05C7,0x05C7,0x05C7}, /* 05C6 */
  {0x05C8,0x05C8,0x05C8},  {0x05C9,0x05C9,0x05C9}, /* 05C8 */
  {0x05CA,0x05CA,0x05CA},  {0x05CB,0x05CB,0x05CB}, /* 05CA */
  {0x05CC,0x05CC,0x05CC},  {0x05CD,0x05CD,0x05CD}, /* 05CC */
  {0x05CE,0x05CE,0x05CE},  {0x05CF,0x05CF,0x05CF}, /* 05CE */
  {0x05D0,0x05D0,0x05D0},  {0x05D1,0x05D1,0x05D1}, /* 05D0 */
  {0x05D2,0x05D2,0x05D2},  {0x05D3,0x05D3,0x05D3}, /* 05D2 */
  {0x05D4,0x05D4,0x05D4},  {0x05D5,0x05D5,0x05D5}, /* 05D4 */
  {0x05D6,0x05D6,0x05D6},  {0x05D7,0x05D7,0x05D7}, /* 05D6 */
  {0x05D8,0x05D8,0x05D8},  {0x05D9,0x05D9,0x05D9}, /* 05D8 */
  {0x05DA,0x05DA,0x05DA},  {0x05DB,0x05DB,0x05DB}, /* 05DA */
  {0x05DC,0x05DC,0x05DC},  {0x05DD,0x05DD,0x05DD}, /* 05DC */
  {0x05DE,0x05DE,0x05DE},  {0x05DF,0x05DF,0x05DF}, /* 05DE */
  {0x05E0,0x05E0,0x05E0},  {0x05E1,0x05E1,0x05E1}, /* 05E0 */
  {0x05E2,0x05E2,0x05E2},  {0x05E3,0x05E3,0x05E3}, /* 05E2 */
  {0x05E4,0x05E4,0x05E4},  {0x05E5,0x05E5,0x05E5}, /* 05E4 */
  {0x05E6,0x05E6,0x05E6},  {0x05E7,0x05E7,0x05E7}, /* 05E6 */
  {0x05E8,0x05E8,0x05E8},  {0x05E9,0x05E9,0x05E9}, /* 05E8 */
  {0x05EA,0x05EA,0x05EA},  {0x05EB,0x05EB,0x05EB}, /* 05EA */
  {0x05EC,0x05EC,0x05EC},  {0x05ED,0x05ED,0x05ED}, /* 05EC */
  {0x05EE,0x05EE,0x05EE},  {0x05EF,0x05EF,0x05EF}, /* 05EE */
  {0x05F0,0x05F0,0x05F0},  {0x05F1,0x05F1,0x05F1}, /* 05F0 */
  {0x05F2,0x05F2,0x05F2},  {0x05F3,0x05F3,0x05F3}, /* 05F2 */
  {0x05F4,0x05F4,0x05F4},  {0x05F5,0x05F5,0x05F5}, /* 05F4 */
  {0x05F6,0x05F6,0x05F6},  {0x05F7,0x05F7,0x05F7}, /* 05F6 */
  {0x05F8,0x05F8,0x05F8},  {0x05F9,0x05F9,0x05F9}, /* 05F8 */
  {0x05FA,0x05FA,0x05FA},  {0x05FB,0x05FB,0x05FB}, /* 05FA */
  {0x05FC,0x05FC,0x05FC},  {0x05FD,0x05FD,0x05FD}, /* 05FC */
  {0x05FE,0x05FE,0x05FE},  {0x05FF,0x05FF,0x05FF} /* 05FE */
};

static MY_UNICASE_CHARACTER u520p10[]={
  {0x1000,0x1000,0x1000},  {0x1001,0x1001,0x1001}, /* 1000 */
  {0x1002,0x1002,0x1002},  {0x1003,0x1003,0x1003}, /* 1002 */
  {0x1004,0x1004,0x1004},  {0x1005,0x1005,0x1005}, /* 1004 */
  {0x1006,0x1006,0x1006},  {0x1007,0x1007,0x1007}, /* 1006 */
  {0x1008,0x1008,0x1008},  {0x1009,0x1009,0x1009}, /* 1008 */
  {0x100A,0x100A,0x100A},  {0x100B,0x100B,0x100B}, /* 100A */
  {0x100C,0x100C,0x100C},  {0x100D,0x100D,0x100D}, /* 100C */
  {0x100E,0x100E,0x100E},  {0x100F,0x100F,0x100F}, /* 100E */
  {0x1010,0x1010,0x1010},  {0x1011,0x1011,0x1011}, /* 1010 */
  {0x1012,0x1012,0x1012},  {0x1013,0x1013,0x1013}, /* 1012 */
  {0x1014,0x1014,0x1014},  {0x1015,0x1015,0x1015}, /* 1014 */
  {0x1016,0x1016,0x1016},  {0x1017,0x1017,0x1017}, /* 1016 */
  {0x1018,0x1018,0x1018},  {0x1019,0x1019,0x1019}, /* 1018 */
  {0x101A,0x101A,0x101A},  {0x101B,0x101B,0x101B}, /* 101A */
  {0x101C,0x101C,0x101C},  {0x101D,0x101D,0x101D}, /* 101C */
  {0x101E,0x101E,0x101E},  {0x101F,0x101F,0x101F}, /* 101E */
  {0x1020,0x1020,0x1020},  {0x1021,0x1021,0x1021}, /* 1020 */
  {0x1022,0x1022,0x1022},  {0x1023,0x1023,0x1023}, /* 1022 */
  {0x1024,0x1024,0x1024},  {0x1025,0x1025,0x1025}, /* 1024 */
  {0x1026,0x1026,0x1026},  {0x1027,0x1027,0x1027}, /* 1026 */
  {0x1028,0x1028,0x1028},  {0x1029,0x1029,0x1029}, /* 1028 */
  {0x102A,0x102A,0x102A},  {0x102B,0x102B,0x102B}, /* 102A */
  {0x102C,0x102C,0x102C},  {0x102D,0x102D,0x102D}, /* 102C */
  {0x102E,0x102E,0x102E},  {0x102F,0x102F,0x102F}, /* 102E */
  {0x1030,0x1030,0x1030},  {0x1031,0x1031,0x1031}, /* 1030 */
  {0x1032,0x1032,0x1032},  {0x1033,0x1033,0x1033}, /* 1032 */
  {0x1034,0x1034,0x1034},  {0x1035,0x1035,0x1035}, /* 1034 */
  {0x1036,0x1036,0x1036},  {0x1037,0x1037,0x1037}, /* 1036 */
  {0x1038,0x1038,0x1038},  {0x1039,0x1039,0x1039}, /* 1038 */
  {0x103A,0x103A,0x103A},  {0x103B,0x103B,0x103B}, /* 103A */
  {0x103C,0x103C,0x103C},  {0x103D,0x103D,0x103D}, /* 103C */
  {0x103E,0x103E,0x103E},  {0x103F,0x103F,0x103F}, /* 103E */
  {0x1040,0x1040,0x1040},  {0x1041,0x1041,0x1041}, /* 1040 */
  {0x1042,0x1042,0x1042},  {0x1043,0x1043,0x1043}, /* 1042 */
  {0x1044,0x1044,0x1044},  {0x1045,0x1045,0x1045}, /* 1044 */
  {0x1046,0x1046,0x1046},  {0x1047,0x1047,0x1047}, /* 1046 */
  {0x1048,0x1048,0x1048},  {0x1049,0x1049,0x1049}, /* 1048 */
  {0x104A,0x104A,0x104A},  {0x104B,0x104B,0x104B}, /* 104A */
  {0x104C,0x104C,0x104C},  {0x104D,0x104D,0x104D}, /* 104C */
  {0x104E,0x104E,0x104E},  {0x104F,0x104F,0x104F}, /* 104E */
  {0x1050,0x1050,0x1050},  {0x1051,0x1051,0x1051}, /* 1050 */
  {0x1052,0x1052,0x1052},  {0x1053,0x1053,0x1053}, /* 1052 */
  {0x1054,0x1054,0x1054},  {0x1055,0x1055,0x1055}, /* 1054 */
  {0x1056,0x1056,0x1056},  {0x1057,0x1057,0x1057}, /* 1056 */
  {0x1058,0x1058,0x1058},  {0x1059,0x1059,0x1059}, /* 1058 */
  {0x105A,0x105A,0x105A},  {0x105B,0x105B,0x105B}, /* 105A */
  {0x105C,0x105C,0x105C},  {0x105D,0x105D,0x105D}, /* 105C */
  {0x105E,0x105E,0x105E},  {0x105F,0x105F,0x105F}, /* 105E */
  {0x1060,0x1060,0x1060},  {0x1061,0x1061,0x1061}, /* 1060 */
  {0x1062,0x1062,0x1062},  {0x1063,0x1063,0x1063}, /* 1062 */
  {0x1064,0x1064,0x1064},  {0x1065,0x1065,0x1065}, /* 1064 */
  {0x1066,0x1066,0x1066},  {0x1067,0x1067,0x1067}, /* 1066 */
  {0x1068,0x1068,0x1068},  {0x1069,0x1069,0x1069}, /* 1068 */
  {0x106A,0x106A,0x106A},  {0x106B,0x106B,0x106B}, /* 106A */
  {0x106C,0x106C,0x106C},  {0x106D,0x106D,0x106D}, /* 106C */
  {0x106E,0x106E,0x106E},  {0x106F,0x106F,0x106F}, /* 106E */
  {0x1070,0x1070,0x1070},  {0x1071,0x1071,0x1071}, /* 1070 */
  {0x1072,0x1072,0x1072},  {0x1073,0x1073,0x1073}, /* 1072 */
  {0x1074,0x1074,0x1074},  {0x1075,0x1075,0x1075}, /* 1074 */
  {0x1076,0x1076,0x1076},  {0x1077,0x1077,0x1077}, /* 1076 */
  {0x1078,0x1078,0x1078},  {0x1079,0x1079,0x1079}, /* 1078 */
  {0x107A,0x107A,0x107A},  {0x107B,0x107B,0x107B}, /* 107A */
  {0x107C,0x107C,0x107C},  {0x107D,0x107D,0x107D}, /* 107C */
  {0x107E,0x107E,0x107E},  {0x107F,0x107F,0x107F}, /* 107E */
  {0x1080,0x1080,0x1080},  {0x1081,0x1081,0x1081}, /* 1080 */
  {0x1082,0x1082,0x1082},  {0x1083,0x1083,0x1083}, /* 1082 */
  {0x1084,0x1084,0x1084},  {0x1085,0x1085,0x1085}, /* 1084 */
  {0x1086,0x1086,0x1086},  {0x1087,0x1087,0x1087}, /* 1086 */
  {0x1088,0x1088,0x1088},  {0x1089,0x1089,0x1089}, /* 1088 */
  {0x108A,0x108A,0x108A},  {0x108B,0x108B,0x108B}, /* 108A */
  {0x108C,0x108C,0x108C},  {0x108D,0x108D,0x108D}, /* 108C */
  {0x108E,0x108E,0x108E},  {0x108F,0x108F,0x108F}, /* 108E */
  {0x1090,0x1090,0x1090},  {0x1091,0x1091,0x1091}, /* 1090 */
  {0x1092,0x1092,0x1092},  {0x1093,0x1093,0x1093}, /* 1092 */
  {0x1094,0x1094,0x1094},  {0x1095,0x1095,0x1095}, /* 1094 */
  {0x1096,0x1096,0x1096},  {0x1097,0x1097,0x1097}, /* 1096 */
  {0x1098,0x1098,0x1098},  {0x1099,0x1099,0x1099}, /* 1098 */
  {0x109A,0x109A,0x109A},  {0x109B,0x109B,0x109B}, /* 109A */
  {0x109C,0x109C,0x109C},  {0x109D,0x109D,0x109D}, /* 109C */
  {0x109E,0x109E,0x109E},  {0x109F,0x109F,0x109F}, /* 109E */
  {0x10A0,0x2D00,0x10A0},  {0x10A1,0x2D01,0x10A1}, /* 10A0 */
  {0x10A2,0x2D02,0x10A2},  {0x10A3,0x2D03,0x10A3}, /* 10A2 */
  {0x10A4,0x2D04,0x10A4},  {0x10A5,0x2D05,0x10A5}, /* 10A4 */
  {0x10A6,0x2D06,0x10A6},  {0x10A7,0x2D07,0x10A7}, /* 10A6 */
  {0x10A8,0x2D08,0x10A8},  {0x10A9,0x2D09,0x10A9}, /* 10A8 */
  {0x10AA,0x2D0A,0x10AA},  {0x10AB,0x2D0B,0x10AB}, /* 10AA */
  {0x10AC,0x2D0C,0x10AC},  {0x10AD,0x2D0D,0x10AD}, /* 10AC */
  {0x10AE,0x2D0E,0x10AE},  {0x10AF,0x2D0F,0x10AF}, /* 10AE */
  {0x10B0,0x2D10,0x10B0},  {0x10B1,0x2D11,0x10B1}, /* 10B0 */
  {0x10B2,0x2D12,0x10B2},  {0x10B3,0x2D13,0x10B3}, /* 10B2 */
  {0x10B4,0x2D14,0x10B4},  {0x10B5,0x2D15,0x10B5}, /* 10B4 */
  {0x10B6,0x2D16,0x10B6},  {0x10B7,0x2D17,0x10B7}, /* 10B6 */
  {0x10B8,0x2D18,0x10B8},  {0x10B9,0x2D19,0x10B9}, /* 10B8 */
  {0x10BA,0x2D1A,0x10BA},  {0x10BB,0x2D1B,0x10BB}, /* 10BA */
  {0x10BC,0x2D1C,0x10BC},  {0x10BD,0x2D1D,0x10BD}, /* 10BC */
  {0x10BE,0x2D1E,0x10BE},  {0x10BF,0x2D1F,0x10BF}, /* 10BE */
  {0x10C0,0x2D20,0x10C0},  {0x10C1,0x2D21,0x10C1}, /* 10C0 */
  {0x10C2,0x2D22,0x10C2},  {0x10C3,0x2D23,0x10C3}, /* 10C2 */
  {0x10C4,0x2D24,0x10C4},  {0x10C5,0x2D25,0x10C5}, /* 10C4 */
  {0x10C6,0x10C6,0x10C6},  {0x10C7,0x10C7,0x10C7}, /* 10C6 */
  {0x10C8,0x10C8,0x10C8},  {0x10C9,0x10C9,0x10C9}, /* 10C8 */
  {0x10CA,0x10CA,0x10CA},  {0x10CB,0x10CB,0x10CB}, /* 10CA */
  {0x10CC,0x10CC,0x10CC},  {0x10CD,0x10CD,0x10CD}, /* 10CC */
  {0x10CE,0x10CE,0x10CE},  {0x10CF,0x10CF,0x10CF}, /* 10CE */
  {0x10D0,0x10D0,0x10D0},  {0x10D1,0x10D1,0x10D1}, /* 10D0 */
  {0x10D2,0x10D2,0x10D2},  {0x10D3,0x10D3,0x10D3}, /* 10D2 */
  {0x10D4,0x10D4,0x10D4},  {0x10D5,0x10D5,0x10D5}, /* 10D4 */
  {0x10D6,0x10D6,0x10D6},  {0x10D7,0x10D7,0x10D7}, /* 10D6 */
  {0x10D8,0x10D8,0x10D8},  {0x10D9,0x10D9,0x10D9}, /* 10D8 */
  {0x10DA,0x10DA,0x10DA},  {0x10DB,0x10DB,0x10DB}, /* 10DA */
  {0x10DC,0x10DC,0x10DC},  {0x10DD,0x10DD,0x10DD}, /* 10DC */
  {0x10DE,0x10DE,0x10DE},  {0x10DF,0x10DF,0x10DF}, /* 10DE */
  {0x10E0,0x10E0,0x10E0},  {0x10E1,0x10E1,0x10E1}, /* 10E0 */
  {0x10E2,0x10E2,0x10E2},  {0x10E3,0x10E3,0x10E3}, /* 10E2 */
  {0x10E4,0x10E4,0x10E4},  {0x10E5,0x10E5,0x10E5}, /* 10E4 */
  {0x10E6,0x10E6,0x10E6},  {0x10E7,0x10E7,0x10E7}, /* 10E6 */
  {0x10E8,0x10E8,0x10E8},  {0x10E9,0x10E9,0x10E9}, /* 10E8 */
  {0x10EA,0x10EA,0x10EA},  {0x10EB,0x10EB,0x10EB}, /* 10EA */
  {0x10EC,0x10EC,0x10EC},  {0x10ED,0x10ED,0x10ED}, /* 10EC */
  {0x10EE,0x10EE,0x10EE},  {0x10EF,0x10EF,0x10EF}, /* 10EE */
  {0x10F0,0x10F0,0x10F0},  {0x10F1,0x10F1,0x10F1}, /* 10F0 */
  {0x10F2,0x10F2,0x10F2},  {0x10F3,0x10F3,0x10F3}, /* 10F2 */
  {0x10F4,0x10F4,0x10F4},  {0x10F5,0x10F5,0x10F5}, /* 10F4 */
  {0x10F6,0x10F6,0x10F6},  {0x10F7,0x10F7,0x10F7}, /* 10F6 */
  {0x10F8,0x10F8,0x10F8},  {0x10F9,0x10F9,0x10F9}, /* 10F8 */
  {0x10FA,0x10FA,0x10FA},  {0x10FB,0x10FB,0x10FB}, /* 10FA */
  {0x10FC,0x10FC,0x10FC},  {0x10FD,0x10FD,0x10FD}, /* 10FC */
  {0x10FE,0x10FE,0x10FE},  {0x10FF,0x10FF,0x10FF} /* 10FE */
};

static MY_UNICASE_CHARACTER u520p1D[]={
  {0x1D00,0x1D00,0x1D00},  {0x1D01,0x1D01,0x1D01}, /* 1D00 */
  {0x1D02,0x1D02,0x1D02},  {0x1D03,0x1D03,0x1D03}, /* 1D02 */
  {0x1D04,0x1D04,0x1D04},  {0x1D05,0x1D05,0x1D05}, /* 1D04 */
  {0x1D06,0x1D06,0x1D06},  {0x1D07,0x1D07,0x1D07}, /* 1D06 */
  {0x1D08,0x1D08,0x1D08},  {0x1D09,0x1D09,0x1D09}, /* 1D08 */
  {0x1D0A,0x1D0A,0x1D0A},  {0x1D0B,0x1D0B,0x1D0B}, /* 1D0A */
  {0x1D0C,0x1D0C,0x1D0C},  {0x1D0D,0x1D0D,0x1D0D}, /* 1D0C */
  {0x1D0E,0x1D0E,0x1D0E},  {0x1D0F,0x1D0F,0x1D0F}, /* 1D0E */
  {0x1D10,0x1D10,0x1D10},  {0x1D11,0x1D11,0x1D11}, /* 1D10 */
  {0x1D12,0x1D12,0x1D12},  {0x1D13,0x1D13,0x1D13}, /* 1D12 */
  {0x1D14,0x1D14,0x1D14},  {0x1D15,0x1D15,0x1D15}, /* 1D14 */
  {0x1D16,0x1D16,0x1D16},  {0x1D17,0x1D17,0x1D17}, /* 1D16 */
  {0x1D18,0x1D18,0x1D18},  {0x1D19,0x1D19,0x1D19}, /* 1D18 */
  {0x1D1A,0x1D1A,0x1D1A},  {0x1D1B,0x1D1B,0x1D1B}, /* 1D1A */
  {0x1D1C,0x1D1C,0x1D1C},  {0x1D1D,0x1D1D,0x1D1D}, /* 1D1C */
  {0x1D1E,0x1D1E,0x1D1E},  {0x1D1F,0x1D1F,0x1D1F}, /* 1D1E */
  {0x1D20,0x1D20,0x1D20},  {0x1D21,0x1D21,0x1D21}, /* 1D20 */
  {0x1D22,0x1D22,0x1D22},  {0x1D23,0x1D23,0x1D23}, /* 1D22 */
  {0x1D24,0x1D24,0x1D24},  {0x1D25,0x1D25,0x1D25}, /* 1D24 */
  {0x1D26,0x1D26,0x1D26},  {0x1D27,0x1D27,0x1D27}, /* 1D26 */
  {0x1D28,0x1D28,0x1D28},  {0x1D29,0x1D29,0x1D29}, /* 1D28 */
  {0x1D2A,0x1D2A,0x1D2A},  {0x1D2B,0x1D2B,0x1D2B}, /* 1D2A */
  {0x1D2C,0x1D2C,0x1D2C},  {0x1D2D,0x1D2D,0x1D2D}, /* 1D2C */
  {0x1D2E,0x1D2E,0x1D2E},  {0x1D2F,0x1D2F,0x1D2F}, /* 1D2E */
  {0x1D30,0x1D30,0x1D30},  {0x1D31,0x1D31,0x1D31}, /* 1D30 */
  {0x1D32,0x1D32,0x1D32},  {0x1D33,0x1D33,0x1D33}, /* 1D32 */
  {0x1D34,0x1D34,0x1D34},  {0x1D35,0x1D35,0x1D35}, /* 1D34 */
  {0x1D36,0x1D36,0x1D36},  {0x1D37,0x1D37,0x1D37}, /* 1D36 */
  {0x1D38,0x1D38,0x1D38},  {0x1D39,0x1D39,0x1D39}, /* 1D38 */
  {0x1D3A,0x1D3A,0x1D3A},  {0x1D3B,0x1D3B,0x1D3B}, /* 1D3A */
  {0x1D3C,0x1D3C,0x1D3C},  {0x1D3D,0x1D3D,0x1D3D}, /* 1D3C */
  {0x1D3E,0x1D3E,0x1D3E},  {0x1D3F,0x1D3F,0x1D3F}, /* 1D3E */
  {0x1D40,0x1D40,0x1D40},  {0x1D41,0x1D41,0x1D41}, /* 1D40 */
  {0x1D42,0x1D42,0x1D42},  {0x1D43,0x1D43,0x1D43}, /* 1D42 */
  {0x1D44,0x1D44,0x1D44},  {0x1D45,0x1D45,0x1D45}, /* 1D44 */
  {0x1D46,0x1D46,0x1D46},  {0x1D47,0x1D47,0x1D47}, /* 1D46 */
  {0x1D48,0x1D48,0x1D48},  {0x1D49,0x1D49,0x1D49}, /* 1D48 */
  {0x1D4A,0x1D4A,0x1D4A},  {0x1D4B,0x1D4B,0x1D4B}, /* 1D4A */
  {0x1D4C,0x1D4C,0x1D4C},  {0x1D4D,0x1D4D,0x1D4D}, /* 1D4C */
  {0x1D4E,0x1D4E,0x1D4E},  {0x1D4F,0x1D4F,0x1D4F}, /* 1D4E */
  {0x1D50,0x1D50,0x1D50},  {0x1D51,0x1D51,0x1D51}, /* 1D50 */
  {0x1D52,0x1D52,0x1D52},  {0x1D53,0x1D53,0x1D53}, /* 1D52 */
  {0x1D54,0x1D54,0x1D54},  {0x1D55,0x1D55,0x1D55}, /* 1D54 */
  {0x1D56,0x1D56,0x1D56},  {0x1D57,0x1D57,0x1D57}, /* 1D56 */
  {0x1D58,0x1D58,0x1D58},  {0x1D59,0x1D59,0x1D59}, /* 1D58 */
  {0x1D5A,0x1D5A,0x1D5A},  {0x1D5B,0x1D5B,0x1D5B}, /* 1D5A */
  {0x1D5C,0x1D5C,0x1D5C},  {0x1D5D,0x1D5D,0x1D5D}, /* 1D5C */
  {0x1D5E,0x1D5E,0x1D5E},  {0x1D5F,0x1D5F,0x1D5F}, /* 1D5E */
  {0x1D60,0x1D60,0x1D60},  {0x1D61,0x1D61,0x1D61}, /* 1D60 */
  {0x1D62,0x1D62,0x1D62},  {0x1D63,0x1D63,0x1D63}, /* 1D62 */
  {0x1D64,0x1D64,0x1D64},  {0x1D65,0x1D65,0x1D65}, /* 1D64 */
  {0x1D66,0x1D66,0x1D66},  {0x1D67,0x1D67,0x1D67}, /* 1D66 */
  {0x1D68,0x1D68,0x1D68},  {0x1D69,0x1D69,0x1D69}, /* 1D68 */
  {0x1D6A,0x1D6A,0x1D6A},  {0x1D6B,0x1D6B,0x1D6B}, /* 1D6A */
  {0x1D6C,0x1D6C,0x1D6C},  {0x1D6D,0x1D6D,0x1D6D}, /* 1D6C */
  {0x1D6E,0x1D6E,0x1D6E},  {0x1D6F,0x1D6F,0x1D6F}, /* 1D6E */
  {0x1D70,0x1D70,0x1D70},  {0x1D71,0x1D71,0x1D71}, /* 1D70 */
  {0x1D72,0x1D72,0x1D72},  {0x1D73,0x1D73,0x1D73}, /* 1D72 */
  {0x1D74,0x1D74,0x1D74},  {0x1D75,0x1D75,0x1D75}, /* 1D74 */
  {0x1D76,0x1D76,0x1D76},  {0x1D77,0x1D77,0x1D77}, /* 1D76 */
  {0x1D78,0x1D78,0x1D78},  {0xA77D,0x1D79,0xA77D}, /* 1D78 */
  {0x1D7A,0x1D7A,0x1D7A},  {0x1D7B,0x1D7B,0x1D7B}, /* 1D7A */
  {0x1D7C,0x1D7C,0x1D7C},  {0x2C63,0x1D7D,0x2C63}, /* 1D7C */
  {0x1D7E,0x1D7E,0x1D7E},  {0x1D7F,0x1D7F,0x1D7F}, /* 1D7E */
  {0x1D80,0x1D80,0x1D80},  {0x1D81,0x1D81,0x1D81}, /* 1D80 */
  {0x1D82,0x1D82,0x1D82},  {0x1D83,0x1D83,0x1D83}, /* 1D82 */
  {0x1D84,0x1D84,0x1D84},  {0x1D85,0x1D85,0x1D85}, /* 1D84 */
  {0x1D86,0x1D86,0x1D86},  {0x1D87,0x1D87,0x1D87}, /* 1D86 */
  {0x1D88,0x1D88,0x1D88},  {0x1D89,0x1D89,0x1D89}, /* 1D88 */
  {0x1D8A,0x1D8A,0x1D8A},  {0x1D8B,0x1D8B,0x1D8B}, /* 1D8A */
  {0x1D8C,0x1D8C,0x1D8C},  {0x1D8D,0x1D8D,0x1D8D}, /* 1D8C */
  {0x1D8E,0x1D8E,0x1D8E},  {0x1D8F,0x1D8F,0x1D8F}, /* 1D8E */
  {0x1D90,0x1D90,0x1D90},  {0x1D91,0x1D91,0x1D91}, /* 1D90 */
  {0x1D92,0x1D92,0x1D92},  {0x1D93,0x1D93,0x1D93}, /* 1D92 */
  {0x1D94,0x1D94,0x1D94},  {0x1D95,0x1D95,0x1D95}, /* 1D94 */
  {0x1D96,0x1D96,0x1D96},  {0x1D97,0x1D97,0x1D97}, /* 1D96 */
  {0x1D98,0x1D98,0x1D98},  {0x1D99,0x1D99,0x1D99}, /* 1D98 */
  {0x1D9A,0x1D9A,0x1D9A},  {0x1D9B,0x1D9B,0x1D9B}, /* 1D9A */
  {0x1D9C,0x1D9C,0x1D9C},  {0x1D9D,0x1D9D,0x1D9D}, /* 1D9C */
  {0x1D9E,0x1D9E,0x1D9E},  {0x1D9F,0x1D9F,0x1D9F}, /* 1D9E */
  {0x1DA0,0x1DA0,0x1DA0},  {0x1DA1,0x1DA1,0x1DA1}, /* 1DA0 */
  {0x1DA2,0x1DA2,0x1DA2},  {0x1DA3,0x1DA3,0x1DA3}, /* 1DA2 */
  {0x1DA4,0x1DA4,0x1DA4},  {0x1DA5,0x1DA5,0x1DA5}, /* 1DA4 */
  {0x1DA6,0x1DA6,0x1DA6},  {0x1DA7,0x1DA7,0x1DA7}, /* 1DA6 */
  {0x1DA8,0x1DA8,0x1DA8},  {0x1DA9,0x1DA9,0x1DA9}, /* 1DA8 */
  {0x1DAA,0x1DAA,0x1DAA},  {0x1DAB,0x1DAB,0x1DAB}, /* 1DAA */
  {0x1DAC,0x1DAC,0x1DAC},  {0x1DAD,0x1DAD,0x1DAD}, /* 1DAC */
  {0x1DAE,0x1DAE,0x1DAE},  {0x1DAF,0x1DAF,0x1DAF}, /* 1DAE */
  {0x1DB0,0x1DB0,0x1DB0},  {0x1DB1,0x1DB1,0x1DB1}, /* 1DB0 */
  {0x1DB2,0x1DB2,0x1DB2},  {0x1DB3,0x1DB3,0x1DB3}, /* 1DB2 */
  {0x1DB4,0x1DB4,0x1DB4},  {0x1DB5,0x1DB5,0x1DB5}, /* 1DB4 */
  {0x1DB6,0x1DB6,0x1DB6},  {0x1DB7,0x1DB7,0x1DB7}, /* 1DB6 */
  {0x1DB8,0x1DB8,0x1DB8},  {0x1DB9,0x1DB9,0x1DB9}, /* 1DB8 */
  {0x1DBA,0x1DBA,0x1DBA},  {0x1DBB,0x1DBB,0x1DBB}, /* 1DBA */
  {0x1DBC,0x1DBC,0x1DBC},  {0x1DBD,0x1DBD,0x1DBD}, /* 1DBC */
  {0x1DBE,0x1DBE,0x1DBE},  {0x1DBF,0x1DBF,0x1DBF}, /* 1DBE */
  {0x1DC0,0x1DC0,0x1DC0},  {0x1DC1,0x1DC1,0x1DC1}, /* 1DC0 */
  {0x1DC2,0x1DC2,0x1DC2},  {0x1DC3,0x1DC3,0x1DC3}, /* 1DC2 */
  {0x1DC4,0x1DC4,0x1DC4},  {0x1DC5,0x1DC5,0x1DC5}, /* 1DC4 */
  {0x1DC6,0x1DC6,0x1DC6},  {0x1DC7,0x1DC7,0x1DC7}, /* 1DC6 */
  {0x1DC8,0x1DC8,0x1DC8},  {0x1DC9,0x1DC9,0x1DC9}, /* 1DC8 */
  {0x1DCA,0x1DCA,0x1DCA},  {0x1DCB,0x1DCB,0x1DCB}, /* 1DCA */
  {0x1DCC,0x1DCC,0x1DCC},  {0x1DCD,0x1DCD,0x1DCD}, /* 1DCC */
  {0x1DCE,0x1DCE,0x1DCE},  {0x1DCF,0x1DCF,0x1DCF}, /* 1DCE */
  {0x1DD0,0x1DD0,0x1DD0},  {0x1DD1,0x1DD1,0x1DD1}, /* 1DD0 */
  {0x1DD2,0x1DD2,0x1DD2},  {0x1DD3,0x1DD3,0x1DD3}, /* 1DD2 */
  {0x1DD4,0x1DD4,0x1DD4},  {0x1DD5,0x1DD5,0x1DD5}, /* 1DD4 */
  {0x1DD6,0x1DD6,0x1DD6},  {0x1DD7,0x1DD7,0x1DD7}, /* 1DD6 */
  {0x1DD8,0x1DD8,0x1DD8},  {0x1DD9,0x1DD9,0x1DD9}, /* 1DD8 */
  {0x1DDA,0x1DDA,0x1DDA},  {0x1DDB,0x1DDB,0x1DDB}, /* 1DDA */
  {0x1DDC,0x1DDC,0x1DDC},  {0x1DDD,0x1DDD,0x1DDD}, /* 1DDC */
  {0x1DDE,0x1DDE,0x1DDE},  {0x1DDF,0x1DDF,0x1DDF}, /* 1DDE */
  {0x1DE0,0x1DE0,0x1DE0},  {0x1DE1,0x1DE1,0x1DE1}, /* 1DE0 */
  {0x1DE2,0x1DE2,0x1DE2},  {0x1DE3,0x1DE3,0x1DE3}, /* 1DE2 */
  {0x1DE4,0x1DE4,0x1DE4},  {0x1DE5,0x1DE5,0x1DE5}, /* 1DE4 */
  {0x1DE6,0x1DE6,0x1DE6},  {0x1DE7,0x1DE7,0x1DE7}, /* 1DE6 */
  {0x1DE8,0x1DE8,0x1DE8},  {0x1DE9,0x1DE9,0x1DE9}, /* 1DE8 */
  {0x1DEA,0x1DEA,0x1DEA},  {0x1DEB,0x1DEB,0x1DEB}, /* 1DEA */
  {0x1DEC,0x1DEC,0x1DEC},  {0x1DED,0x1DED,0x1DED}, /* 1DEC */
  {0x1DEE,0x1DEE,0x1DEE},  {0x1DEF,0x1DEF,0x1DEF}, /* 1DEE */
  {0x1DF0,0x1DF0,0x1DF0},  {0x1DF1,0x1DF1,0x1DF1}, /* 1DF0 */
  {0x1DF2,0x1DF2,0x1DF2},  {0x1DF3,0x1DF3,0x1DF3}, /* 1DF2 */
  {0x1DF4,0x1DF4,0x1DF4},  {0x1DF5,0x1DF5,0x1DF5}, /* 1DF4 */
  {0x1DF6,0x1DF6,0x1DF6},  {0x1DF7,0x1DF7,0x1DF7}, /* 1DF6 */
  {0x1DF8,0x1DF8,0x1DF8},  {0x1DF9,0x1DF9,0x1DF9}, /* 1DF8 */
  {0x1DFA,0x1DFA,0x1DFA},  {0x1DFB,0x1DFB,0x1DFB}, /* 1DFA */
  {0x1DFC,0x1DFC,0x1DFC},  {0x1DFD,0x1DFD,0x1DFD}, /* 1DFC */
  {0x1DFE,0x1DFE,0x1DFE},  {0x1DFF,0x1DFF,0x1DFF} /* 1DFE */
};

static MY_UNICASE_CHARACTER u520p1E[]={
  {0x1E00,0x1E01,0x0041},  {0x1E00,0x1E01,0x0041}, /* 1E00 */
  {0x1E02,0x1E03,0x0042},  {0x1E02,0x1E03,0x0042}, /* 1E02 */
  {0x1E04,0x1E05,0x0042},  {0x1E04,0x1E05,0x0042}, /* 1E04 */
  {0x1E06,0x1E07,0x0042},  {0x1E06,0x1E07,0x0042}, /* 1E06 */
  {0x1E08,0x1E09,0x0043},  {0x1E08,0x1E09,0x0043}, /* 1E08 */
  {0x1E0A,0x1E0B,0x0044},  {0x1E0A,0x1E0B,0x0044}, /* 1E0A */
  {0x1E0C,0x1E0D,0x0044},  {0x1E0C,0x1E0D,0x0044}, /* 1E0C */
  {0x1E0E,0x1E0F,0x0044},  {0x1E0E,0x1E0F,0x0044}, /* 1E0E */
  {0x1E10,0x1E11,0x0044},  {0x1E10,0x1E11,0x0044}, /* 1E10 */
  {0x1E12,0x1E13,0x0044},  {0x1E12,0x1E13,0x0044}, /* 1E12 */
  {0x1E14,0x1E15,0x0045},  {0x1E14,0x1E15,0x0045}, /* 1E14 */
  {0x1E16,0x1E17,0x0045},  {0x1E16,0x1E17,0x0045}, /* 1E16 */
  {0x1E18,0x1E19,0x0045},  {0x1E18,0x1E19,0x0045}, /* 1E18 */
  {0x1E1A,0x1E1B,0x0045},  {0x1E1A,0x1E1B,0x0045}, /* 1E1A */
  {0x1E1C,0x1E1D,0x0045},  {0x1E1C,0x1E1D,0x0045}, /* 1E1C */
  {0x1E1E,0x1E1F,0x0046},  {0x1E1E,0x1E1F,0x0046}, /* 1E1E */
  {0x1E20,0x1E21,0x0047},  {0x1E20,0x1E21,0x0047}, /* 1E20 */
  {0x1E22,0x1E23,0x0048},  {0x1E22,0x1E23,0x0048}, /* 1E22 */
  {0x1E24,0x1E25,0x0048},  {0x1E24,0x1E25,0x0048}, /* 1E24 */
  {0x1E26,0x1E27,0x0048},  {0x1E26,0x1E27,0x0048}, /* 1E26 */
  {0x1E28,0x1E29,0x0048},  {0x1E28,0x1E29,0x0048}, /* 1E28 */
  {0x1E2A,0x1E2B,0x0048},  {0x1E2A,0x1E2B,0x0048}, /* 1E2A */
  {0x1E2C,0x1E2D,0x0049},  {0x1E2C,0x1E2D,0x0049}, /* 1E2C */
  {0x1E2E,0x1E2F,0x0049},  {0x1E2E,0x1E2F,0x0049}, /* 1E2E */
  {0x1E30,0x1E31,0x004B},  {0x1E30,0x1E31,0x004B}, /* 1E30 */
  {0x1E32,0x1E33,0x004B},  {0x1E32,0x1E33,0x004B}, /* 1E32 */
  {0x1E34,0x1E35,0x004B},  {0x1E34,0x1E35,0x004B}, /* 1E34 */
  {0x1E36,0x1E37,0x004C},  {0x1E36,0x1E37,0x004C}, /* 1E36 */
  {0x1E38,0x1E39,0x004C},  {0x1E38,0x1E39,0x004C}, /* 1E38 */
  {0x1E3A,0x1E3B,0x004C},  {0x1E3A,0x1E3B,0x004C}, /* 1E3A */
  {0x1E3C,0x1E3D,0x004C},  {0x1E3C,0x1E3D,0x004C}, /* 1E3C */
  {0x1E3E,0x1E3F,0x004D},  {0x1E3E,0x1E3F,0x004D}, /* 1E3E */
  {0x1E40,0x1E41,0x004D},  {0x1E40,0x1E41,0x004D}, /* 1E40 */
  {0x1E42,0x1E43,0x004D},  {0x1E42,0x1E43,0x004D}, /* 1E42 */
  {0x1E44,0x1E45,0x004E},  {0x1E44,0x1E45,0x004E}, /* 1E44 */
  {0x1E46,0x1E47,0x004E},  {0x1E46,0x1E47,0x004E}, /* 1E46 */
  {0x1E48,0x1E49,0x004E},  {0x1E48,0x1E49,0x004E}, /* 1E48 */
  {0x1E4A,0x1E4B,0x004E},  {0x1E4A,0x1E4B,0x004E}, /* 1E4A */
  {0x1E4C,0x1E4D,0x004F},  {0x1E4C,0x1E4D,0x004F}, /* 1E4C */
  {0x1E4E,0x1E4F,0x004F},  {0x1E4E,0x1E4F,0x004F}, /* 1E4E */
  {0x1E50,0x1E51,0x004F},  {0x1E50,0x1E51,0x004F}, /* 1E50 */
  {0x1E52,0x1E53,0x004F},  {0x1E52,0x1E53,0x004F}, /* 1E52 */
  {0x1E54,0x1E55,0x0050},  {0x1E54,0x1E55,0x0050}, /* 1E54 */
  {0x1E56,0x1E57,0x0050},  {0x1E56,0x1E57,0x0050}, /* 1E56 */
  {0x1E58,0x1E59,0x0052},  {0x1E58,0x1E59,0x0052}, /* 1E58 */
  {0x1E5A,0x1E5B,0x0052},  {0x1E5A,0x1E5B,0x0052}, /* 1E5A */
  {0x1E5C,0x1E5D,0x0052},  {0x1E5C,0x1E5D,0x0052}, /* 1E5C */
  {0x1E5E,0x1E5F,0x0052},  {0x1E5E,0x1E5F,0x0052}, /* 1E5E */
  {0x1E60,0x1E61,0x0053},  {0x1E60,0x1E61,0x0053}, /* 1E60 */
  {0x1E62,0x1E63,0x0053},  {0x1E62,0x1E63,0x0053}, /* 1E62 */
  {0x1E64,0x1E65,0x0053},  {0x1E64,0x1E65,0x0053}, /* 1E64 */
  {0x1E66,0x1E67,0x0053},  {0x1E66,0x1E67,0x0053}, /* 1E66 */
  {0x1E68,0x1E69,0x0053},  {0x1E68,0x1E69,0x0053}, /* 1E68 */
  {0x1E6A,0x1E6B,0x0054},  {0x1E6A,0x1E6B,0x0054}, /* 1E6A */
  {0x1E6C,0x1E6D,0x0054},  {0x1E6C,0x1E6D,0x0054}, /* 1E6C */
  {0x1E6E,0x1E6F,0x0054},  {0x1E6E,0x1E6F,0x0054}, /* 1E6E */
  {0x1E70,0x1E71,0x0054},  {0x1E70,0x1E71,0x0054}, /* 1E70 */
  {0x1E72,0x1E73,0x0055},  {0x1E72,0x1E73,0x0055}, /* 1E72 */
  {0x1E74,0x1E75,0x0055},  {0x1E74,0x1E75,0x0055}, /* 1E74 */
  {0x1E76,0x1E77,0x0055},  {0x1E76,0x1E77,0x0055}, /* 1E76 */
  {0x1E78,0x1E79,0x0055},  {0x1E78,0x1E79,0x0055}, /* 1E78 */
  {0x1E7A,0x1E7B,0x0055},  {0x1E7A,0x1E7B,0x0055}, /* 1E7A */
  {0x1E7C,0x1E7D,0x0056},  {0x1E7C,0x1E7D,0x0056}, /* 1E7C */
  {0x1E7E,0x1E7F,0x0056},  {0x1E7E,0x1E7F,0x0056}, /* 1E7E */
  {0x1E80,0x1E81,0x0057},  {0x1E80,0x1E81,0x0057}, /* 1E80 */
  {0x1E82,0x1E83,0x0057},  {0x1E82,0x1E83,0x0057}, /* 1E82 */
  {0x1E84,0x1E85,0x0057},  {0x1E84,0x1E85,0x0057}, /* 1E84 */
  {0x1E86,0x1E87,0x0057},  {0x1E86,0x1E87,0x0057}, /* 1E86 */
  {0x1E88,0x1E89,0x0057},  {0x1E88,0x1E89,0x0057}, /* 1E88 */
  {0x1E8A,0x1E8B,0x0058},  {0x1E8A,0x1E8B,0x0058}, /* 1E8A */
  {0x1E8C,0x1E8D,0x0058},  {0x1E8C,0x1E8D,0x0058}, /* 1E8C */
  {0x1E8E,0x1E8F,0x0059},  {0x1E8E,0x1E8F,0x0059}, /* 1E8E */
  {0x1E90,0x1E91,0x005A},  {0x1E90,0x1E91,0x005A}, /* 1E90 */
  {0x1E92,0x1E93,0x005A},  {0x1E92,0x1E93,0x005A}, /* 1E92 */
  {0x1E94,0x1E95,0x005A},  {0x1E94,0x1E95,0x005A}, /* 1E94 */
  {0x1E96,0x1E96,0x0048},  {0x1E97,0x1E97,0x0054}, /* 1E96 */
  {0x1E98,0x1E98,0x0057},  {0x1E99,0x1E99,0x0059}, /* 1E98 */
  {0x1E9A,0x1E9A,0x1E9A},  {0x1E60,0x1E9B,0x0053}, /* 1E9A */
  {0x1E9C,0x1E9C,0x1E9C},  {0x1E9D,0x1E9D,0x1E9D}, /* 1E9C */
  {0x1E9E,0x00DF,0x1E9E},  {0x1E9F,0x1E9F,0x1E9F}, /* 1E9E */
  {0x1EA0,0x1EA1,0x0041},  {0x1EA0,0x1EA1,0x0041}, /* 1EA0 */
  {0x1EA2,0x1EA3,0x0041},  {0x1EA2,0x1EA3,0x0041}, /* 1EA2 */
  {0x1EA4,0x1EA5,0x0041},  {0x1EA4,0x1EA5,0x0041}, /* 1EA4 */
  {0x1EA6,0x1EA7,0x0041},  {0x1EA6,0x1EA7,0x0041}, /* 1EA6 */
  {0x1EA8,0x1EA9,0x0041},  {0x1EA8,0x1EA9,0x0041}, /* 1EA8 */
  {0x1EAA,0x1EAB,0x0041},  {0x1EAA,0x1EAB,0x0041}, /* 1EAA */
  {0x1EAC,0x1EAD,0x0041},  {0x1EAC,0x1EAD,0x0041}, /* 1EAC */
  {0x1EAE,0x1EAF,0x0041},  {0x1EAE,0x1EAF,0x0041}, /* 1EAE */
  {0x1EB0,0x1EB1,0x0041},  {0x1EB0,0x1EB1,0x0041}, /* 1EB0 */
  {0x1EB2,0x1EB3,0x0041},  {0x1EB2,0x1EB3,0x0041}, /* 1EB2 */
  {0x1EB4,0x1EB5,0x0041},  {0x1EB4,0x1EB5,0x0041}, /* 1EB4 */
  {0x1EB6,0x1EB7,0x0041},  {0x1EB6,0x1EB7,0x0041}, /* 1EB6 */
  {0x1EB8,0x1EB9,0x0045},  {0x1EB8,0x1EB9,0x0045}, /* 1EB8 */
  {0x1EBA,0x1EBB,0x0045},  {0x1EBA,0x1EBB,0x0045}, /* 1EBA */
  {0x1EBC,0x1EBD,0x0045},  {0x1EBC,0x1EBD,0x0045}, /* 1EBC */
  {0x1EBE,0x1EBF,0x0045},  {0x1EBE,0x1EBF,0x0045}, /* 1EBE */
  {0x1EC0,0x1EC1,0x0045},  {0x1EC0,0x1EC1,0x0045}, /* 1EC0 */
  {0x1EC2,0x1EC3,0x0045},  {0x1EC2,0x1EC3,0x0045}, /* 1EC2 */
  {0x1EC4,0x1EC5,0x0045},  {0x1EC4,0x1EC5,0x0045}, /* 1EC4 */
  {0x1EC6,0x1EC7,0x0045},  {0x1EC6,0x1EC7,0x0045}, /* 1EC6 */
  {0x1EC8,0x1EC9,0x0049},  {0x1EC8,0x1EC9,0x0049}, /* 1EC8 */
  {0x1ECA,0x1ECB,0x0049},  {0x1ECA,0x1ECB,0x0049}, /* 1ECA */
  {0x1ECC,0x1ECD,0x004F},  {0x1ECC,0x1ECD,0x004F}, /* 1ECC */
  {0x1ECE,0x1ECF,0x004F},  {0x1ECE,0x1ECF,0x004F}, /* 1ECE */
  {0x1ED0,0x1ED1,0x004F},  {0x1ED0,0x1ED1,0x004F}, /* 1ED0 */
  {0x1ED2,0x1ED3,0x004F},  {0x1ED2,0x1ED3,0x004F}, /* 1ED2 */
  {0x1ED4,0x1ED5,0x004F},  {0x1ED4,0x1ED5,0x004F}, /* 1ED4 */
  {0x1ED6,0x1ED7,0x004F},  {0x1ED6,0x1ED7,0x004F}, /* 1ED6 */
  {0x1ED8,0x1ED9,0x004F},  {0x1ED8,0x1ED9,0x004F}, /* 1ED8 */
  {0x1EDA,0x1EDB,0x004F},  {0x1EDA,0x1EDB,0x004F}, /* 1EDA */
  {0x1EDC,0x1EDD,0x004F},  {0x1EDC,0x1EDD,0x004F}, /* 1EDC */
  {0x1EDE,0x1EDF,0x004F},  {0x1EDE,0x1EDF,0x004F}, /* 1EDE */
  {0x1EE0,0x1EE1,0x004F},  {0x1EE0,0x1EE1,0x004F}, /* 1EE0 */
  {0x1EE2,0x1EE3,0x004F},  {0x1EE2,0x1EE3,0x004F}, /* 1EE2 */
  {0x1EE4,0x1EE5,0x0055},  {0x1EE4,0x1EE5,0x0055}, /* 1EE4 */
  {0x1EE6,0x1EE7,0x0055},  {0x1EE6,0x1EE7,0x0055}, /* 1EE6 */
  {0x1EE8,0x1EE9,0x0055},  {0x1EE8,0x1EE9,0x0055}, /* 1EE8 */
  {0x1EEA,0x1EEB,0x0055},  {0x1EEA,0x1EEB,0x0055}, /* 1EEA */
  {0x1EEC,0x1EED,0x0055},  {0x1EEC,0x1EED,0x0055}, /* 1EEC */
  {0x1EEE,0x1EEF,0x0055},  {0x1EEE,0x1EEF,0x0055}, /* 1EEE */
  {0x1EF0,0x1EF1,0x0055},  {0x1EF0,0x1EF1,0x0055}, /* 1EF0 */
  {0x1EF2,0x1EF3,0x0059},  {0x1EF2,0x1EF3,0x0059}, /* 1EF2 */
  {0x1EF4,0x1EF5,0x0059},  {0x1EF4,0x1EF5,0x0059}, /* 1EF4 */
  {0x1EF6,0x1EF7,0x0059},  {0x1EF6,0x1EF7,0x0059}, /* 1EF6 */
  {0x1EF8,0x1EF9,0x0059},  {0x1EF8,0x1EF9,0x0059}, /* 1EF8 */
  {0x1EFA,0x1EFB,0x1EFA},  {0x1EFA,0x1EFB,0x1EFA}, /* 1EFA */
  {0x1EFC,0x1EFD,0x1EFC},  {0x1EFC,0x1EFD,0x1EFC}, /* 1EFC */
  {0x1EFE,0x1EFF,0x1EFE},  {0x1EFE,0x1EFF,0x1EFE} /* 1EFE */
};

static MY_UNICASE_CHARACTER u520p1F[]={
  {0x1F08,0x1F00,0x0391},  {0x1F09,0x1F01,0x0391}, /* 1F00 */
  {0x1F0A,0x1F02,0x0391},  {0x1F0B,0x1F03,0x0391}, /* 1F02 */
  {0x1F0C,0x1F04,0x0391},  {0x1F0D,0x1F05,0x0391}, /* 1F04 */
  {0x1F0E,0x1F06,0x0391},  {0x1F0F,0x1F07,0x0391}, /* 1F06 */
  {0x1F08,0x1F00,0x0391},  {0x1F09,0x1F01,0x0391}, /* 1F08 */
  {0x1F0A,0x1F02,0x0391},  {0x1F0B,0x1F03,0x0391}, /* 1F0A */
  {0x1F0C,0x1F04,0x0391},  {0x1F0D,0x1F05,0x0391}, /* 1F0C */
  {0x1F0E,0x1F06,0x0391},  {0x1F0F,0x1F07,0x0391}, /* 1F0E */
  {0x1F18,0x1F10,0x0395},  {0x1F19,0x1F11,0x0395}, /* 1F10 */
  {0x1F1A,0x1F12,0x0395},  {0x1F1B,0x1F13,0x0395}, /* 1F12 */
  {0x1F1C,0x1F14,0x0395},  {0x1F1D,0x1F15,0x0395}, /* 1F14 */
  {0x1F16,0x1F16,0x1F16},  {0x1F17,0x1F17,0x1F17}, /* 1F16 */
  {0x1F18,0x1F10,0x0395},  {0x1F19,0x1F11,0x0395}, /* 1F18 */
  {0x1F1A,0x1F12,0x0395},  {0x1F1B,0x1F13,0x0395}, /* 1F1A */
  {0x1F1C,0x1F14,0x0395},  {0x1F1D,0x1F15,0x0395}, /* 1F1C */
  {0x1F1E,0x1F1E,0x1F1E},  {0x1F1F,0x1F1F,0x1F1F}, /* 1F1E */
  {0x1F28,0x1F20,0x0397},  {0x1F29,0x1F21,0x0397}, /* 1F20 */
  {0x1F2A,0x1F22,0x0397},  {0x1F2B,0x1F23,0x0397}, /* 1F22 */
  {0x1F2C,0x1F24,0x0397},  {0x1F2D,0x1F25,0x0397}, /* 1F24 */
  {0x1F2E,0x1F26,0x0397},  {0x1F2F,0x1F27,0x0397}, /* 1F26 */
  {0x1F28,0x1F20,0x0397},  {0x1F29,0x1F21,0x0397}, /* 1F28 */
  {0x1F2A,0x1F22,0x0397},  {0x1F2B,0x1F23,0x0397}, /* 1F2A */
  {0x1F2C,0x1F24,0x0397},  {0x1F2D,0x1F25,0x0397}, /* 1F2C */
  {0x1F2E,0x1F26,0x0397},  {0x1F2F,0x1F27,0x0397}, /* 1F2E */
  {0x1F38,0x1F30,0x0399},  {0x1F39,0x1F31,0x0399}, /* 1F30 */
  {0x1F3A,0x1F32,0x0399},  {0x1F3B,0x1F33,0x0399}, /* 1F32 */
  {0x1F3C,0x1F34,0x0399},  {0x1F3D,0x1F35,0x0399}, /* 1F34 */
  {0x1F3E,0x1F36,0x0399},  {0x1F3F,0x1F37,0x0399}, /* 1F36 */
  {0x1F38,0x1F30,0x0399},  {0x1F39,0x1F31,0x0399}, /* 1F38 */
  {0x1F3A,0x1F32,0x0399},  {0x1F3B,0x1F33,0x0399}, /* 1F3A */
  {0x1F3C,0x1F34,0x0399},  {0x1F3D,0x1F35,0x0399}, /* 1F3C */
  {0x1F3E,0x1F36,0x0399},  {0x1F3F,0x1F37,0x0399}, /* 1F3E */
  {0x1F48,0x1F40,0x039F},  {0x1F49,0x1F41,0x039F}, /* 1F40 */
  {0x1F4A,0x1F42,0x039F},  {0x1F4B,0x1F43,0x039F}, /* 1F42 */
  {0x1F4C,0x1F44,0x039F},  {0x1F4D,0x1F45,0x039F}, /* 1F44 */
  {0x1F46,0x1F46,0x1F46},  {0x1F47,0x1F47,0x1F47}, /* 1F46 */
  {0x1F48,0x1F40,0x039F},  {0x1F49,0x1F41,0x039F}, /* 1F48 */
  {0x1F4A,0x1F42,0x039F},  {0x1F4B,0x1F43,0x039F}, /* 1F4A */
  {0x1F4C,0x1F44,0x039F},  {0x1F4D,0x1F45,0x039F}, /* 1F4C */
  {0x1F4E,0x1F4E,0x1F4E},  {0x1F4F,0x1F4F,0x1F4F}, /* 1F4E */
  {0x1F50,0x1F50,0x03A5},  {0x1F59,0x1F51,0x03A5}, /* 1F50 */
  {0x1F52,0x1F52,0x03A5},  {0x1F5B,0x1F53,0x03A5}, /* 1F52 */
  {0x1F54,0x1F54,0x03A5},  {0x1F5D,0x1F55,0x03A5}, /* 1F54 */
  {0x1F56,0x1F56,0x03A5},  {0x1F5F,0x1F57,0x03A5}, /* 1F56 */
  {0x1F58,0x1F58,0x1F58},  {0x1F59,0x1F51,0x03A5}, /* 1F58 */
  {0x1F5A,0x1F5A,0x1F5A},  {0x1F5B,0x1F53,0x03A5}, /* 1F5A */
  {0x1F5C,0x1F5C,0x1F5C},  {0x1F5D,0x1F55,0x03A5}, /* 1F5C */
  {0x1F5E,0x1F5E,0x1F5E},  {0x1F5F,0x1F57,0x03A5}, /* 1F5E */
  {0x1F68,0x1F60,0x03A9},  {0x1F69,0x1F61,0x03A9}, /* 1F60 */
  {0x1F6A,0x1F62,0x03A9},  {0x1F6B,0x1F63,0x03A9}, /* 1F62 */
  {0x1F6C,0x1F64,0x03A9},  {0x1F6D,0x1F65,0x03A9}, /* 1F64 */
  {0x1F6E,0x1F66,0x03A9},  {0x1F6F,0x1F67,0x03A9}, /* 1F66 */
  {0x1F68,0x1F60,0x03A9},  {0x1F69,0x1F61,0x03A9}, /* 1F68 */
  {0x1F6A,0x1F62,0x03A9},  {0x1F6B,0x1F63,0x03A9}, /* 1F6A */
  {0x1F6C,0x1F64,0x03A9},  {0x1F6D,0x1F65,0x03A9}, /* 1F6C */
  {0x1F6E,0x1F66,0x03A9},  {0x1F6F,0x1F67,0x03A9}, /* 1F6E */
  {0x1FBA,0x1F70,0x0391},  {0x1FBB,0x1F71,0x1FBB}, /* 1F70 */
  {0x1FC8,0x1F72,0x0395},  {0x1FC9,0x1F73,0x1FC9}, /* 1F72 */
  {0x1FCA,0x1F74,0x0397},  {0x1FCB,0x1F75,0x1FCB}, /* 1F74 */
  {0x1FDA,0x1F76,0x0399},  {0x1FDB,0x1F77,0x1FDB}, /* 1F76 */
  {0x1FF8,0x1F78,0x039F},  {0x1FF9,0x1F79,0x1FF9}, /* 1F78 */
  {0x1FEA,0x1F7A,0x03A5},  {0x1FEB,0x1F7B,0x1FEB}, /* 1F7A */
  {0x1FFA,0x1F7C,0x03A9},  {0x1FFB,0x1F7D,0x1FFB}, /* 1F7C */
  {0x1F7E,0x1F7E,0x1F7E},  {0x1F7F,0x1F7F,0x1F7F}, /* 1F7E */
  {0x1F88,0x1F80,0x0391},  {0x1F89,0x1F81,0x0391}, /* 1F80 */
  {0x1F8A,0x1F82,0x0391},  {0x1F8B,0x1F83,0x0391}, /* 1F82 */
  {0x1F8C,0x1F84,0x0391},  {0x1F8D,0x1F85,0x0391}, /* 1F84 */
  {0x1F8E,0x1F86,0x0391},  {0x1F8F,0x1F87,0x0391}, /* 1F86 */
  {0x1F88,0x1F80,0x0391},  {0x1F89,0x1F81,0x0391}, /* 1F88 */
  {0x1F8A,0x1F82,0x0391},  {0x1F8B,0x1F83,0x0391}, /* 1F8A */
  {0x1F8C,0x1F84,0x0391},  {0x1F8D,0x1F85,0x0391}, /* 1F8C */
  {0x1F8E,0x1F86,0x0391},  {0x1F8F,0x1F87,0x0391}, /* 1F8E */
  {0x1F98,0x1F90,0x0397},  {0x1F99,0x1F91,0x0397}, /* 1F90 */
  {0x1F9A,0x1F92,0x0397},  {0x1F9B,0x1F93,0x0397}, /* 1F92 */
  {0x1F9C,0x1F94,0x0397},  {0x1F9D,0x1F95,0x0397}, /* 1F94 */
  {0x1F9E,0x1F96,0x0397},  {0x1F9F,0x1F97,0x0397}, /* 1F96 */
  {0x1F98,0x1F90,0x0397},  {0x1F99,0x1F91,0x0397}, /* 1F98 */
  {0x1F9A,0x1F92,0x0397},  {0x1F9B,0x1F93,0x0397}, /* 1F9A */
  {0x1F9C,0x1F94,0x0397},  {0x1F9D,0x1F95,0x0397}, /* 1F9C */
  {0x1F9E,0x1F96,0x0397},  {0x1F9F,0x1F97,0x0397}, /* 1F9E */
  {0x1FA8,0x1FA0,0x03A9},  {0x1FA9,0x1FA1,0x03A9}, /* 1FA0 */
  {0x1FAA,0x1FA2,0x03A9},  {0x1FAB,0x1FA3,0x03A9}, /* 1FA2 */
  {0x1FAC,0x1FA4,0x03A9},  {0x1FAD,0x1FA5,0x03A9}, /* 1FA4 */
  {0x1FAE,0x1FA6,0x03A9},  {0x1FAF,0x1FA7,0x03A9}, /* 1FA6 */
  {0x1FA8,0x1FA0,0x03A9},  {0x1FA9,0x1FA1,0x03A9}, /* 1FA8 */
  {0x1FAA,0x1FA2,0x03A9},  {0x1FAB,0x1FA3,0x03A9}, /* 1FAA */
  {0x1FAC,0x1FA4,0x03A9},  {0x1FAD,0x1FA5,0x03A9}, /* 1FAC */
  {0x1FAE,0x1FA6,0x03A9},  {0x1FAF,0x1FA7,0x03A9}, /* 1FAE */
  {0x1FB8,0x1FB0,0x0391},  {0x1FB9,0x1FB1,0x0391}, /* 1FB0 */
  {0x1FB2,0x1FB2,0x0391},  {0x1FBC,0x1FB3,0x0391}, /* 1FB2 */
  {0x1FB4,0x1FB4,0x0391},  {0x1FB5,0x1FB5,0x1FB5}, /* 1FB4 */
  {0x1FB6,0x1FB6,0x0391},  {0x1FB7,0x1FB7,0x0391}, /* 1FB6 */
  {0x1FB8,0x1FB0,0x0391},  {0x1FB9,0x1FB1,0x0391}, /* 1FB8 */
  {0x1FBA,0x1F70,0x0391},  {0x1FBB,0x1F71,0x1FBB}, /* 1FBA */
  {0x1FBC,0x1FB3,0x0391},  {0x1FBD,0x1FBD,0x1FBD}, /* 1FBC */
  {0x0399,0x1FBE,0x0399},  {0x1FBF,0x1FBF,0x1FBF}, /* 1FBE */
  {0x1FC0,0x1FC0,0x1FC0},  {0x1FC1,0x1FC1,0x1FC1}, /* 1FC0 */
  {0x1FC2,0x1FC2,0x0397},  {0x1FCC,0x1FC3,0x0397}, /* 1FC2 */
  {0x1FC4,0x1FC4,0x0397},  {0x1FC5,0x1FC5,0x1FC5}, /* 1FC4 */
  {0x1FC6,0x1FC6,0x0397},  {0x1FC7,0x1FC7,0x0397}, /* 1FC6 */
  {0x1FC8,0x1F72,0x0395},  {0x1FC9,0x1F73,0x1FC9}, /* 1FC8 */
  {0x1FCA,0x1F74,0x0397},  {0x1FCB,0x1F75,0x1FCB}, /* 1FCA */
  {0x1FCC,0x1FC3,0x0397},  {0x1FCD,0x1FCD,0x1FCD}, /* 1FCC */
  {0x1FCE,0x1FCE,0x1FCE},  {0x1FCF,0x1FCF,0x1FCF}, /* 1FCE */
  {0x1FD8,0x1FD0,0x0399},  {0x1FD9,0x1FD1,0x0399}, /* 1FD0 */
  {0x1FD2,0x1FD2,0x0399},  {0x1FD3,0x1FD3,0x1FD3}, /* 1FD2 */
  {0x1FD4,0x1FD4,0x1FD4},  {0x1FD5,0x1FD5,0x1FD5}, /* 1FD4 */
  {0x1FD6,0x1FD6,0x0399},  {0x1FD7,0x1FD7,0x0399}, /* 1FD6 */
  {0x1FD8,0x1FD0,0x0399},  {0x1FD9,0x1FD1,0x0399}, /* 1FD8 */
  {0x1FDA,0x1F76,0x0399},  {0x1FDB,0x1F77,0x1FDB}, /* 1FDA */
  {0x1FDC,0x1FDC,0x1FDC},  {0x1FDD,0x1FDD,0x1FDD}, /* 1FDC */
  {0x1FDE,0x1FDE,0x1FDE},  {0x1FDF,0x1FDF,0x1FDF}, /* 1FDE */
  {0x1FE8,0x1FE0,0x03A5},  {0x1FE9,0x1FE1,0x03A5}, /* 1FE0 */
  {0x1FE2,0x1FE2,0x03A5},  {0x1FE3,0x1FE3,0x1FE3}, /* 1FE2 */
  {0x1FE4,0x1FE4,0x03A1},  {0x1FEC,0x1FE5,0x03A1}, /* 1FE4 */
  {0x1FE6,0x1FE6,0x03A5},  {0x1FE7,0x1FE7,0x03A5}, /* 1FE6 */
  {0x1FE8,0x1FE0,0x03A5},  {0x1FE9,0x1FE1,0x03A5}, /* 1FE8 */
  {0x1FEA,0x1F7A,0x03A5},  {0x1FEB,0x1F7B,0x1FEB}, /* 1FEA */
  {0x1FEC,0x1FE5,0x03A1},  {0x1FED,0x1FED,0x1FED}, /* 1FEC */
  {0x1FEE,0x1FEE,0x1FEE},  {0x1FEF,0x1FEF,0x1FEF}, /* 1FEE */
  {0x1FF0,0x1FF0,0x1FF0},  {0x1FF1,0x1FF1,0x1FF1}, /* 1FF0 */
  {0x1FF2,0x1FF2,0x03A9},  {0x1FFC,0x1FF3,0x03A9}, /* 1FF2 */
  {0x1FF4,0x1FF4,0x03A9},  {0x1FF5,0x1FF5,0x1FF5}, /* 1FF4 */
  {0x1FF6,0x1FF6,0x03A9},  {0x1FF7,0x1FF7,0x03A9}, /* 1FF6 */
  {0x1FF8,0x1F78,0x039F},  {0x1FF9,0x1F79,0x1FF9}, /* 1FF8 */
  {0x1FFA,0x1F7C,0x03A9},  {0x1FFB,0x1F7D,0x1FFB}, /* 1FFA */
  {0x1FFC,0x1FF3,0x03A9},  {0x1FFD,0x1FFD,0x1FFD}, /* 1FFC */
  {0x1FFE,0x1FFE,0x1FFE},  {0x1FFF,0x1FFF,0x1FFF} /* 1FFE */
};

static MY_UNICASE_CHARACTER u520p21[]={
  {0x2100,0x2100,0x2100},  {0x2101,0x2101,0x2101}, /* 2100 */
  {0x2102,0x2102,0x2102},  {0x2103,0x2103,0x2103}, /* 2102 */
  {0x2104,0x2104,0x2104},  {0x2105,0x2105,0x2105}, /* 2104 */
  {0x2106,0x2106,0x2106},  {0x2107,0x2107,0x2107}, /* 2106 */
  {0x2108,0x2108,0x2108},  {0x2109,0x2109,0x2109}, /* 2108 */
  {0x210A,0x210A,0x210A},  {0x210B,0x210B,0x210B}, /* 210A */
  {0x210C,0x210C,0x210C},  {0x210D,0x210D,0x210D}, /* 210C */
  {0x210E,0x210E,0x210E},  {0x210F,0x210F,0x210F}, /* 210E */
  {0x2110,0x2110,0x2110},  {0x2111,0x2111,0x2111}, /* 2110 */
  {0x2112,0x2112,0x2112},  {0x2113,0x2113,0x2113}, /* 2112 */
  {0x2114,0x2114,0x2114},  {0x2115,0x2115,0x2115}, /* 2114 */
  {0x2116,0x2116,0x2116},  {0x2117,0x2117,0x2117}, /* 2116 */
  {0x2118,0x2118,0x2118},  {0x2119,0x2119,0x2119}, /* 2118 */
  {0x211A,0x211A,0x211A},  {0x211B,0x211B,0x211B}, /* 211A */
  {0x211C,0x211C,0x211C},  {0x211D,0x211D,0x211D}, /* 211C */
  {0x211E,0x211E,0x211E},  {0x211F,0x211F,0x211F}, /* 211E */
  {0x2120,0x2120,0x2120},  {0x2121,0x2121,0x2121}, /* 2120 */
  {0x2122,0x2122,0x2122},  {0x2123,0x2123,0x2123}, /* 2122 */
  {0x2124,0x2124,0x2124},  {0x2125,0x2125,0x2125}, /* 2124 */
  {0x2126,0x03C9,0x2126},  {0x2127,0x2127,0x2127}, /* 2126 */
  {0x2128,0x2128,0x2128},  {0x2129,0x2129,0x2129}, /* 2128 */
  {0x212A,0x006B,0x212A},  {0x212B,0x00E5,0x212B}, /* 212A */
  {0x212C,0x212C,0x212C},  {0x212D,0x212D,0x212D}, /* 212C */
  {0x212E,0x212E,0x212E},  {0x212F,0x212F,0x212F}, /* 212E */
  {0x2130,0x2130,0x2130},  {0x2131,0x2131,0x2131}, /* 2130 */
  {0x2132,0x214E,0x2132},  {0x2133,0x2133,0x2133}, /* 2132 */
  {0x2134,0x2134,0x2134},  {0x2135,0x2135,0x2135}, /* 2134 */
  {0x2136,0x2136,0x2136},  {0x2137,0x2137,0x2137}, /* 2136 */
  {0x2138,0x2138,0x2138},  {0x2139,0x2139,0x2139}, /* 2138 */
  {0x213A,0x213A,0x213A},  {0x213B,0x213B,0x213B}, /* 213A */
  {0x213C,0x213C,0x213C},  {0x213D,0x213D,0x213D}, /* 213C */
  {0x213E,0x213E,0x213E},  {0x213F,0x213F,0x213F}, /* 213E */
  {0x2140,0x2140,0x2140},  {0x2141,0x2141,0x2141}, /* 2140 */
  {0x2142,0x2142,0x2142},  {0x2143,0x2143,0x2143}, /* 2142 */
  {0x2144,0x2144,0x2144},  {0x2145,0x2145,0x2145}, /* 2144 */
  {0x2146,0x2146,0x2146},  {0x2147,0x2147,0x2147}, /* 2146 */
  {0x2148,0x2148,0x2148},  {0x2149,0x2149,0x2149}, /* 2148 */
  {0x214A,0x214A,0x214A},  {0x214B,0x214B,0x214B}, /* 214A */
  {0x214C,0x214C,0x214C},  {0x214D,0x214D,0x214D}, /* 214C */
  {0x2132,0x214E,0x2132},  {0x214F,0x214F,0x214F}, /* 214E */
  {0x2150,0x2150,0x2150},  {0x2151,0x2151,0x2151}, /* 2150 */
  {0x2152,0x2152,0x2152},  {0x2153,0x2153,0x2153}, /* 2152 */
  {0x2154,0x2154,0x2154},  {0x2155,0x2155,0x2155}, /* 2154 */
  {0x2156,0x2156,0x2156},  {0x2157,0x2157,0x2157}, /* 2156 */
  {0x2158,0x2158,0x2158},  {0x2159,0x2159,0x2159}, /* 2158 */
  {0x215A,0x215A,0x215A},  {0x215B,0x215B,0x215B}, /* 215A */
  {0x215C,0x215C,0x215C},  {0x215D,0x215D,0x215D}, /* 215C */
  {0x215E,0x215E,0x215E},  {0x215F,0x215F,0x215F}, /* 215E */
  {0x2160,0x2170,0x2160},  {0x2161,0x2171,0x2161}, /* 2160 */
  {0x2162,0x2172,0x2162},  {0x2163,0x2173,0x2163}, /* 2162 */
  {0x2164,0x2174,0x2164},  {0x2165,0x2175,0x2165}, /* 2164 */
  {0x2166,0x2176,0x2166},  {0x2167,0x2177,0x2167}, /* 2166 */
  {0x2168,0x2178,0x2168},  {0x2169,0x2179,0x2169}, /* 2168 */
  {0x216A,0x217A,0x216A},  {0x216B,0x217B,0x216B}, /* 216A */
  {0x216C,0x217C,0x216C},  {0x216D,0x217D,0x216D}, /* 216C */
  {0x216E,0x217E,0x216E},  {0x216F,0x217F,0x216F}, /* 216E */
  {0x2160,0x2170,0x2160},  {0x2161,0x2171,0x2161}, /* 2170 */
  {0x2162,0x2172,0x2162},  {0x2163,0x2173,0x2163}, /* 2172 */
  {0x2164,0x2174,0x2164},  {0x2165,0x2175,0x2165}, /* 2174 */
  {0x2166,0x2176,0x2166},  {0x2167,0x2177,0x2167}, /* 2176 */
  {0x2168,0x2178,0x2168},  {0x2169,0x2179,0x2169}, /* 2178 */
  {0x216A,0x217A,0x216A},  {0x216B,0x217B,0x216B}, /* 217A */
  {0x216C,0x217C,0x216C},  {0x216D,0x217D,0x216D}, /* 217C */
  {0x216E,0x217E,0x216E},  {0x216F,0x217F,0x216F}, /* 217E */
  {0x2180,0x2180,0x2180},  {0x2181,0x2181,0x2181}, /* 2180 */
  {0x2182,0x2182,0x2182},  {0x2183,0x2184,0x2183}, /* 2182 */
  {0x2183,0x2184,0x2183},  {0x2185,0x2185,0x2185}, /* 2184 */
  {0x2186,0x2186,0x2186},  {0x2187,0x2187,0x2187}, /* 2186 */
  {0x2188,0x2188,0x2188},  {0x2189,0x2189,0x2189}, /* 2188 */
  {0x218A,0x218A,0x218A},  {0x218B,0x218B,0x218B}, /* 218A */
  {0x218C,0x218C,0x218C},  {0x218D,0x218D,0x218D}, /* 218C */
  {0x218E,0x218E,0x218E},  {0x218F,0x218F,0x218F}, /* 218E */
  {0x2190,0x2190,0x2190},  {0x2191,0x2191,0x2191}, /* 2190 */
  {0x2192,0x2192,0x2192},  {0x2193,0x2193,0x2193}, /* 2192 */
  {0x2194,0x2194,0x2194},  {0x2195,0x2195,0x2195}, /* 2194 */
  {0x2196,0x2196,0x2196},  {0x2197,0x2197,0x2197}, /* 2196 */
  {0x2198,0x2198,0x2198},  {0x2199,0x2199,0x2199}, /* 2198 */
  {0x219A,0x219A,0x219A},  {0x219B,0x219B,0x219B}, /* 219A */
  {0x219C,0x219C,0x219C},  {0x219D,0x219D,0x219D}, /* 219C */
  {0x219E,0x219E,0x219E},  {0x219F,0x219F,0x219F}, /* 219E */
  {0x21A0,0x21A0,0x21A0},  {0x21A1,0x21A1,0x21A1}, /* 21A0 */
  {0x21A2,0x21A2,0x21A2},  {0x21A3,0x21A3,0x21A3}, /* 21A2 */
  {0x21A4,0x21A4,0x21A4},  {0x21A5,0x21A5,0x21A5}, /* 21A4 */
  {0x21A6,0x21A6,0x21A6},  {0x21A7,0x21A7,0x21A7}, /* 21A6 */
  {0x21A8,0x21A8,0x21A8},  {0x21A9,0x21A9,0x21A9}, /* 21A8 */
  {0x21AA,0x21AA,0x21AA},  {0x21AB,0x21AB,0x21AB}, /* 21AA */
  {0x21AC,0x21AC,0x21AC},  {0x21AD,0x21AD,0x21AD}, /* 21AC */
  {0x21AE,0x21AE,0x21AE},  {0x21AF,0x21AF,0x21AF}, /* 21AE */
  {0x21B0,0x21B0,0x21B0},  {0x21B1,0x21B1,0x21B1}, /* 21B0 */
  {0x21B2,0x21B2,0x21B2},  {0x21B3,0x21B3,0x21B3}, /* 21B2 */
  {0x21B4,0x21B4,0x21B4},  {0x21B5,0x21B5,0x21B5}, /* 21B4 */
  {0x21B6,0x21B6,0x21B6},  {0x21B7,0x21B7,0x21B7}, /* 21B6 */
  {0x21B8,0x21B8,0x21B8},  {0x21B9,0x21B9,0x21B9}, /* 21B8 */
  {0x21BA,0x21BA,0x21BA},  {0x21BB,0x21BB,0x21BB}, /* 21BA */
  {0x21BC,0x21BC,0x21BC},  {0x21BD,0x21BD,0x21BD}, /* 21BC */
  {0x21BE,0x21BE,0x21BE},  {0x21BF,0x21BF,0x21BF}, /* 21BE */
  {0x21C0,0x21C0,0x21C0},  {0x21C1,0x21C1,0x21C1}, /* 21C0 */
  {0x21C2,0x21C2,0x21C2},  {0x21C3,0x21C3,0x21C3}, /* 21C2 */
  {0x21C4,0x21C4,0x21C4},  {0x21C5,0x21C5,0x21C5}, /* 21C4 */
  {0x21C6,0x21C6,0x21C6},  {0x21C7,0x21C7,0x21C7}, /* 21C6 */
  {0x21C8,0x21C8,0x21C8},  {0x21C9,0x21C9,0x21C9}, /* 21C8 */
  {0x21CA,0x21CA,0x21CA},  {0x21CB,0x21CB,0x21CB}, /* 21CA */
  {0x21CC,0x21CC,0x21CC},  {0x21CD,0x21CD,0x21CD}, /* 21CC */
  {0x21CE,0x21CE,0x21CE},  {0x21CF,0x21CF,0x21CF}, /* 21CE */
  {0x21D0,0x21D0,0x21D0},  {0x21D1,0x21D1,0x21D1}, /* 21D0 */
  {0x21D2,0x21D2,0x21D2},  {0x21D3,0x21D3,0x21D3}, /* 21D2 */
  {0x21D4,0x21D4,0x21D4},  {0x21D5,0x21D5,0x21D5}, /* 21D4 */
  {0x21D6,0x21D6,0x21D6},  {0x21D7,0x21D7,0x21D7}, /* 21D6 */
  {0x21D8,0x21D8,0x21D8},  {0x21D9,0x21D9,0x21D9}, /* 21D8 */
  {0x21DA,0x21DA,0x21DA},  {0x21DB,0x21DB,0x21DB}, /* 21DA */
  {0x21DC,0x21DC,0x21DC},  {0x21DD,0x21DD,0x21DD}, /* 21DC */
  {0x21DE,0x21DE,0x21DE},  {0x21DF,0x21DF,0x21DF}, /* 21DE */
  {0x21E0,0x21E0,0x21E0},  {0x21E1,0x21E1,0x21E1}, /* 21E0 */
  {0x21E2,0x21E2,0x21E2},  {0x21E3,0x21E3,0x21E3}, /* 21E2 */
  {0x21E4,0x21E4,0x21E4},  {0x21E5,0x21E5,0x21E5}, /* 21E4 */
  {0x21E6,0x21E6,0x21E6},  {0x21E7,0x21E7,0x21E7}, /* 21E6 */
  {0x21E8,0x21E8,0x21E8},  {0x21E9,0x21E9,0x21E9}, /* 21E8 */
  {0x21EA,0x21EA,0x21EA},  {0x21EB,0x21EB,0x21EB}, /* 21EA */
  {0x21EC,0x21EC,0x21EC},  {0x21ED,0x21ED,0x21ED}, /* 21EC */
  {0x21EE,0x21EE,0x21EE},  {0x21EF,0x21EF,0x21EF}, /* 21EE */
  {0x21F0,0x21F0,0x21F0},  {0x21F1,0x21F1,0x21F1}, /* 21F0 */
  {0x21F2,0x21F2,0x21F2},  {0x21F3,0x21F3,0x21F3}, /* 21F2 */
  {0x21F4,0x21F4,0x21F4},  {0x21F5,0x21F5,0x21F5}, /* 21F4 */
  {0x21F6,0x21F6,0x21F6},  {0x21F7,0x21F7,0x21F7}, /* 21F6 */
  {0x21F8,0x21F8,0x21F8},  {0x21F9,0x21F9,0x21F9}, /* 21F8 */
  {0x21FA,0x21FA,0x21FA},  {0x21FB,0x21FB,0x21FB}, /* 21FA */
  {0x21FC,0x21FC,0x21FC},  {0x21FD,0x21FD,0x21FD}, /* 21FC */
  {0x21FE,0x21FE,0x21FE},  {0x21FF,0x21FF,0x21FF} /* 21FE */
};

static MY_UNICASE_CHARACTER u520p24[]={
  {0x2400,0x2400,0x2400},  {0x2401,0x2401,0x2401}, /* 2400 */
  {0x2402,0x2402,0x2402},  {0x2403,0x2403,0x2403}, /* 2402 */
  {0x2404,0x2404,0x2404},  {0x2405,0x2405,0x2405}, /* 2404 */
  {0x2406,0x2406,0x2406},  {0x2407,0x2407,0x2407}, /* 2406 */
  {0x2408,0x2408,0x2408},  {0x2409,0x2409,0x2409}, /* 2408 */
  {0x240A,0x240A,0x240A},  {0x240B,0x240B,0x240B}, /* 240A */
  {0x240C,0x240C,0x240C},  {0x240D,0x240D,0x240D}, /* 240C */
  {0x240E,0x240E,0x240E},  {0x240F,0x240F,0x240F}, /* 240E */
  {0x2410,0x2410,0x2410},  {0x2411,0x2411,0x2411}, /* 2410 */
  {0x2412,0x2412,0x2412},  {0x2413,0x2413,0x2413}, /* 2412 */
  {0x2414,0x2414,0x2414},  {0x2415,0x2415,0x2415}, /* 2414 */
  {0x2416,0x2416,0x2416},  {0x2417,0x2417,0x2417}, /* 2416 */
  {0x2418,0x2418,0x2418},  {0x2419,0x2419,0x2419}, /* 2418 */
  {0x241A,0x241A,0x241A},  {0x241B,0x241B,0x241B}, /* 241A */
  {0x241C,0x241C,0x241C},  {0x241D,0x241D,0x241D}, /* 241C */
  {0x241E,0x241E,0x241E},  {0x241F,0x241F,0x241F}, /* 241E */
  {0x2420,0x2420,0x2420},  {0x2421,0x2421,0x2421}, /* 2420 */
  {0x2422,0x2422,0x2422},  {0x2423,0x2423,0x2423}, /* 2422 */
  {0x2424,0x2424,0x2424},  {0x2425,0x2425,0x2425}, /* 2424 */
  {0x2426,0x2426,0x2426},  {0x2427,0x2427,0x2427}, /* 2426 */
  {0x2428,0x2428,0x2428},  {0x2429,0x2429,0x2429}, /* 2428 */
  {0x242A,0x242A,0x242A},  {0x242B,0x242B,0x242B}, /* 242A */
  {0x242C,0x242C,0x242C},  {0x242D,0x242D,0x242D}, /* 242C */
  {0x242E,0x242E,0x242E},  {0x242F,0x242F,0x242F}, /* 242E */
  {0x2430,0x2430,0x2430},  {0x2431,0x2431,0x2431}, /* 2430 */
  {0x2432,0x2432,0x2432},  {0x2433,0x2433,0x2433}, /* 2432 */
  {0x2434,0x2434,0x2434},  {0x2435,0x2435,0x2435}, /* 2434 */
  {0x2436,0x2436,0x2436},  {0x2437,0x2437,0x2437}, /* 2436 */
  {0x2438,0x2438,0x2438},  {0x2439,0x2439,0x2439}, /* 2438 */
  {0x243A,0x243A,0x243A},  {0x243B,0x243B,0x243B}, /* 243A */
  {0x243C,0x243C,0x243C},  {0x243D,0x243D,0x243D}, /* 243C */
  {0x243E,0x243E,0x243E},  {0x243F,0x243F,0x243F}, /* 243E */
  {0x2440,0x2440,0x2440},  {0x2441,0x2441,0x2441}, /* 2440 */
  {0x2442,0x2442,0x2442},  {0x2443,0x2443,0x2443}, /* 2442 */
  {0x2444,0x2444,0x2444},  {0x2445,0x2445,0x2445}, /* 2444 */
  {0x2446,0x2446,0x2446},  {0x2447,0x2447,0x2447}, /* 2446 */
  {0x2448,0x2448,0x2448},  {0x2449,0x2449,0x2449}, /* 2448 */
  {0x244A,0x244A,0x244A},  {0x244B,0x244B,0x244B}, /* 244A */
  {0x244C,0x244C,0x244C},  {0x244D,0x244D,0x244D}, /* 244C */
  {0x244E,0x244E,0x244E},  {0x244F,0x244F,0x244F}, /* 244E */
  {0x2450,0x2450,0x2450},  {0x2451,0x2451,0x2451}, /* 2450 */
  {0x2452,0x2452,0x2452},  {0x2453,0x2453,0x2453}, /* 2452 */
  {0x2454,0x2454,0x2454},  {0x2455,0x2455,0x2455}, /* 2454 */
  {0x2456,0x2456,0x2456},  {0x2457,0x2457,0x2457}, /* 2456 */
  {0x2458,0x2458,0x2458},  {0x2459,0x2459,0x2459}, /* 2458 */
  {0x245A,0x245A,0x245A},  {0x245B,0x245B,0x245B}, /* 245A */
  {0x245C,0x245C,0x245C},  {0x245D,0x245D,0x245D}, /* 245C */
  {0x245E,0x245E,0x245E},  {0x245F,0x245F,0x245F}, /* 245E */
  {0x2460,0x2460,0x2460},  {0x2461,0x2461,0x2461}, /* 2460 */
  {0x2462,0x2462,0x2462},  {0x2463,0x2463,0x2463}, /* 2462 */
  {0x2464,0x2464,0x2464},  {0x2465,0x2465,0x2465}, /* 2464 */
  {0x2466,0x2466,0x2466},  {0x2467,0x2467,0x2467}, /* 2466 */
  {0x2468,0x2468,0x2468},  {0x2469,0x2469,0x2469}, /* 2468 */
  {0x246A,0x246A,0x246A},  {0x246B,0x246B,0x246B}, /* 246A */
  {0x246C,0x246C,0x246C},  {0x246D,0x246D,0x246D}, /* 246C */
  {0x246E,0x246E,0x246E},  {0x246F,0x246F,0x246F}, /* 246E */
  {0x2470,0x2470,0x2470},  {0x2471,0x2471,0x2471}, /* 2470 */
  {0x2472,0x2472,0x2472},  {0x2473,0x2473,0x2473}, /* 2472 */
  {0x2474,0x2474,0x2474},  {0x2475,0x2475,0x2475}, /* 2474 */
  {0x2476,0x2476,0x2476},  {0x2477,0x2477,0x2477}, /* 2476 */
  {0x2478,0x2478,0x2478},  {0x2479,0x2479,0x2479}, /* 2478 */
  {0x247A,0x247A,0x247A},  {0x247B,0x247B,0x247B}, /* 247A */
  {0x247C,0x247C,0x247C},  {0x247D,0x247D,0x247D}, /* 247C */
  {0x247E,0x247E,0x247E},  {0x247F,0x247F,0x247F}, /* 247E */
  {0x2480,0x2480,0x2480},  {0x2481,0x2481,0x2481}, /* 2480 */
  {0x2482,0x2482,0x2482},  {0x2483,0x2483,0x2483}, /* 2482 */
  {0x2484,0x2484,0x2484},  {0x2485,0x2485,0x2485}, /* 2484 */
  {0x2486,0x2486,0x2486},  {0x2487,0x2487,0x2487}, /* 2486 */
  {0x2488,0x2488,0x2488},  {0x2489,0x2489,0x2489}, /* 2488 */
  {0x248A,0x248A,0x248A},  {0x248B,0x248B,0x248B}, /* 248A */
  {0x248C,0x248C,0x248C},  {0x248D,0x248D,0x248D}, /* 248C */
  {0x248E,0x248E,0x248E},  {0x248F,0x248F,0x248F}, /* 248E */
  {0x2490,0x2490,0x2490},  {0x2491,0x2491,0x2491}, /* 2490 */
  {0x2492,0x2492,0x2492},  {0x2493,0x2493,0x2493}, /* 2492 */
  {0x2494,0x2494,0x2494},  {0x2495,0x2495,0x2495}, /* 2494 */
  {0x2496,0x2496,0x2496},  {0x2497,0x2497,0x2497}, /* 2496 */
  {0x2498,0x2498,0x2498},  {0x2499,0x2499,0x2499}, /* 2498 */
  {0x249A,0x249A,0x249A},  {0x249B,0x249B,0x249B}, /* 249A */
  {0x249C,0x249C,0x249C},  {0x249D,0x249D,0x249D}, /* 249C */
  {0x249E,0x249E,0x249E},  {0x249F,0x249F,0x249F}, /* 249E */
  {0x24A0,0x24A0,0x24A0},  {0x24A1,0x24A1,0x24A1}, /* 24A0 */
  {0x24A2,0x24A2,0x24A2},  {0x24A3,0x24A3,0x24A3}, /* 24A2 */
  {0x24A4,0x24A4,0x24A4},  {0x24A5,0x24A5,0x24A5}, /* 24A4 */
  {0x24A6,0x24A6,0x24A6},  {0x24A7,0x24A7,0x24A7}, /* 24A6 */
  {0x24A8,0x24A8,0x24A8},  {0x24A9,0x24A9,0x24A9}, /* 24A8 */
  {0x24AA,0x24AA,0x24AA},  {0x24AB,0x24AB,0x24AB}, /* 24AA */
  {0x24AC,0x24AC,0x24AC},  {0x24AD,0x24AD,0x24AD}, /* 24AC */
  {0x24AE,0x24AE,0x24AE},  {0x24AF,0x24AF,0x24AF}, /* 24AE */
  {0x24B0,0x24B0,0x24B0},  {0x24B1,0x24B1,0x24B1}, /* 24B0 */
  {0x24B2,0x24B2,0x24B2},  {0x24B3,0x24B3,0x24B3}, /* 24B2 */
  {0x24B4,0x24B4,0x24B4},  {0x24B5,0x24B5,0x24B5}, /* 24B4 */
  {0x24B6,0x24D0,0x24B6},  {0x24B7,0x24D1,0x24B7}, /* 24B6 */
  {0x24B8,0x24D2,0x24B8},  {0x24B9,0x24D3,0x24B9}, /* 24B8 */
  {0x24BA,0x24D4,0x24BA},  {0x24BB,0x24D5,0x24BB}, /* 24BA */
  {0x24BC,0x24D6,0x24BC},  {0x24BD,0x24D7,0x24BD}, /* 24BC */
  {0x24BE,0x24D8,0x24BE},  {0x24BF,0x24D9,0x24BF}, /* 24BE */
  {0x24C0,0x24DA,0x24C0},  {0x24C1,0x24DB,0x24C1}, /* 24C0 */
  {0x24C2,0x24DC,0x24C2},  {0x24C3,0x24DD,0x24C3}, /* 24C2 */
  {0x24C4,0x24DE,0x24C4},  {0x24C5,0x24DF,0x24C5}, /* 24C4 */
  {0x24C6,0x24E0,0x24C6},  {0x24C7,0x24E1,0x24C7}, /* 24C6 */
  {0x24C8,0x24E2,0x24C8},  {0x24C9,0x24E3,0x24C9}, /* 24C8 */
  {0x24CA,0x24E4,0x24CA},  {0x24CB,0x24E5,0x24CB}, /* 24CA */
  {0x24CC,0x24E6,0x24CC},  {0x24CD,0x24E7,0x24CD}, /* 24CC */
  {0x24CE,0x24E8,0x24CE},  {0x24CF,0x24E9,0x24CF}, /* 24CE */
  {0x24B6,0x24D0,0x24B6},  {0x24B7,0x24D1,0x24B7}, /* 24D0 */
  {0x24B8,0x24D2,0x24B8},  {0x24B9,0x24D3,0x24B9}, /* 24D2 */
  {0x24BA,0x24D4,0x24BA},  {0x24BB,0x24D5,0x24BB}, /* 24D4 */
  {0x24BC,0x24D6,0x24BC},  {0x24BD,0x24D7,0x24BD}, /* 24D6 */
  {0x24BE,0x24D8,0x24BE},  {0x24BF,0x24D9,0x24BF}, /* 24D8 */
  {0x24C0,0x24DA,0x24C0},  {0x24C1,0x24DB,0x24C1}, /* 24DA */
  {0x24C2,0x24DC,0x24C2},  {0x24C3,0x24DD,0x24C3}, /* 24DC */
  {0x24C4,0x24DE,0x24C4},  {0x24C5,0x24DF,0x24C5}, /* 24DE */
  {0x24C6,0x24E0,0x24C6},  {0x24C7,0x24E1,0x24C7}, /* 24E0 */
  {0x24C8,0x24E2,0x24C8},  {0x24C9,0x24E3,0x24C9}, /* 24E2 */
  {0x24CA,0x24E4,0x24CA},  {0x24CB,0x24E5,0x24CB}, /* 24E4 */
  {0x24CC,0x24E6,0x24CC},  {0x24CD,0x24E7,0x24CD}, /* 24E6 */
  {0x24CE,0x24E8,0x24CE},  {0x24CF,0x24E9,0x24CF}, /* 24E8 */
  {0x24EA,0x24EA,0x24EA},  {0x24EB,0x24EB,0x24EB}, /* 24EA */
  {0x24EC,0x24EC,0x24EC},  {0x24ED,0x24ED,0x24ED}, /* 24EC */
  {0x24EE,0x24EE,0x24EE},  {0x24EF,0x24EF,0x24EF}, /* 24EE */
  {0x24F0,0x24F0,0x24F0},  {0x24F1,0x24F1,0x24F1}, /* 24F0 */
  {0x24F2,0x24F2,0x24F2},  {0x24F3,0x24F3,0x24F3}, /* 24F2 */
  {0x24F4,0x24F4,0x24F4},  {0x24F5,0x24F5,0x24F5}, /* 24F4 */
  {0x24F6,0x24F6,0x24F6},  {0x24F7,0x24F7,0x24F7}, /* 24F6 */
  {0x24F8,0x24F8,0x24F8},  {0x24F9,0x24F9,0x24F9}, /* 24F8 */
  {0x24FA,0x24FA,0x24FA},  {0x24FB,0x24FB,0x24FB}, /* 24FA */
  {0x24FC,0x24FC,0x24FC},  {0x24FD,0x24FD,0x24FD}, /* 24FC */
  {0x24FE,0x24FE,0x24FE},  {0x24FF,0x24FF,0x24FF} /* 24FE */
};

static MY_UNICASE_CHARACTER u520p2C[]={
  {0x2C00,0x2C30,0x2C00},  {0x2C01,0x2C31,0x2C01}, /* 2C00 */
  {0x2C02,0x2C32,0x2C02},  {0x2C03,0x2C33,0x2C03}, /* 2C02 */
  {0x2C04,0x2C34,0x2C04},  {0x2C05,0x2C35,0x2C05}, /* 2C04 */
  {0x2C06,0x2C36,0x2C06},  {0x2C07,0x2C37,0x2C07}, /* 2C06 */
  {0x2C08,0x2C38,0x2C08},  {0x2C09,0x2C39,0x2C09}, /* 2C08 */
  {0x2C0A,0x2C3A,0x2C0A},  {0x2C0B,0x2C3B,0x2C0B}, /* 2C0A */
  {0x2C0C,0x2C3C,0x2C0C},  {0x2C0D,0x2C3D,0x2C0D}, /* 2C0C */
  {0x2C0E,0x2C3E,0x2C0E},  {0x2C0F,0x2C3F,0x2C0F}, /* 2C0E */
  {0x2C10,0x2C40,0x2C10},  {0x2C11,0x2C41,0x2C11}, /* 2C10 */
  {0x2C12,0x2C42,0x2C12},  {0x2C13,0x2C43,0x2C13}, /* 2C12 */
  {0x2C14,0x2C44,0x2C14},  {0x2C15,0x2C45,0x2C15}, /* 2C14 */
  {0x2C16,0x2C46,0x2C16},  {0x2C17,0x2C47,0x2C17}, /* 2C16 */
  {0x2C18,0x2C48,0x2C18},  {0x2C19,0x2C49,0x2C19}, /* 2C18 */
  {0x2C1A,0x2C4A,0x2C1A},  {0x2C1B,0x2C4B,0x2C1B}, /* 2C1A */
  {0x2C1C,0x2C4C,0x2C1C},  {0x2C1D,0x2C4D,0x2C1D}, /* 2C1C */
  {0x2C1E,0x2C4E,0x2C1E},  {0x2C1F,0x2C4F,0x2C1F}, /* 2C1E */
  {0x2C20,0x2C50,0x2C20},  {0x2C21,0x2C51,0x2C21}, /* 2C20 */
  {0x2C22,0x2C52,0x2C22},  {0x2C23,0x2C53,0x2C23}, /* 2C22 */
  {0x2C24,0x2C54,0x2C24},  {0x2C25,0x2C55,0x2C25}, /* 2C24 */
  {0x2C26,0x2C56,0x2C26},  {0x2C27,0x2C57,0x2C27}, /* 2C26 */
  {0x2C28,0x2C58,0x2C28},  {0x2C29,0x2C59,0x2C29}, /* 2C28 */
  {0x2C2A,0x2C5A,0x2C2A},  {0x2C2B,0x2C5B,0x2C2B}, /* 2C2A */
  {0x2C2C,0x2C5C,0x2C2C},  {0x2C2D,0x2C5D,0x2C2D}, /* 2C2C */
  {0x2C2E,0x2C5E,0x2C2E},  {0x2C2F,0x2C2F,0x2C2F}, /* 2C2E */
  {0x2C00,0x2C30,0x2C00},  {0x2C01,0x2C31,0x2C01}, /* 2C30 */
  {0x2C02,0x2C32,0x2C02},  {0x2C03,0x2C33,0x2C03}, /* 2C32 */
  {0x2C04,0x2C34,0x2C04},  {0x2C05,0x2C35,0x2C05}, /* 2C34 */
  {0x2C06,0x2C36,0x2C06},  {0x2C07,0x2C37,0x2C07}, /* 2C36 */
  {0x2C08,0x2C38,0x2C08},  {0x2C09,0x2C39,0x2C09}, /* 2C38 */
  {0x2C0A,0x2C3A,0x2C0A},  {0x2C0B,0x2C3B,0x2C0B}, /* 2C3A */
  {0x2C0C,0x2C3C,0x2C0C},  {0x2C0D,0x2C3D,0x2C0D}, /* 2C3C */
  {0x2C0E,0x2C3E,0x2C0E},  {0x2C0F,0x2C3F,0x2C0F}, /* 2C3E */
  {0x2C10,0x2C40,0x2C10},  {0x2C11,0x2C41,0x2C11}, /* 2C40 */
  {0x2C12,0x2C42,0x2C12},  {0x2C13,0x2C43,0x2C13}, /* 2C42 */
  {0x2C14,0x2C44,0x2C14},  {0x2C15,0x2C45,0x2C15}, /* 2C44 */
  {0x2C16,0x2C46,0x2C16},  {0x2C17,0x2C47,0x2C17}, /* 2C46 */
  {0x2C18,0x2C48,0x2C18},  {0x2C19,0x2C49,0x2C19}, /* 2C48 */
  {0x2C1A,0x2C4A,0x2C1A},  {0x2C1B,0x2C4B,0x2C1B}, /* 2C4A */
  {0x2C1C,0x2C4C,0x2C1C},  {0x2C1D,0x2C4D,0x2C1D}, /* 2C4C */
  {0x2C1E,0x2C4E,0x2C1E},  {0x2C1F,0x2C4F,0x2C1F}, /* 2C4E */
  {0x2C20,0x2C50,0x2C20},  {0x2C21,0x2C51,0x2C21}, /* 2C50 */
  {0x2C22,0x2C52,0x2C22},  {0x2C23,0x2C53,0x2C23}, /* 2C52 */
  {0x2C24,0x2C54,0x2C24},  {0x2C25,0x2C55,0x2C25}, /* 2C54 */
  {0x2C26,0x2C56,0x2C26},  {0x2C27,0x2C57,0x2C27}, /* 2C56 */
  {0x2C28,0x2C58,0x2C28},  {0x2C29,0x2C59,0x2C29}, /* 2C58 */
  {0x2C2A,0x2C5A,0x2C2A},  {0x2C2B,0x2C5B,0x2C2B}, /* 2C5A */
  {0x2C2C,0x2C5C,0x2C2C},  {0x2C2D,0x2C5D,0x2C2D}, /* 2C5C */
  {0x2C2E,0x2C5E,0x2C2E},  {0x2C5F,0x2C5F,0x2C5F}, /* 2C5E */
  {0x2C60,0x2C61,0x2C60},  {0x2C60,0x2C61,0x2C60}, /* 2C60 */
  {0x2C62,0x026B,0x2C62},  {0x2C63,0x1D7D,0x2C63}, /* 2C62 */
  {0x2C64,0x027D,0x2C64},  {0x023A,0x2C65,0x023A}, /* 2C64 */
  {0x023E,0x2C66,0x023E},  {0x2C67,0x2C68,0x2C67}, /* 2C66 */
  {0x2C67,0x2C68,0x2C67},  {0x2C69,0x2C6A,0x2C69}, /* 2C68 */
  {0x2C69,0x2C6A,0x2C69},  {0x2C6B,0x2C6C,0x2C6B}, /* 2C6A */
  {0x2C6B,0x2C6C,0x2C6B},  {0x2C6D,0x0251,0x2C6D}, /* 2C6C */
  {0x2C6E,0x0271,0x2C6E},  {0x2C6F,0x0250,0x2C6F}, /* 2C6E */
  {0x2C70,0x0252,0x2C70},  {0x2C71,0x2C71,0x2C71}, /* 2C70 */
  {0x2C72,0x2C73,0x2C72},  {0x2C72,0x2C73,0x2C72}, /* 2C72 */
  {0x2C74,0x2C74,0x2C74},  {0x2C75,0x2C76,0x2C75}, /* 2C74 */
  {0x2C75,0x2C76,0x2C75},  {0x2C77,0x2C77,0x2C77}, /* 2C76 */
  {0x2C78,0x2C78,0x2C78},  {0x2C79,0x2C79,0x2C79}, /* 2C78 */
  {0x2C7A,0x2C7A,0x2C7A},  {0x2C7B,0x2C7B,0x2C7B}, /* 2C7A */
  {0x2C7C,0x2C7C,0x2C7C},  {0x2C7D,0x2C7D,0x2C7D}, /* 2C7C */
  {0x2C7E,0x023F,0x2C7E},  {0x2C7F,0x0240,0x2C7F}, /* 2C7E */
  {0x2C80,0x2C81,0x2C80},  {0x2C80,0x2C81,0x2C80}, /* 2C80 */
  {0x2C82,0x2C83,0x2C82},  {0x2C82,0x2C83,0x2C82}, /* 2C82 */
  {0x2C84,0x2C85,0x2C84},  {0x2C84,0x2C85,0x2C84}, /* 2C84 */
  {0x2C86,0x2C87,0x2C86},  {0x2C86,0x2C87,0x2C86}, /* 2C86 */
  {0x2C88,0x2C89,0x2C88},  {0x2C88,0x2C89,0x2C88}, /* 2C88 */
  {0x2C8A,0x2C8B,0x2C8A},  {0x2C8A,0x2C8B,0x2C8A}, /* 2C8A */
  {0x2C8C,0x2C8D,0x2C8C},  {0x2C8C,0x2C8D,0x2C8C}, /* 2C8C */
  {0x2C8E,0x2C8F,0x2C8E},  {0x2C8E,0x2C8F,0x2C8E}, /* 2C8E */
  {0x2C90,0x2C91,0x2C90},  {0x2C90,0x2C91,0x2C90}, /* 2C90 */
  {0x2C92,0x2C93,0x2C92},  {0x2C92,0x2C93,0x2C92}, /* 2C92 */
  {0x2C94,0x2C95,0x2C94},  {0x2C94,0x2C95,0x2C94}, /* 2C94 */
  {0x2C96,0x2C97,0x2C96},  {0x2C96,0x2C97,0x2C96}, /* 2C96 */
  {0x2C98,0x2C99,0x2C98},  {0x2C98,0x2C99,0x2C98}, /* 2C98 */
  {0x2C9A,0x2C9B,0x2C9A},  {0x2C9A,0x2C9B,0x2C9A}, /* 2C9A */
  {0x2C9C,0x2C9D,0x2C9C},  {0x2C9C,0x2C9D,0x2C9C}, /* 2C9C */
  {0x2C9E,0x2C9F,0x2C9E},  {0x2C9E,0x2C9F,0x2C9E}, /* 2C9E */
  {0x2CA0,0x2CA1,0x2CA0},  {0x2CA0,0x2CA1,0x2CA0}, /* 2CA0 */
  {0x2CA2,0x2CA3,0x2CA2},  {0x2CA2,0x2CA3,0x2CA2}, /* 2CA2 */
  {0x2CA4,0x2CA5,0x2CA4},  {0x2CA4,0x2CA5,0x2CA4}, /* 2CA4 */
  {0x2CA6,0x2CA7,0x2CA6},  {0x2CA6,0x2CA7,0x2CA6}, /* 2CA6 */
  {0x2CA8,0x2CA9,0x2CA8},  {0x2CA8,0x2CA9,0x2CA8}, /* 2CA8 */
  {0x2CAA,0x2CAB,0x2CAA},  {0x2CAA,0x2CAB,0x2CAA}, /* 2CAA */
  {0x2CAC,0x2CAD,0x2CAC},  {0x2CAC,0x2CAD,0x2CAC}, /* 2CAC */
  {0x2CAE,0x2CAF,0x2CAE},  {0x2CAE,0x2CAF,0x2CAE}, /* 2CAE */
  {0x2CB0,0x2CB1,0x2CB0},  {0x2CB0,0x2CB1,0x2CB0}, /* 2CB0 */
  {0x2CB2,0x2CB3,0x2CB2},  {0x2CB2,0x2CB3,0x2CB2}, /* 2CB2 */
  {0x2CB4,0x2CB5,0x2CB4},  {0x2CB4,0x2CB5,0x2CB4}, /* 2CB4 */
  {0x2CB6,0x2CB7,0x2CB6},  {0x2CB6,0x2CB7,0x2CB6}, /* 2CB6 */
  {0x2CB8,0x2CB9,0x2CB8},  {0x2CB8,0x2CB9,0x2CB8}, /* 2CB8 */
  {0x2CBA,0x2CBB,0x2CBA},  {0x2CBA,0x2CBB,0x2CBA}, /* 2CBA */
  {0x2CBC,0x2CBD,0x2CBC},  {0x2CBC,0x2CBD,0x2CBC}, /* 2CBC */
  {0x2CBE,0x2CBF,0x2CBE},  {0x2CBE,0x2CBF,0x2CBE}, /* 2CBE */
  {0x2CC0,0x2CC1,0x2CC0},  {0x2CC0,0x2CC1,0x2CC0}, /* 2CC0 */
  {0x2CC2,0x2CC3,0x2CC2},  {0x2CC2,0x2CC3,0x2CC2}, /* 2CC2 */
  {0x2CC4,0x2CC5,0x2CC4},  {0x2CC4,0x2CC5,0x2CC4}, /* 2CC4 */
  {0x2CC6,0x2CC7,0x2CC6},  {0x2CC6,0x2CC7,0x2CC6}, /* 2CC6 */
  {0x2CC8,0x2CC9,0x2CC8},  {0x2CC8,0x2CC9,0x2CC8}, /* 2CC8 */
  {0x2CCA,0x2CCB,0x2CCA},  {0x2CCA,0x2CCB,0x2CCA}, /* 2CCA */
  {0x2CCC,0x2CCD,0x2CCC},  {0x2CCC,0x2CCD,0x2CCC}, /* 2CCC */
  {0x2CCE,0x2CCF,0x2CCE},  {0x2CCE,0x2CCF,0x2CCE}, /* 2CCE */
  {0x2CD0,0x2CD1,0x2CD0},  {0x2CD0,0x2CD1,0x2CD0}, /* 2CD0 */
  {0x2CD2,0x2CD3,0x2CD2},  {0x2CD2,0x2CD3,0x2CD2}, /* 2CD2 */
  {0x2CD4,0x2CD5,0x2CD4},  {0x2CD4,0x2CD5,0x2CD4}, /* 2CD4 */
  {0x2CD6,0x2CD7,0x2CD6},  {0x2CD6,0x2CD7,0x2CD6}, /* 2CD6 */
  {0x2CD8,0x2CD9,0x2CD8},  {0x2CD8,0x2CD9,0x2CD8}, /* 2CD8 */
  {0x2CDA,0x2CDB,0x2CDA},  {0x2CDA,0x2CDB,0x2CDA}, /* 2CDA */
  {0x2CDC,0x2CDD,0x2CDC},  {0x2CDC,0x2CDD,0x2CDC}, /* 2CDC */
  {0x2CDE,0x2CDF,0x2CDE},  {0x2CDE,0x2CDF,0x2CDE}, /* 2CDE */
  {0x2CE0,0x2CE1,0x2CE0},  {0x2CE0,0x2CE1,0x2CE0}, /* 2CE0 */
  {0x2CE2,0x2CE3,0x2CE2},  {0x2CE2,0x2CE3,0x2CE2}, /* 2CE2 */
  {0x2CE4,0x2CE4,0x2CE4},  {0x2CE5,0x2CE5,0x2CE5}, /* 2CE4 */
  {0x2CE6,0x2CE6,0x2CE6},  {0x2CE7,0x2CE7,0x2CE7}, /* 2CE6 */
  {0x2CE8,0x2CE8,0x2CE8},  {0x2CE9,0x2CE9,0x2CE9}, /* 2CE8 */
  {0x2CEA,0x2CEA,0x2CEA},  {0x2CEB,0x2CEC,0x2CEB}, /* 2CEA */
  {0x2CEB,0x2CEC,0x2CEB},  {0x2CED,0x2CEE,0x2CED}, /* 2CEC */
  {0x2CED,0x2CEE,0x2CED},  {0x2CEF,0x2CEF,0x2CEF}, /* 2CEE */
  {0x2CF0,0x2CF0,0x2CF0},  {0x2CF1,0x2CF1,0x2CF1}, /* 2CF0 */
  {0x2CF2,0x2CF2,0x2CF2},  {0x2CF3,0x2CF3,0x2CF3}, /* 2CF2 */
  {0x2CF4,0x2CF4,0x2CF4},  {0x2CF5,0x2CF5,0x2CF5}, /* 2CF4 */
  {0x2CF6,0x2CF6,0x2CF6},  {0x2CF7,0x2CF7,0x2CF7}, /* 2CF6 */
  {0x2CF8,0x2CF8,0x2CF8},  {0x2CF9,0x2CF9,0x2CF9}, /* 2CF8 */
  {0x2CFA,0x2CFA,0x2CFA},  {0x2CFB,0x2CFB,0x2CFB}, /* 2CFA */
  {0x2CFC,0x2CFC,0x2CFC},  {0x2CFD,0x2CFD,0x2CFD}, /* 2CFC */
  {0x2CFE,0x2CFE,0x2CFE},  {0x2CFF,0x2CFF,0x2CFF} /* 2CFE */
};

static MY_UNICASE_CHARACTER u520p2D[]={
  {0x10A0,0x2D00,0x10A0},  {0x10A1,0x2D01,0x10A1}, /* 2D00 */
  {0x10A2,0x2D02,0x10A2},  {0x10A3,0x2D03,0x10A3}, /* 2D02 */
  {0x10A4,0x2D04,0x10A4},  {0x10A5,0x2D05,0x10A5}, /* 2D04 */
  {0x10A6,0x2D06,0x10A6},  {0x10A7,0x2D07,0x10A7}, /* 2D06 */
  {0x10A8,0x2D08,0x10A8},  {0x10A9,0x2D09,0x10A9}, /* 2D08 */
  {0x10AA,0x2D0A,0x10AA},  {0x10AB,0x2D0B,0x10AB}, /* 2D0A */
  {0x10AC,0x2D0C,0x10AC},  {0x10AD,0x2D0D,0x10AD}, /* 2D0C */
  {0x10AE,0x2D0E,0x10AE},  {0x10AF,0x2D0F,0x10AF}, /* 2D0E */
  {0x10B0,0x2D10,0x10B0},  {0x10B1,0x2D11,0x10B1}, /* 2D10 */
  {0x10B2,0x2D12,0x10B2},  {0x10B3,0x2D13,0x10B3}, /* 2D12 */
  {0x10B4,0x2D14,0x10B4},  {0x10B5,0x2D15,0x10B5}, /* 2D14 */
  {0x10B6,0x2D16,0x10B6},  {0x10B7,0x2D17,0x10B7}, /* 2D16 */
  {0x10B8,0x2D18,0x10B8},  {0x10B9,0x2D19,0x10B9}, /* 2D18 */
  {0x10BA,0x2D1A,0x10BA},  {0x10BB,0x2D1B,0x10BB}, /* 2D1A */
  {0x10BC,0x2D1C,0x10BC},  {0x10BD,0x2D1D,0x10BD}, /* 2D1C */
  {0x10BE,0x2D1E,0x10BE},  {0x10BF,0x2D1F,0x10BF}, /* 2D1E */
  {0x10C0,0x2D20,0x10C0},  {0x10C1,0x2D21,0x10C1}, /* 2D20 */
  {0x10C2,0x2D22,0x10C2},  {0x10C3,0x2D23,0x10C3}, /* 2D22 */
  {0x10C4,0x2D24,0x10C4},  {0x10C5,0x2D25,0x10C5}, /* 2D24 */
  {0x2D26,0x2D26,0x2D26},  {0x2D27,0x2D27,0x2D27}, /* 2D26 */
  {0x2D28,0x2D28,0x2D28},  {0x2D29,0x2D29,0x2D29}, /* 2D28 */
  {0x2D2A,0x2D2A,0x2D2A},  {0x2D2B,0x2D2B,0x2D2B}, /* 2D2A */
  {0x2D2C,0x2D2C,0x2D2C},  {0x2D2D,0x2D2D,0x2D2D}, /* 2D2C */
  {0x2D2E,0x2D2E,0x2D2E},  {0x2D2F,0x2D2F,0x2D2F}, /* 2D2E */
  {0x2D30,0x2D30,0x2D30},  {0x2D31,0x2D31,0x2D31}, /* 2D30 */
  {0x2D32,0x2D32,0x2D32},  {0x2D33,0x2D33,0x2D33}, /* 2D32 */
  {0x2D34,0x2D34,0x2D34},  {0x2D35,0x2D35,0x2D35}, /* 2D34 */
  {0x2D36,0x2D36,0x2D36},  {0x2D37,0x2D37,0x2D37}, /* 2D36 */
  {0x2D38,0x2D38,0x2D38},  {0x2D39,0x2D39,0x2D39}, /* 2D38 */
  {0x2D3A,0x2D3A,0x2D3A},  {0x2D3B,0x2D3B,0x2D3B}, /* 2D3A */
  {0x2D3C,0x2D3C,0x2D3C},  {0x2D3D,0x2D3D,0x2D3D}, /* 2D3C */
  {0x2D3E,0x2D3E,0x2D3E},  {0x2D3F,0x2D3F,0x2D3F}, /* 2D3E */
  {0x2D40,0x2D40,0x2D40},  {0x2D41,0x2D41,0x2D41}, /* 2D40 */
  {0x2D42,0x2D42,0x2D42},  {0x2D43,0x2D43,0x2D43}, /* 2D42 */
  {0x2D44,0x2D44,0x2D44},  {0x2D45,0x2D45,0x2D45}, /* 2D44 */
  {0x2D46,0x2D46,0x2D46},  {0x2D47,0x2D47,0x2D47}, /* 2D46 */
  {0x2D48,0x2D48,0x2D48},  {0x2D49,0x2D49,0x2D49}, /* 2D48 */
  {0x2D4A,0x2D4A,0x2D4A},  {0x2D4B,0x2D4B,0x2D4B}, /* 2D4A */
  {0x2D4C,0x2D4C,0x2D4C},  {0x2D4D,0x2D4D,0x2D4D}, /* 2D4C */
  {0x2D4E,0x2D4E,0x2D4E},  {0x2D4F,0x2D4F,0x2D4F}, /* 2D4E */
  {0x2D50,0x2D50,0x2D50},  {0x2D51,0x2D51,0x2D51}, /* 2D50 */
  {0x2D52,0x2D52,0x2D52},  {0x2D53,0x2D53,0x2D53}, /* 2D52 */
  {0x2D54,0x2D54,0x2D54},  {0x2D55,0x2D55,0x2D55}, /* 2D54 */
  {0x2D56,0x2D56,0x2D56},  {0x2D57,0x2D57,0x2D57}, /* 2D56 */
  {0x2D58,0x2D58,0x2D58},  {0x2D59,0x2D59,0x2D59}, /* 2D58 */
  {0x2D5A,0x2D5A,0x2D5A},  {0x2D5B,0x2D5B,0x2D5B}, /* 2D5A */
  {0x2D5C,0x2D5C,0x2D5C},  {0x2D5D,0x2D5D,0x2D5D}, /* 2D5C */
  {0x2D5E,0x2D5E,0x2D5E},  {0x2D5F,0x2D5F,0x2D5F}, /* 2D5E */
  {0x2D60,0x2D60,0x2D60},  {0x2D61,0x2D61,0x2D61}, /* 2D60 */
  {0x2D62,0x2D62,0x2D62},  {0x2D63,0x2D63,0x2D63}, /* 2D62 */
  {0x2D64,0x2D64,0x2D64},  {0x2D65,0x2D65,0x2D65}, /* 2D64 */
  {0x2D66,0x2D66,0x2D66},  {0x2D67,0x2D67,0x2D67}, /* 2D66 */
  {0x2D68,0x2D68,0x2D68},  {0x2D69,0x2D69,0x2D69}, /* 2D68 */
  {0x2D6A,0x2D6A,0x2D6A},  {0x2D6B,0x2D6B,0x2D6B}, /* 2D6A */
  {0x2D6C,0x2D6C,0x2D6C},  {0x2D6D,0x2D6D,0x2D6D}, /* 2D6C */
  {0x2D6E,0x2D6E,0x2D6E},  {0x2D6F,0x2D6F,0x2D6F}, /* 2D6E */
  {0x2D70,0x2D70,0x2D70},  {0x2D71,0x2D71,0x2D71}, /* 2D70 */
  {0x2D72,0x2D72,0x2D72},  {0x2D73,0x2D73,0x2D73}, /* 2D72 */
  {0x2D74,0x2D74,0x2D74},  {0x2D75,0x2D75,0x2D75}, /* 2D74 */
  {0x2D76,0x2D76,0x2D76},  {0x2D77,0x2D77,0x2D77}, /* 2D76 */
  {0x2D78,0x2D78,0x2D78},  {0x2D79,0x2D79,0x2D79}, /* 2D78 */
  {0x2D7A,0x2D7A,0x2D7A},  {0x2D7B,0x2D7B,0x2D7B}, /* 2D7A */
  {0x2D7C,0x2D7C,0x2D7C},  {0x2D7D,0x2D7D,0x2D7D}, /* 2D7C */
  {0x2D7E,0x2D7E,0x2D7E},  {0x2D7F,0x2D7F,0x2D7F}, /* 2D7E */
  {0x2D80,0x2D80,0x2D80},  {0x2D81,0x2D81,0x2D81}, /* 2D80 */
  {0x2D82,0x2D82,0x2D82},  {0x2D83,0x2D83,0x2D83}, /* 2D82 */
  {0x2D84,0x2D84,0x2D84},  {0x2D85,0x2D85,0x2D85}, /* 2D84 */
  {0x2D86,0x2D86,0x2D86},  {0x2D87,0x2D87,0x2D87}, /* 2D86 */
  {0x2D88,0x2D88,0x2D88},  {0x2D89,0x2D89,0x2D89}, /* 2D88 */
  {0x2D8A,0x2D8A,0x2D8A},  {0x2D8B,0x2D8B,0x2D8B}, /* 2D8A */
  {0x2D8C,0x2D8C,0x2D8C},  {0x2D8D,0x2D8D,0x2D8D}, /* 2D8C */
  {0x2D8E,0x2D8E,0x2D8E},  {0x2D8F,0x2D8F,0x2D8F}, /* 2D8E */
  {0x2D90,0x2D90,0x2D90},  {0x2D91,0x2D91,0x2D91}, /* 2D90 */
  {0x2D92,0x2D92,0x2D92},  {0x2D93,0x2D93,0x2D93}, /* 2D92 */
  {0x2D94,0x2D94,0x2D94},  {0x2D95,0x2D95,0x2D95}, /* 2D94 */
  {0x2D96,0x2D96,0x2D96},  {0x2D97,0x2D97,0x2D97}, /* 2D96 */
  {0x2D98,0x2D98,0x2D98},  {0x2D99,0x2D99,0x2D99}, /* 2D98 */
  {0x2D9A,0x2D9A,0x2D9A},  {0x2D9B,0x2D9B,0x2D9B}, /* 2D9A */
  {0x2D9C,0x2D9C,0x2D9C},  {0x2D9D,0x2D9D,0x2D9D}, /* 2D9C */
  {0x2D9E,0x2D9E,0x2D9E},  {0x2D9F,0x2D9F,0x2D9F}, /* 2D9E */
  {0x2DA0,0x2DA0,0x2DA0},  {0x2DA1,0x2DA1,0x2DA1}, /* 2DA0 */
  {0x2DA2,0x2DA2,0x2DA2},  {0x2DA3,0x2DA3,0x2DA3}, /* 2DA2 */
  {0x2DA4,0x2DA4,0x2DA4},  {0x2DA5,0x2DA5,0x2DA5}, /* 2DA4 */
  {0x2DA6,0x2DA6,0x2DA6},  {0x2DA7,0x2DA7,0x2DA7}, /* 2DA6 */
  {0x2DA8,0x2DA8,0x2DA8},  {0x2DA9,0x2DA9,0x2DA9}, /* 2DA8 */
  {0x2DAA,0x2DAA,0x2DAA},  {0x2DAB,0x2DAB,0x2DAB}, /* 2DAA */
  {0x2DAC,0x2DAC,0x2DAC},  {0x2DAD,0x2DAD,0x2DAD}, /* 2DAC */
  {0x2DAE,0x2DAE,0x2DAE},  {0x2DAF,0x2DAF,0x2DAF}, /* 2DAE */
  {0x2DB0,0x2DB0,0x2DB0},  {0x2DB1,0x2DB1,0x2DB1}, /* 2DB0 */
  {0x2DB2,0x2DB2,0x2DB2},  {0x2DB3,0x2DB3,0x2DB3}, /* 2DB2 */
  {0x2DB4,0x2DB4,0x2DB4},  {0x2DB5,0x2DB5,0x2DB5}, /* 2DB4 */
  {0x2DB6,0x2DB6,0x2DB6},  {0x2DB7,0x2DB7,0x2DB7}, /* 2DB6 */
  {0x2DB8,0x2DB8,0x2DB8},  {0x2DB9,0x2DB9,0x2DB9}, /* 2DB8 */
  {0x2DBA,0x2DBA,0x2DBA},  {0x2DBB,0x2DBB,0x2DBB}, /* 2DBA */
  {0x2DBC,0x2DBC,0x2DBC},  {0x2DBD,0x2DBD,0x2DBD}, /* 2DBC */
  {0x2DBE,0x2DBE,0x2DBE},  {0x2DBF,0x2DBF,0x2DBF}, /* 2DBE */
  {0x2DC0,0x2DC0,0x2DC0},  {0x2DC1,0x2DC1,0x2DC1}, /* 2DC0 */
  {0x2DC2,0x2DC2,0x2DC2},  {0x2DC3,0x2DC3,0x2DC3}, /* 2DC2 */
  {0x2DC4,0x2DC4,0x2DC4},  {0x2DC5,0x2DC5,0x2DC5}, /* 2DC4 */
  {0x2DC6,0x2DC6,0x2DC6},  {0x2DC7,0x2DC7,0x2DC7}, /* 2DC6 */
  {0x2DC8,0x2DC8,0x2DC8},  {0x2DC9,0x2DC9,0x2DC9}, /* 2DC8 */
  {0x2DCA,0x2DCA,0x2DCA},  {0x2DCB,0x2DCB,0x2DCB}, /* 2DCA */
  {0x2DCC,0x2DCC,0x2DCC},  {0x2DCD,0x2DCD,0x2DCD}, /* 2DCC */
  {0x2DCE,0x2DCE,0x2DCE},  {0x2DCF,0x2DCF,0x2DCF}, /* 2DCE */
  {0x2DD0,0x2DD0,0x2DD0},  {0x2DD1,0x2DD1,0x2DD1}, /* 2DD0 */
  {0x2DD2,0x2DD2,0x2DD2},  {0x2DD3,0x2DD3,0x2DD3}, /* 2DD2 */
  {0x2DD4,0x2DD4,0x2DD4},  {0x2DD5,0x2DD5,0x2DD5}, /* 2DD4 */
  {0x2DD6,0x2DD6,0x2DD6},  {0x2DD7,0x2DD7,0x2DD7}, /* 2DD6 */
  {0x2DD8,0x2DD8,0x2DD8},  {0x2DD9,0x2DD9,0x2DD9}, /* 2DD8 */
  {0x2DDA,0x2DDA,0x2DDA},  {0x2DDB,0x2DDB,0x2DDB}, /* 2DDA */
  {0x2DDC,0x2DDC,0x2DDC},  {0x2DDD,0x2DDD,0x2DDD}, /* 2DDC */
  {0x2DDE,0x2DDE,0x2DDE},  {0x2DDF,0x2DDF,0x2DDF}, /* 2DDE */
  {0x2DE0,0x2DE0,0x2DE0},  {0x2DE1,0x2DE1,0x2DE1}, /* 2DE0 */
  {0x2DE2,0x2DE2,0x2DE2},  {0x2DE3,0x2DE3,0x2DE3}, /* 2DE2 */
  {0x2DE4,0x2DE4,0x2DE4},  {0x2DE5,0x2DE5,0x2DE5}, /* 2DE4 */
  {0x2DE6,0x2DE6,0x2DE6},  {0x2DE7,0x2DE7,0x2DE7}, /* 2DE6 */
  {0x2DE8,0x2DE8,0x2DE8},  {0x2DE9,0x2DE9,0x2DE9}, /* 2DE8 */
  {0x2DEA,0x2DEA,0x2DEA},  {0x2DEB,0x2DEB,0x2DEB}, /* 2DEA */
  {0x2DEC,0x2DEC,0x2DEC},  {0x2DED,0x2DED,0x2DED}, /* 2DEC */
  {0x2DEE,0x2DEE,0x2DEE},  {0x2DEF,0x2DEF,0x2DEF}, /* 2DEE */
  {0x2DF0,0x2DF0,0x2DF0},  {0x2DF1,0x2DF1,0x2DF1}, /* 2DF0 */
  {0x2DF2,0x2DF2,0x2DF2},  {0x2DF3,0x2DF3,0x2DF3}, /* 2DF2 */
  {0x2DF4,0x2DF4,0x2DF4},  {0x2DF5,0x2DF5,0x2DF5}, /* 2DF4 */
  {0x2DF6,0x2DF6,0x2DF6},  {0x2DF7,0x2DF7,0x2DF7}, /* 2DF6 */
  {0x2DF8,0x2DF8,0x2DF8},  {0x2DF9,0x2DF9,0x2DF9}, /* 2DF8 */
  {0x2DFA,0x2DFA,0x2DFA},  {0x2DFB,0x2DFB,0x2DFB}, /* 2DFA */
  {0x2DFC,0x2DFC,0x2DFC},  {0x2DFD,0x2DFD,0x2DFD}, /* 2DFC */
  {0x2DFE,0x2DFE,0x2DFE},  {0x2DFF,0x2DFF,0x2DFF} /* 2DFE */
};

static MY_UNICASE_CHARACTER u520pA6[]={
  {0xA600,0xA600,0xA600},  {0xA601,0xA601,0xA601}, /* A600 */
  {0xA602,0xA602,0xA602},  {0xA603,0xA603,0xA603}, /* A602 */
  {0xA604,0xA604,0xA604},  {0xA605,0xA605,0xA605}, /* A604 */
  {0xA606,0xA606,0xA606},  {0xA607,0xA607,0xA607}, /* A606 */
  {0xA608,0xA608,0xA608},  {0xA609,0xA609,0xA609}, /* A608 */
  {0xA60A,0xA60A,0xA60A},  {0xA60B,0xA60B,0xA60B}, /* A60A */
  {0xA60C,0xA60C,0xA60C},  {0xA60D,0xA60D,0xA60D}, /* A60C */
  {0xA60E,0xA60E,0xA60E},  {0xA60F,0xA60F,0xA60F}, /* A60E */
  {0xA610,0xA610,0xA610},  {0xA611,0xA611,0xA611}, /* A610 */
  {0xA612,0xA612,0xA612},  {0xA613,0xA613,0xA613}, /* A612 */
  {0xA614,0xA614,0xA614},  {0xA615,0xA615,0xA615}, /* A614 */
  {0xA616,0xA616,0xA616},  {0xA617,0xA617,0xA617}, /* A616 */
  {0xA618,0xA618,0xA618},  {0xA619,0xA619,0xA619}, /* A618 */
  {0xA61A,0xA61A,0xA61A},  {0xA61B,0xA61B,0xA61B}, /* A61A */
  {0xA61C,0xA61C,0xA61C},  {0xA61D,0xA61D,0xA61D}, /* A61C */
  {0xA61E,0xA61E,0xA61E},  {0xA61F,0xA61F,0xA61F}, /* A61E */
  {0xA620,0xA620,0xA620},  {0xA621,0xA621,0xA621}, /* A620 */
  {0xA622,0xA622,0xA622},  {0xA623,0xA623,0xA623}, /* A622 */
  {0xA624,0xA624,0xA624},  {0xA625,0xA625,0xA625}, /* A624 */
  {0xA626,0xA626,0xA626},  {0xA627,0xA627,0xA627}, /* A626 */
  {0xA628,0xA628,0xA628},  {0xA629,0xA629,0xA629}, /* A628 */
  {0xA62A,0xA62A,0xA62A},  {0xA62B,0xA62B,0xA62B}, /* A62A */
  {0xA62C,0xA62C,0xA62C},  {0xA62D,0xA62D,0xA62D}, /* A62C */
  {0xA62E,0xA62E,0xA62E},  {0xA62F,0xA62F,0xA62F}, /* A62E */
  {0xA630,0xA630,0xA630},  {0xA631,0xA631,0xA631}, /* A630 */
  {0xA632,0xA632,0xA632},  {0xA633,0xA633,0xA633}, /* A632 */
  {0xA634,0xA634,0xA634},  {0xA635,0xA635,0xA635}, /* A634 */
  {0xA636,0xA636,0xA636},  {0xA637,0xA637,0xA637}, /* A636 */
  {0xA638,0xA638,0xA638},  {0xA639,0xA639,0xA639}, /* A638 */
  {0xA63A,0xA63A,0xA63A},  {0xA63B,0xA63B,0xA63B}, /* A63A */
  {0xA63C,0xA63C,0xA63C},  {0xA63D,0xA63D,0xA63D}, /* A63C */
  {0xA63E,0xA63E,0xA63E},  {0xA63F,0xA63F,0xA63F}, /* A63E */
  {0xA640,0xA641,0xA640},  {0xA640,0xA641,0xA640}, /* A640 */
  {0xA642,0xA643,0xA642},  {0xA642,0xA643,0xA642}, /* A642 */
  {0xA644,0xA645,0xA644},  {0xA644,0xA645,0xA644}, /* A644 */
  {0xA646,0xA647,0xA646},  {0xA646,0xA647,0xA646}, /* A646 */
  {0xA648,0xA649,0xA648},  {0xA648,0xA649,0xA648}, /* A648 */
  {0xA64A,0xA64B,0xA64A},  {0xA64A,0xA64B,0xA64A}, /* A64A */
  {0xA64C,0xA64D,0xA64C},  {0xA64C,0xA64D,0xA64C}, /* A64C */
  {0xA64E,0xA64F,0xA64E},  {0xA64E,0xA64F,0xA64E}, /* A64E */
  {0xA650,0xA651,0xA650},  {0xA650,0xA651,0xA650}, /* A650 */
  {0xA652,0xA653,0xA652},  {0xA652,0xA653,0xA652}, /* A652 */
  {0xA654,0xA655,0xA654},  {0xA654,0xA655,0xA654}, /* A654 */
  {0xA656,0xA657,0xA656},  {0xA656,0xA657,0xA656}, /* A656 */
  {0xA658,0xA659,0xA658},  {0xA658,0xA659,0xA658}, /* A658 */
  {0xA65A,0xA65B,0xA65A},  {0xA65A,0xA65B,0xA65A}, /* A65A */
  {0xA65C,0xA65D,0xA65C},  {0xA65C,0xA65D,0xA65C}, /* A65C */
  {0xA65E,0xA65F,0xA65E},  {0xA65E,0xA65F,0xA65E}, /* A65E */
  {0xA660,0xA660,0xA660},  {0xA661,0xA661,0xA661}, /* A660 */
  {0xA662,0xA663,0xA662},  {0xA662,0xA663,0xA662}, /* A662 */
  {0xA664,0xA665,0xA664},  {0xA664,0xA665,0xA664}, /* A664 */
  {0xA666,0xA667,0xA666},  {0xA666,0xA667,0xA666}, /* A666 */
  {0xA668,0xA669,0xA668},  {0xA668,0xA669,0xA668}, /* A668 */
  {0xA66A,0xA66B,0xA66A},  {0xA66A,0xA66B,0xA66A}, /* A66A */
  {0xA66C,0xA66D,0xA66C},  {0xA66C,0xA66D,0xA66C}, /* A66C */
  {0xA66E,0xA66E,0xA66E},  {0xA66F,0xA66F,0xA66F}, /* A66E */
  {0xA670,0xA670,0xA670},  {0xA671,0xA671,0xA671}, /* A670 */
  {0xA672,0xA672,0xA672},  {0xA673,0xA673,0xA673}, /* A672 */
  {0xA674,0xA674,0xA674},  {0xA675,0xA675,0xA675}, /* A674 */
  {0xA676,0xA676,0xA676},  {0xA677,0xA677,0xA677}, /* A676 */
  {0xA678,0xA678,0xA678},  {0xA679,0xA679,0xA679}, /* A678 */
  {0xA67A,0xA67A,0xA67A},  {0xA67B,0xA67B,0xA67B}, /* A67A */
  {0xA67C,0xA67C,0xA67C},  {0xA67D,0xA67D,0xA67D}, /* A67C */
  {0xA67E,0xA67E,0xA67E},  {0xA67F,0xA67F,0xA67F}, /* A67E */
  {0xA680,0xA681,0xA680},  {0xA680,0xA681,0xA680}, /* A680 */
  {0xA682,0xA683,0xA682},  {0xA682,0xA683,0xA682}, /* A682 */
  {0xA684,0xA685,0xA684},  {0xA684,0xA685,0xA684}, /* A684 */
  {0xA686,0xA687,0xA686},  {0xA686,0xA687,0xA686}, /* A686 */
  {0xA688,0xA689,0xA688},  {0xA688,0xA689,0xA688}, /* A688 */
  {0xA68A,0xA68B,0xA68A},  {0xA68A,0xA68B,0xA68A}, /* A68A */
  {0xA68C,0xA68D,0xA68C},  {0xA68C,0xA68D,0xA68C}, /* A68C */
  {0xA68E,0xA68F,0xA68E},  {0xA68E,0xA68F,0xA68E}, /* A68E */
  {0xA690,0xA691,0xA690},  {0xA690,0xA691,0xA690}, /* A690 */
  {0xA692,0xA693,0xA692},  {0xA692,0xA693,0xA692}, /* A692 */
  {0xA694,0xA695,0xA694},  {0xA694,0xA695,0xA694}, /* A694 */
  {0xA696,0xA697,0xA696},  {0xA696,0xA697,0xA696}, /* A696 */
  {0xA698,0xA698,0xA698},  {0xA699,0xA699,0xA699}, /* A698 */
  {0xA69A,0xA69A,0xA69A},  {0xA69B,0xA69B,0xA69B}, /* A69A */
  {0xA69C,0xA69C,0xA69C},  {0xA69D,0xA69D,0xA69D}, /* A69C */
  {0xA69E,0xA69E,0xA69E},  {0xA69F,0xA69F,0xA69F}, /* A69E */
  {0xA6A0,0xA6A0,0xA6A0},  {0xA6A1,0xA6A1,0xA6A1}, /* A6A0 */
  {0xA6A2,0xA6A2,0xA6A2},  {0xA6A3,0xA6A3,0xA6A3}, /* A6A2 */
  {0xA6A4,0xA6A4,0xA6A4},  {0xA6A5,0xA6A5,0xA6A5}, /* A6A4 */
  {0xA6A6,0xA6A6,0xA6A6},  {0xA6A7,0xA6A7,0xA6A7}, /* A6A6 */
  {0xA6A8,0xA6A8,0xA6A8},  {0xA6A9,0xA6A9,0xA6A9}, /* A6A8 */
  {0xA6AA,0xA6AA,0xA6AA},  {0xA6AB,0xA6AB,0xA6AB}, /* A6AA */
  {0xA6AC,0xA6AC,0xA6AC},  {0xA6AD,0xA6AD,0xA6AD}, /* A6AC */
  {0xA6AE,0xA6AE,0xA6AE},  {0xA6AF,0xA6AF,0xA6AF}, /* A6AE */
  {0xA6B0,0xA6B0,0xA6B0},  {0xA6B1,0xA6B1,0xA6B1}, /* A6B0 */
  {0xA6B2,0xA6B2,0xA6B2},  {0xA6B3,0xA6B3,0xA6B3}, /* A6B2 */
  {0xA6B4,0xA6B4,0xA6B4},  {0xA6B5,0xA6B5,0xA6B5}, /* A6B4 */
  {0xA6B6,0xA6B6,0xA6B6},  {0xA6B7,0xA6B7,0xA6B7}, /* A6B6 */
  {0xA6B8,0xA6B8,0xA6B8},  {0xA6B9,0xA6B9,0xA6B9}, /* A6B8 */
  {0xA6BA,0xA6BA,0xA6BA},  {0xA6BB,0xA6BB,0xA6BB}, /* A6BA */
  {0xA6BC,0xA6BC,0xA6BC},  {0xA6BD,0xA6BD,0xA6BD}, /* A6BC */
  {0xA6BE,0xA6BE,0xA6BE},  {0xA6BF,0xA6BF,0xA6BF}, /* A6BE */
  {0xA6C0,0xA6C0,0xA6C0},  {0xA6C1,0xA6C1,0xA6C1}, /* A6C0 */
  {0xA6C2,0xA6C2,0xA6C2},  {0xA6C3,0xA6C3,0xA6C3}, /* A6C2 */
  {0xA6C4,0xA6C4,0xA6C4},  {0xA6C5,0xA6C5,0xA6C5}, /* A6C4 */
  {0xA6C6,0xA6C6,0xA6C6},  {0xA6C7,0xA6C7,0xA6C7}, /* A6C6 */
  {0xA6C8,0xA6C8,0xA6C8},  {0xA6C9,0xA6C9,0xA6C9}, /* A6C8 */
  {0xA6CA,0xA6CA,0xA6CA},  {0xA6CB,0xA6CB,0xA6CB}, /* A6CA */
  {0xA6CC,0xA6CC,0xA6CC},  {0xA6CD,0xA6CD,0xA6CD}, /* A6CC */
  {0xA6CE,0xA6CE,0xA6CE},  {0xA6CF,0xA6CF,0xA6CF}, /* A6CE */
  {0xA6D0,0xA6D0,0xA6D0},  {0xA6D1,0xA6D1,0xA6D1}, /* A6D0 */
  {0xA6D2,0xA6D2,0xA6D2},  {0xA6D3,0xA6D3,0xA6D3}, /* A6D2 */
  {0xA6D4,0xA6D4,0xA6D4},  {0xA6D5,0xA6D5,0xA6D5}, /* A6D4 */
  {0xA6D6,0xA6D6,0xA6D6},  {0xA6D7,0xA6D7,0xA6D7}, /* A6D6 */
  {0xA6D8,0xA6D8,0xA6D8},  {0xA6D9,0xA6D9,0xA6D9}, /* A6D8 */
  {0xA6DA,0xA6DA,0xA6DA},  {0xA6DB,0xA6DB,0xA6DB}, /* A6DA */
  {0xA6DC,0xA6DC,0xA6DC},  {0xA6DD,0xA6DD,0xA6DD}, /* A6DC */
  {0xA6DE,0xA6DE,0xA6DE},  {0xA6DF,0xA6DF,0xA6DF}, /* A6DE */
  {0xA6E0,0xA6E0,0xA6E0},  {0xA6E1,0xA6E1,0xA6E1}, /* A6E0 */
  {0xA6E2,0xA6E2,0xA6E2},  {0xA6E3,0xA6E3,0xA6E3}, /* A6E2 */
  {0xA6E4,0xA6E4,0xA6E4},  {0xA6E5,0xA6E5,0xA6E5}, /* A6E4 */
  {0xA6E6,0xA6E6,0xA6E6},  {0xA6E7,0xA6E7,0xA6E7}, /* A6E6 */
  {0xA6E8,0xA6E8,0xA6E8},  {0xA6E9,0xA6E9,0xA6E9}, /* A6E8 */
  {0xA6EA,0xA6EA,0xA6EA},  {0xA6EB,0xA6EB,0xA6EB}, /* A6EA */
  {0xA6EC,0xA6EC,0xA6EC},  {0xA6ED,0xA6ED,0xA6ED}, /* A6EC */
  {0xA6EE,0xA6EE,0xA6EE},  {0xA6EF,0xA6EF,0xA6EF}, /* A6EE */
  {0xA6F0,0xA6F0,0xA6F0},  {0xA6F1,0xA6F1,0xA6F1}, /* A6F0 */
  {0xA6F2,0xA6F2,0xA6F2},  {0xA6F3,0xA6F3,0xA6F3}, /* A6F2 */
  {0xA6F4,0xA6F4,0xA6F4},  {0xA6F5,0xA6F5,0xA6F5}, /* A6F4 */
  {0xA6F6,0xA6F6,0xA6F6},  {0xA6F7,0xA6F7,0xA6F7}, /* A6F6 */
  {0xA6F8,0xA6F8,0xA6F8},  {0xA6F9,0xA6F9,0xA6F9}, /* A6F8 */
  {0xA6FA,0xA6FA,0xA6FA},  {0xA6FB,0xA6FB,0xA6FB}, /* A6FA */
  {0xA6FC,0xA6FC,0xA6FC},  {0xA6FD,0xA6FD,0xA6FD}, /* A6FC */
  {0xA6FE,0xA6FE,0xA6FE},  {0xA6FF,0xA6FF,0xA6FF} /* A6FE */
};

static MY_UNICASE_CHARACTER u520pA7[]={
  {0xA700,0xA700,0xA700},  {0xA701,0xA701,0xA701}, /* A700 */
  {0xA702,0xA702,0xA702},  {0xA703,0xA703,0xA703}, /* A702 */
  {0xA704,0xA704,0xA704},  {0xA705,0xA705,0xA705}, /* A704 */
  {0xA706,0xA706,0xA706},  {0xA707,0xA707,0xA707}, /* A706 */
  {0xA708,0xA708,0xA708},  {0xA709,0xA709,0xA709}, /* A708 */
  {0xA70A,0xA70A,0xA70A},  {0xA70B,0xA70B,0xA70B}, /* A70A */
  {0xA70C,0xA70C,0xA70C},  {0xA70D,0xA70D,0xA70D}, /* A70C */
  {0xA70E,0xA70E,0xA70E},  {0xA70F,0xA70F,0xA70F}, /* A70E */
  {0xA710,0xA710,0xA710},  {0xA711,0xA711,0xA711}, /* A710 */
  {0xA712,0xA712,0xA712},  {0xA713,0xA713,0xA713}, /* A712 */
  {0xA714,0xA714,0xA714},  {0xA715,0xA715,0xA715}, /* A714 */
  {0xA716,0xA716,0xA716},  {0xA717,0xA717,0xA717}, /* A716 */
  {0xA718,0xA718,0xA718},  {0xA719,0xA719,0xA719}, /* A718 */
  {0xA71A,0xA71A,0xA71A},  {0xA71B,0xA71B,0xA71B}, /* A71A */
  {0xA71C,0xA71C,0xA71C},  {0xA71D,0xA71D,0xA71D}, /* A71C */
  {0xA71E,0xA71E,0xA71E},  {0xA71F,0xA71F,0xA71F}, /* A71E */
  {0xA720,0xA720,0xA720},  {0xA721,0xA721,0xA721}, /* A720 */
  {0xA722,0xA723,0xA722},  {0xA722,0xA723,0xA722}, /* A722 */
  {0xA724,0xA725,0xA724},  {0xA724,0xA725,0xA724}, /* A724 */
  {0xA726,0xA727,0xA726},  {0xA726,0xA727,0xA726}, /* A726 */
  {0xA728,0xA729,0xA728},  {0xA728,0xA729,0xA728}, /* A728 */
  {0xA72A,0xA72B,0xA72A},  {0xA72A,0xA72B,0xA72A}, /* A72A */
  {0xA72C,0xA72D,0xA72C},  {0xA72C,0xA72D,0xA72C}, /* A72C */
  {0xA72E,0xA72F,0xA72E},  {0xA72E,0xA72F,0xA72E}, /* A72E */
  {0xA730,0xA730,0xA730},  {0xA731,0xA731,0xA731}, /* A730 */
  {0xA732,0xA733,0xA732},  {0xA732,0xA733,0xA732}, /* A732 */
  {0xA734,0xA735,0xA734},  {0xA734,0xA735,0xA734}, /* A734 */
  {0xA736,0xA737,0xA736},  {0xA736,0xA737,0xA736}, /* A736 */
  {0xA738,0xA739,0xA738},  {0xA738,0xA739,0xA738}, /* A738 */
  {0xA73A,0xA73B,0xA73A},  {0xA73A,0xA73B,0xA73A}, /* A73A */
  {0xA73C,0xA73D,0xA73C},  {0xA73C,0xA73D,0xA73C}, /* A73C */
  {0xA73E,0xA73F,0xA73E},  {0xA73E,0xA73F,0xA73E}, /* A73E */
  {0xA740,0xA741,0xA740},  {0xA740,0xA741,0xA740}, /* A740 */
  {0xA742,0xA743,0xA742},  {0xA742,0xA743,0xA742}, /* A742 */
  {0xA744,0xA745,0xA744},  {0xA744,0xA745,0xA744}, /* A744 */
  {0xA746,0xA747,0xA746},  {0xA746,0xA747,0xA746}, /* A746 */
  {0xA748,0xA749,0xA748},  {0xA748,0xA749,0xA748}, /* A748 */
  {0xA74A,0xA74B,0xA74A},  {0xA74A,0xA74B,0xA74A}, /* A74A */
  {0xA74C,0xA74D,0xA74C},  {0xA74C,0xA74D,0xA74C}, /* A74C */
  {0xA74E,0xA74F,0xA74E},  {0xA74E,0xA74F,0xA74E}, /* A74E */
  {0xA750,0xA751,0xA750},  {0xA750,0xA751,0xA750}, /* A750 */
  {0xA752,0xA753,0xA752},  {0xA752,0xA753,0xA752}, /* A752 */
  {0xA754,0xA755,0xA754},  {0xA754,0xA755,0xA754}, /* A754 */
  {0xA756,0xA757,0xA756},  {0xA756,0xA757,0xA756}, /* A756 */
  {0xA758,0xA759,0xA758},  {0xA758,0xA759,0xA758}, /* A758 */
  {0xA75A,0xA75B,0xA75A},  {0xA75A,0xA75B,0xA75A}, /* A75A */
  {0xA75C,0xA75D,0xA75C},  {0xA75C,0xA75D,0xA75C}, /* A75C */
  {0xA75E,0xA75F,0xA75E},  {0xA75E,0xA75F,0xA75E}, /* A75E */
  {0xA760,0xA761,0xA760},  {0xA760,0xA761,0xA760}, /* A760 */
  {0xA762,0xA763,0xA762},  {0xA762,0xA763,0xA762}, /* A762 */
  {0xA764,0xA765,0xA764},  {0xA764,0xA765,0xA764}, /* A764 */
  {0xA766,0xA767,0xA766},  {0xA766,0xA767,0xA766}, /* A766 */
  {0xA768,0xA769,0xA768},  {0xA768,0xA769,0xA768}, /* A768 */
  {0xA76A,0xA76B,0xA76A},  {0xA76A,0xA76B,0xA76A}, /* A76A */
  {0xA76C,0xA76D,0xA76C},  {0xA76C,0xA76D,0xA76C}, /* A76C */
  {0xA76E,0xA76F,0xA76E},  {0xA76E,0xA76F,0xA76E}, /* A76E */
  {0xA770,0xA770,0xA770},  {0xA771,0xA771,0xA771}, /* A770 */
  {0xA772,0xA772,0xA772},  {0xA773,0xA773,0xA773}, /* A772 */
  {0xA774,0xA774,0xA774},  {0xA775,0xA775,0xA775}, /* A774 */
  {0xA776,0xA776,0xA776},  {0xA777,0xA777,0xA777}, /* A776 */
  {0xA778,0xA778,0xA778},  {0xA779,0xA77A,0xA779}, /* A778 */
  {0xA779,0xA77A,0xA779},  {0xA77B,0xA77C,0xA77B}, /* A77A */
  {0xA77B,0xA77C,0xA77B},  {0xA77D,0x1D79,0xA77D}, /* A77C */
  {0xA77E,0xA77F,0xA77E},  {0xA77E,0xA77F,0xA77E}, /* A77E */
  {0xA780,0xA781,0xA780},  {0xA780,0xA781,0xA780}, /* A780 */
  {0xA782,0xA783,0xA782},  {0xA782,0xA783,0xA782}, /* A782 */
  {0xA784,0xA785,0xA784},  {0xA784,0xA785,0xA784}, /* A784 */
  {0xA786,0xA787,0xA786},  {0xA786,0xA787,0xA786}, /* A786 */
  {0xA788,0xA788,0xA788},  {0xA789,0xA789,0xA789}, /* A788 */
  {0xA78A,0xA78A,0xA78A},  {0xA78B,0xA78C,0xA78B}, /* A78A */
  {0xA78B,0xA78C,0xA78B},  {0xA78D,0xA78D,0xA78D}, /* A78C */
  {0xA78E,0xA78E,0xA78E},  {0xA78F,0xA78F,0xA78F}, /* A78E */
  {0xA790,0xA790,0xA790},  {0xA791,0xA791,0xA791}, /* A790 */
  {0xA792,0xA792,0xA792},  {0xA793,0xA793,0xA793}, /* A792 */
  {0xA794,0xA794,0xA794},  {0xA795,0xA795,0xA795}, /* A794 */
  {0xA796,0xA796,0xA796},  {0xA797,0xA797,0xA797}, /* A796 */
  {0xA798,0xA798,0xA798},  {0xA799,0xA799,0xA799}, /* A798 */
  {0xA79A,0xA79A,0xA79A},  {0xA79B,0xA79B,0xA79B}, /* A79A */
  {0xA79C,0xA79C,0xA79C},  {0xA79D,0xA79D,0xA79D}, /* A79C */
  {0xA79E,0xA79E,0xA79E},  {0xA79F,0xA79F,0xA79F}, /* A79E */
  {0xA7A0,0xA7A0,0xA7A0},  {0xA7A1,0xA7A1,0xA7A1}, /* A7A0 */
  {0xA7A2,0xA7A2,0xA7A2},  {0xA7A3,0xA7A3,0xA7A3}, /* A7A2 */
  {0xA7A4,0xA7A4,0xA7A4},  {0xA7A5,0xA7A5,0xA7A5}, /* A7A4 */
  {0xA7A6,0xA7A6,0xA7A6},  {0xA7A7,0xA7A7,0xA7A7}, /* A7A6 */
  {0xA7A8,0xA7A8,0xA7A8},  {0xA7A9,0xA7A9,0xA7A9}, /* A7A8 */
  {0xA7AA,0xA7AA,0xA7AA},  {0xA7AB,0xA7AB,0xA7AB}, /* A7AA */
  {0xA7AC,0xA7AC,0xA7AC},  {0xA7AD,0xA7AD,0xA7AD}, /* A7AC */
  {0xA7AE,0xA7AE,0xA7AE},  {0xA7AF,0xA7AF,0xA7AF}, /* A7AE */
  {0xA7B0,0xA7B0,0xA7B0},  {0xA7B1,0xA7B1,0xA7B1}, /* A7B0 */
  {0xA7B2,0xA7B2,0xA7B2},  {0xA7B3,0xA7B3,0xA7B3}, /* A7B2 */
  {0xA7B4,0xA7B4,0xA7B4},  {0xA7B5,0xA7B5,0xA7B5}, /* A7B4 */
  {0xA7B6,0xA7B6,0xA7B6},  {0xA7B7,0xA7B7,0xA7B7}, /* A7B6 */
  {0xA7B8,0xA7B8,0xA7B8},  {0xA7B9,0xA7B9,0xA7B9}, /* A7B8 */
  {0xA7BA,0xA7BA,0xA7BA},  {0xA7BB,0xA7BB,0xA7BB}, /* A7BA */
  {0xA7BC,0xA7BC,0xA7BC},  {0xA7BD,0xA7BD,0xA7BD}, /* A7BC */
  {0xA7BE,0xA7BE,0xA7BE},  {0xA7BF,0xA7BF,0xA7BF}, /* A7BE */
  {0xA7C0,0xA7C0,0xA7C0},  {0xA7C1,0xA7C1,0xA7C1}, /* A7C0 */
  {0xA7C2,0xA7C2,0xA7C2},  {0xA7C3,0xA7C3,0xA7C3}, /* A7C2 */
  {0xA7C4,0xA7C4,0xA7C4},  {0xA7C5,0xA7C5,0xA7C5}, /* A7C4 */
  {0xA7C6,0xA7C6,0xA7C6},  {0xA7C7,0xA7C7,0xA7C7}, /* A7C6 */
  {0xA7C8,0xA7C8,0xA7C8},  {0xA7C9,0xA7C9,0xA7C9}, /* A7C8 */
  {0xA7CA,0xA7CA,0xA7CA},  {0xA7CB,0xA7CB,0xA7CB}, /* A7CA */
  {0xA7CC,0xA7CC,0xA7CC},  {0xA7CD,0xA7CD,0xA7CD}, /* A7CC */
  {0xA7CE,0xA7CE,0xA7CE},  {0xA7CF,0xA7CF,0xA7CF}, /* A7CE */
  {0xA7D0,0xA7D0,0xA7D0},  {0xA7D1,0xA7D1,0xA7D1}, /* A7D0 */
  {0xA7D2,0xA7D2,0xA7D2},  {0xA7D3,0xA7D3,0xA7D3}, /* A7D2 */
  {0xA7D4,0xA7D4,0xA7D4},  {0xA7D5,0xA7D5,0xA7D5}, /* A7D4 */
  {0xA7D6,0xA7D6,0xA7D6},  {0xA7D7,0xA7D7,0xA7D7}, /* A7D6 */
  {0xA7D8,0xA7D8,0xA7D8},  {0xA7D9,0xA7D9,0xA7D9}, /* A7D8 */
  {0xA7DA,0xA7DA,0xA7DA},  {0xA7DB,0xA7DB,0xA7DB}, /* A7DA */
  {0xA7DC,0xA7DC,0xA7DC},  {0xA7DD,0xA7DD,0xA7DD}, /* A7DC */
  {0xA7DE,0xA7DE,0xA7DE},  {0xA7DF,0xA7DF,0xA7DF}, /* A7DE */
  {0xA7E0,0xA7E0,0xA7E0},  {0xA7E1,0xA7E1,0xA7E1}, /* A7E0 */
  {0xA7E2,0xA7E2,0xA7E2},  {0xA7E3,0xA7E3,0xA7E3}, /* A7E2 */
  {0xA7E4,0xA7E4,0xA7E4},  {0xA7E5,0xA7E5,0xA7E5}, /* A7E4 */
  {0xA7E6,0xA7E6,0xA7E6},  {0xA7E7,0xA7E7,0xA7E7}, /* A7E6 */
  {0xA7E8,0xA7E8,0xA7E8},  {0xA7E9,0xA7E9,0xA7E9}, /* A7E8 */
  {0xA7EA,0xA7EA,0xA7EA},  {0xA7EB,0xA7EB,0xA7EB}, /* A7EA */
  {0xA7EC,0xA7EC,0xA7EC},  {0xA7ED,0xA7ED,0xA7ED}, /* A7EC */
  {0xA7EE,0xA7EE,0xA7EE},  {0xA7EF,0xA7EF,0xA7EF}, /* A7EE */
  {0xA7F0,0xA7F0,0xA7F0},  {0xA7F1,0xA7F1,0xA7F1}, /* A7F0 */
  {0xA7F2,0xA7F2,0xA7F2},  {0xA7F3,0xA7F3,0xA7F3}, /* A7F2 */
  {0xA7F4,0xA7F4,0xA7F4},  {0xA7F5,0xA7F5,0xA7F5}, /* A7F4 */
  {0xA7F6,0xA7F6,0xA7F6},  {0xA7F7,0xA7F7,0xA7F7}, /* A7F6 */
  {0xA7F8,0xA7F8,0xA7F8},  {0xA7F9,0xA7F9,0xA7F9}, /* A7F8 */
  {0xA7FA,0xA7FA,0xA7FA},  {0xA7FB,0xA7FB,0xA7FB}, /* A7FA */
  {0xA7FC,0xA7FC,0xA7FC},  {0xA7FD,0xA7FD,0xA7FD}, /* A7FC */
  {0xA7FE,0xA7FE,0xA7FE},  {0xA7FF,0xA7FF,0xA7FF} /* A7FE */
};

static MY_UNICASE_CHARACTER u520pFF[]={
  {0xFF00,0xFF00,0xFF00},  {0xFF01,0xFF01,0xFF01}, /* FF00 */
  {0xFF02,0xFF02,0xFF02},  {0xFF03,0xFF03,0xFF03}, /* FF02 */
  {0xFF04,0xFF04,0xFF04},  {0xFF05,0xFF05,0xFF05}, /* FF04 */
  {0xFF06,0xFF06,0xFF06},  {0xFF07,0xFF07,0xFF07}, /* FF06 */
  {0xFF08,0xFF08,0xFF08},  {0xFF09,0xFF09,0xFF09}, /* FF08 */
  {0xFF0A,0xFF0A,0xFF0A},  {0xFF0B,0xFF0B,0xFF0B}, /* FF0A */
  {0xFF0C,0xFF0C,0xFF0C},  {0xFF0D,0xFF0D,0xFF0D}, /* FF0C */
  {0xFF0E,0xFF0E,0xFF0E},  {0xFF0F,0xFF0F,0xFF0F}, /* FF0E */
  {0xFF10,0xFF10,0xFF10},  {0xFF11,0xFF11,0xFF11}, /* FF10 */
  {0xFF12,0xFF12,0xFF12},  {0xFF13,0xFF13,0xFF13}, /* FF12 */
  {0xFF14,0xFF14,0xFF14},  {0xFF15,0xFF15,0xFF15}, /* FF14 */
  {0xFF16,0xFF16,0xFF16},  {0xFF17,0xFF17,0xFF17}, /* FF16 */
  {0xFF18,0xFF18,0xFF18},  {0xFF19,0xFF19,0xFF19}, /* FF18 */
  {0xFF1A,0xFF1A,0xFF1A},  {0xFF1B,0xFF1B,0xFF1B}, /* FF1A */
  {0xFF1C,0xFF1C,0xFF1C},  {0xFF1D,0xFF1D,0xFF1D}, /* FF1C */
  {0xFF1E,0xFF1E,0xFF1E},  {0xFF1F,0xFF1F,0xFF1F}, /* FF1E */
  {0xFF20,0xFF20,0xFF20},  {0xFF21,0xFF41,0xFF21}, /* FF20 */
  {0xFF22,0xFF42,0xFF22},  {0xFF23,0xFF43,0xFF23}, /* FF22 */
  {0xFF24,0xFF44,0xFF24},  {0xFF25,0xFF45,0xFF25}, /* FF24 */
  {0xFF26,0xFF46,0xFF26},  {0xFF27,0xFF47,0xFF27}, /* FF26 */
  {0xFF28,0xFF48,0xFF28},  {0xFF29,0xFF49,0xFF29}, /* FF28 */
  {0xFF2A,0xFF4A,0xFF2A},  {0xFF2B,0xFF4B,0xFF2B}, /* FF2A */
  {0xFF2C,0xFF4C,0xFF2C},  {0xFF2D,0xFF4D,0xFF2D}, /* FF2C */
  {0xFF2E,0xFF4E,0xFF2E},  {0xFF2F,0xFF4F,0xFF2F}, /* FF2E */
  {0xFF30,0xFF50,0xFF30},  {0xFF31,0xFF51,0xFF31}, /* FF30 */
  {0xFF32,0xFF52,0xFF32},  {0xFF33,0xFF53,0xFF33}, /* FF32 */
  {0xFF34,0xFF54,0xFF34},  {0xFF35,0xFF55,0xFF35}, /* FF34 */
  {0xFF36,0xFF56,0xFF36},  {0xFF37,0xFF57,0xFF37}, /* FF36 */
  {0xFF38,0xFF58,0xFF38},  {0xFF39,0xFF59,0xFF39}, /* FF38 */
  {0xFF3A,0xFF5A,0xFF3A},  {0xFF3B,0xFF3B,0xFF3B}, /* FF3A */
  {0xFF3C,0xFF3C,0xFF3C},  {0xFF3D,0xFF3D,0xFF3D}, /* FF3C */
  {0xFF3E,0xFF3E,0xFF3E},  {0xFF3F,0xFF3F,0xFF3F}, /* FF3E */
  {0xFF40,0xFF40,0xFF40},  {0xFF21,0xFF41,0xFF21}, /* FF40 */
  {0xFF22,0xFF42,0xFF22},  {0xFF23,0xFF43,0xFF23}, /* FF42 */
  {0xFF24,0xFF44,0xFF24},  {0xFF25,0xFF45,0xFF25}, /* FF44 */
  {0xFF26,0xFF46,0xFF26},  {0xFF27,0xFF47,0xFF27}, /* FF46 */
  {0xFF28,0xFF48,0xFF28},  {0xFF29,0xFF49,0xFF29}, /* FF48 */
  {0xFF2A,0xFF4A,0xFF2A},  {0xFF2B,0xFF4B,0xFF2B}, /* FF4A */
  {0xFF2C,0xFF4C,0xFF2C},  {0xFF2D,0xFF4D,0xFF2D}, /* FF4C */
  {0xFF2E,0xFF4E,0xFF2E},  {0xFF2F,0xFF4F,0xFF2F}, /* FF4E */
  {0xFF30,0xFF50,0xFF30},  {0xFF31,0xFF51,0xFF31}, /* FF50 */
  {0xFF32,0xFF52,0xFF32},  {0xFF33,0xFF53,0xFF33}, /* FF52 */
  {0xFF34,0xFF54,0xFF34},  {0xFF35,0xFF55,0xFF35}, /* FF54 */
  {0xFF36,0xFF56,0xFF36},  {0xFF37,0xFF57,0xFF37}, /* FF56 */
  {0xFF38,0xFF58,0xFF38},  {0xFF39,0xFF59,0xFF39}, /* FF58 */
  {0xFF3A,0xFF5A,0xFF3A},  {0xFF5B,0xFF5B,0xFF5B}, /* FF5A */
  {0xFF5C,0xFF5C,0xFF5C},  {0xFF5D,0xFF5D,0xFF5D}, /* FF5C */
  {0xFF5E,0xFF5E,0xFF5E},  {0xFF5F,0xFF5F,0xFF5F}, /* FF5E */
  {0xFF60,0xFF60,0xFF60},  {0xFF61,0xFF61,0xFF61}, /* FF60 */
  {0xFF62,0xFF62,0xFF62},  {0xFF63,0xFF63,0xFF63}, /* FF62 */
  {0xFF64,0xFF64,0xFF64},  {0xFF65,0xFF65,0xFF65}, /* FF64 */
  {0xFF66,0xFF66,0xFF66},  {0xFF67,0xFF67,0xFF67}, /* FF66 */
  {0xFF68,0xFF68,0xFF68},  {0xFF69,0xFF69,0xFF69}, /* FF68 */
  {0xFF6A,0xFF6A,0xFF6A},  {0xFF6B,0xFF6B,0xFF6B}, /* FF6A */
  {0xFF6C,0xFF6C,0xFF6C},  {0xFF6D,0xFF6D,0xFF6D}, /* FF6C */
  {0xFF6E,0xFF6E,0xFF6E},  {0xFF6F,0xFF6F,0xFF6F}, /* FF6E */
  {0xFF70,0xFF70,0xFF70},  {0xFF71,0xFF71,0xFF71}, /* FF70 */
  {0xFF72,0xFF72,0xFF72},  {0xFF73,0xFF73,0xFF73}, /* FF72 */
  {0xFF74,0xFF74,0xFF74},  {0xFF75,0xFF75,0xFF75}, /* FF74 */
  {0xFF76,0xFF76,0xFF76},  {0xFF77,0xFF77,0xFF77}, /* FF76 */
  {0xFF78,0xFF78,0xFF78},  {0xFF79,0xFF79,0xFF79}, /* FF78 */
  {0xFF7A,0xFF7A,0xFF7A},  {0xFF7B,0xFF7B,0xFF7B}, /* FF7A */
  {0xFF7C,0xFF7C,0xFF7C},  {0xFF7D,0xFF7D,0xFF7D}, /* FF7C */
  {0xFF7E,0xFF7E,0xFF7E},  {0xFF7F,0xFF7F,0xFF7F}, /* FF7E */
  {0xFF80,0xFF80,0xFF80},  {0xFF81,0xFF81,0xFF81}, /* FF80 */
  {0xFF82,0xFF82,0xFF82},  {0xFF83,0xFF83,0xFF83}, /* FF82 */
  {0xFF84,0xFF84,0xFF84},  {0xFF85,0xFF85,0xFF85}, /* FF84 */
  {0xFF86,0xFF86,0xFF86},  {0xFF87,0xFF87,0xFF87}, /* FF86 */
  {0xFF88,0xFF88,0xFF88},  {0xFF89,0xFF89,0xFF89}, /* FF88 */
  {0xFF8A,0xFF8A,0xFF8A},  {0xFF8B,0xFF8B,0xFF8B}, /* FF8A */
  {0xFF8C,0xFF8C,0xFF8C},  {0xFF8D,0xFF8D,0xFF8D}, /* FF8C */
  {0xFF8E,0xFF8E,0xFF8E},  {0xFF8F,0xFF8F,0xFF8F}, /* FF8E */
  {0xFF90,0xFF90,0xFF90},  {0xFF91,0xFF91,0xFF91}, /* FF90 */
  {0xFF92,0xFF92,0xFF92},  {0xFF93,0xFF93,0xFF93}, /* FF92 */
  {0xFF94,0xFF94,0xFF94},  {0xFF95,0xFF95,0xFF95}, /* FF94 */
  {0xFF96,0xFF96,0xFF96},  {0xFF97,0xFF97,0xFF97}, /* FF96 */
  {0xFF98,0xFF98,0xFF98},  {0xFF99,0xFF99,0xFF99}, /* FF98 */
  {0xFF9A,0xFF9A,0xFF9A},  {0xFF9B,0xFF9B,0xFF9B}, /* FF9A */
  {0xFF9C,0xFF9C,0xFF9C},  {0xFF9D,0xFF9D,0xFF9D}, /* FF9C */
  {0xFF9E,0xFF9E,0xFF9E},  {0xFF9F,0xFF9F,0xFF9F}, /* FF9E */
  {0xFFA0,0xFFA0,0xFFA0},  {0xFFA1,0xFFA1,0xFFA1}, /* FFA0 */
  {0xFFA2,0xFFA2,0xFFA2},  {0xFFA3,0xFFA3,0xFFA3}, /* FFA2 */
  {0xFFA4,0xFFA4,0xFFA4},  {0xFFA5,0xFFA5,0xFFA5}, /* FFA4 */
  {0xFFA6,0xFFA6,0xFFA6},  {0xFFA7,0xFFA7,0xFFA7}, /* FFA6 */
  {0xFFA8,0xFFA8,0xFFA8},  {0xFFA9,0xFFA9,0xFFA9}, /* FFA8 */
  {0xFFAA,0xFFAA,0xFFAA},  {0xFFAB,0xFFAB,0xFFAB}, /* FFAA */
  {0xFFAC,0xFFAC,0xFFAC},  {0xFFAD,0xFFAD,0xFFAD}, /* FFAC */
  {0xFFAE,0xFFAE,0xFFAE},  {0xFFAF,0xFFAF,0xFFAF}, /* FFAE */
  {0xFFB0,0xFFB0,0xFFB0},  {0xFFB1,0xFFB1,0xFFB1}, /* FFB0 */
  {0xFFB2,0xFFB2,0xFFB2},  {0xFFB3,0xFFB3,0xFFB3}, /* FFB2 */
  {0xFFB4,0xFFB4,0xFFB4},  {0xFFB5,0xFFB5,0xFFB5}, /* FFB4 */
  {0xFFB6,0xFFB6,0xFFB6},  {0xFFB7,0xFFB7,0xFFB7}, /* FFB6 */
  {0xFFB8,0xFFB8,0xFFB8},  {0xFFB9,0xFFB9,0xFFB9}, /* FFB8 */
  {0xFFBA,0xFFBA,0xFFBA},  {0xFFBB,0xFFBB,0xFFBB}, /* FFBA */
  {0xFFBC,0xFFBC,0xFFBC},  {0xFFBD,0xFFBD,0xFFBD}, /* FFBC */
  {0xFFBE,0xFFBE,0xFFBE},  {0xFFBF,0xFFBF,0xFFBF}, /* FFBE */
  {0xFFC0,0xFFC0,0xFFC0},  {0xFFC1,0xFFC1,0xFFC1}, /* FFC0 */
  {0xFFC2,0xFFC2,0xFFC2},  {0xFFC3,0xFFC3,0xFFC3}, /* FFC2 */
  {0xFFC4,0xFFC4,0xFFC4},  {0xFFC5,0xFFC5,0xFFC5}, /* FFC4 */
  {0xFFC6,0xFFC6,0xFFC6},  {0xFFC7,0xFFC7,0xFFC7}, /* FFC6 */
  {0xFFC8,0xFFC8,0xFFC8},  {0xFFC9,0xFFC9,0xFFC9}, /* FFC8 */
  {0xFFCA,0xFFCA,0xFFCA},  {0xFFCB,0xFFCB,0xFFCB}, /* FFCA */
  {0xFFCC,0xFFCC,0xFFCC},  {0xFFCD,0xFFCD,0xFFCD}, /* FFCC */
  {0xFFCE,0xFFCE,0xFFCE},  {0xFFCF,0xFFCF,0xFFCF}, /* FFCE */
  {0xFFD0,0xFFD0,0xFFD0},  {0xFFD1,0xFFD1,0xFFD1}, /* FFD0 */
  {0xFFD2,0xFFD2,0xFFD2},  {0xFFD3,0xFFD3,0xFFD3}, /* FFD2 */
  {0xFFD4,0xFFD4,0xFFD4},  {0xFFD5,0xFFD5,0xFFD5}, /* FFD4 */
  {0xFFD6,0xFFD6,0xFFD6},  {0xFFD7,0xFFD7,0xFFD7}, /* FFD6 */
  {0xFFD8,0xFFD8,0xFFD8},  {0xFFD9,0xFFD9,0xFFD9}, /* FFD8 */
  {0xFFDA,0xFFDA,0xFFDA},  {0xFFDB,0xFFDB,0xFFDB}, /* FFDA */
  {0xFFDC,0xFFDC,0xFFDC},  {0xFFDD,0xFFDD,0xFFDD}, /* FFDC */
  {0xFFDE,0xFFDE,0xFFDE},  {0xFFDF,0xFFDF,0xFFDF}, /* FFDE */
  {0xFFE0,0xFFE0,0xFFE0},  {0xFFE1,0xFFE1,0xFFE1}, /* FFE0 */
  {0xFFE2,0xFFE2,0xFFE2},  {0xFFE3,0xFFE3,0xFFE3}, /* FFE2 */
  {0xFFE4,0xFFE4,0xFFE4},  {0xFFE5,0xFFE5,0xFFE5}, /* FFE4 */
  {0xFFE6,0xFFE6,0xFFE6},  {0xFFE7,0xFFE7,0xFFE7}, /* FFE6 */
  {0xFFE8,0xFFE8,0xFFE8},  {0xFFE9,0xFFE9,0xFFE9}, /* FFE8 */
  {0xFFEA,0xFFEA,0xFFEA},  {0xFFEB,0xFFEB,0xFFEB}, /* FFEA */
  {0xFFEC,0xFFEC,0xFFEC},  {0xFFED,0xFFED,0xFFED}, /* FFEC */
  {0xFFEE,0xFFEE,0xFFEE},  {0xFFEF,0xFFEF,0xFFEF}, /* FFEE */
  {0xFFF0,0xFFF0,0xFFF0},  {0xFFF1,0xFFF1,0xFFF1}, /* FFF0 */
  {0xFFF2,0xFFF2,0xFFF2},  {0xFFF3,0xFFF3,0xFFF3}, /* FFF2 */
  {0xFFF4,0xFFF4,0xFFF4},  {0xFFF5,0xFFF5,0xFFF5}, /* FFF4 */
  {0xFFF6,0xFFF6,0xFFF6},  {0xFFF7,0xFFF7,0xFFF7}, /* FFF6 */
  {0xFFF8,0xFFF8,0xFFF8},  {0xFFF9,0xFFF9,0xFFF9}, /* FFF8 */
  {0xFFFA,0xFFFA,0xFFFA},  {0xFFFB,0xFFFB,0xFFFB}, /* FFFA */
  {0xFFFC,0xFFFC,0xFFFC},  {0xFFFD,0xFFFD,0xFFFD}, /* FFFC */
  {0xFFFE,0xFFFE,0xFFFE},  {0xFFFF,0xFFFF,0xFFFF} /* FFFE */
};

static MY_UNICASE_CHARACTER u520p104[]={
  {0x10400,0x10428,0x10400},  {0x10401,0x10429,0x10401}, /* 10400 */
  {0x10402,0x1042A,0x10402},  {0x10403,0x1042B,0x10403}, /* 10402 */
  {0x10404,0x1042C,0x10404},  {0x10405,0x1042D,0x10405}, /* 10404 */
  {0x10406,0x1042E,0x10406},  {0x10407,0x1042F,0x10407}, /* 10406 */
  {0x10408,0x10430,0x10408},  {0x10409,0x10431,0x10409}, /* 10408 */
  {0x1040A,0x10432,0x1040A},  {0x1040B,0x10433,0x1040B}, /* 1040A */
  {0x1040C,0x10434,0x1040C},  {0x1040D,0x10435,0x1040D}, /* 1040C */
  {0x1040E,0x10436,0x1040E},  {0x1040F,0x10437,0x1040F}, /* 1040E */
  {0x10410,0x10438,0x10410},  {0x10411,0x10439,0x10411}, /* 10410 */
  {0x10412,0x1043A,0x10412},  {0x10413,0x1043B,0x10413}, /* 10412 */
  {0x10414,0x1043C,0x10414},  {0x10415,0x1043D,0x10415}, /* 10414 */
  {0x10416,0x1043E,0x10416},  {0x10417,0x1043F,0x10417}, /* 10416 */
  {0x10418,0x10440,0x10418},  {0x10419,0x10441,0x10419}, /* 10418 */
  {0x1041A,0x10442,0x1041A},  {0x1041B,0x10443,0x1041B}, /* 1041A */
  {0x1041C,0x10444,0x1041C},  {0x1041D,0x10445,0x1041D}, /* 1041C */
  {0x1041E,0x10446,0x1041E},  {0x1041F,0x10447,0x1041F}, /* 1041E */
  {0x10420,0x10448,0x10420},  {0x10421,0x10449,0x10421}, /* 10420 */
  {0x10422,0x1044A,0x10422},  {0x10423,0x1044B,0x10423}, /* 10422 */
  {0x10424,0x1044C,0x10424},  {0x10425,0x1044D,0x10425}, /* 10424 */
  {0x10426,0x1044E,0x10426},  {0x10427,0x1044F,0x10427}, /* 10426 */
  {0x10400,0x10428,0x10400},  {0x10401,0x10429,0x10401}, /* 10428 */
  {0x10402,0x1042A,0x10402},  {0x10403,0x1042B,0x10403}, /* 1042A */
  {0x10404,0x1042C,0x10404},  {0x10405,0x1042D,0x10405}, /* 1042C */
  {0x10406,0x1042E,0x10406},  {0x10407,0x1042F,0x10407}, /* 1042E */
  {0x10408,0x10430,0x10408},  {0x10409,0x10431,0x10409}, /* 10430 */
  {0x1040A,0x10432,0x1040A},  {0x1040B,0x10433,0x1040B}, /* 10432 */
  {0x1040C,0x10434,0x1040C},  {0x1040D,0x10435,0x1040D}, /* 10434 */
  {0x1040E,0x10436,0x1040E},  {0x1040F,0x10437,0x1040F}, /* 10436 */
  {0x10410,0x10438,0x10410},  {0x10411,0x10439,0x10411}, /* 10438 */
  {0x10412,0x1043A,0x10412},  {0x10413,0x1043B,0x10413}, /* 1043A */
  {0x10414,0x1043C,0x10414},  {0x10415,0x1043D,0x10415}, /* 1043C */
  {0x10416,0x1043E,0x10416},  {0x10417,0x1043F,0x10417}, /* 1043E */
  {0x10418,0x10440,0x10418},  {0x10419,0x10441,0x10419}, /* 10440 */
  {0x1041A,0x10442,0x1041A},  {0x1041B,0x10443,0x1041B}, /* 10442 */
  {0x1041C,0x10444,0x1041C},  {0x1041D,0x10445,0x1041D}, /* 10444 */
  {0x1041E,0x10446,0x1041E},  {0x1041F,0x10447,0x1041F}, /* 10446 */
  {0x10420,0x10448,0x10420},  {0x10421,0x10449,0x10421}, /* 10448 */
  {0x10422,0x1044A,0x10422},  {0x10423,0x1044B,0x10423}, /* 1044A */
  {0x10424,0x1044C,0x10424},  {0x10425,0x1044D,0x10425}, /* 1044C */
  {0x10426,0x1044E,0x10426},  {0x10427,0x1044F,0x10427}, /* 1044E */
  {0x10450,0x10450,0x10450},  {0x10451,0x10451,0x10451}, /* 10450 */
  {0x10452,0x10452,0x10452},  {0x10453,0x10453,0x10453}, /* 10452 */
  {0x10454,0x10454,0x10454},  {0x10455,0x10455,0x10455}, /* 10454 */
  {0x10456,0x10456,0x10456},  {0x10457,0x10457,0x10457}, /* 10456 */
  {0x10458,0x10458,0x10458},  {0x10459,0x10459,0x10459}, /* 10458 */
  {0x1045A,0x1045A,0x1045A},  {0x1045B,0x1045B,0x1045B}, /* 1045A */
  {0x1045C,0x1045C,0x1045C},  {0x1045D,0x1045D,0x1045D}, /* 1045C */
  {0x1045E,0x1045E,0x1045E},  {0x1045F,0x1045F,0x1045F}, /* 1045E */
  {0x10460,0x10460,0x10460},  {0x10461,0x10461,0x10461}, /* 10460 */
  {0x10462,0x10462,0x10462},  {0x10463,0x10463,0x10463}, /* 10462 */
  {0x10464,0x10464,0x10464},  {0x10465,0x10465,0x10465}, /* 10464 */
  {0x10466,0x10466,0x10466},  {0x10467,0x10467,0x10467}, /* 10466 */
  {0x10468,0x10468,0x10468},  {0x10469,0x10469,0x10469}, /* 10468 */
  {0x1046A,0x1046A,0x1046A},  {0x1046B,0x1046B,0x1046B}, /* 1046A */
  {0x1046C,0x1046C,0x1046C},  {0x1046D,0x1046D,0x1046D}, /* 1046C */
  {0x1046E,0x1046E,0x1046E},  {0x1046F,0x1046F,0x1046F}, /* 1046E */
  {0x10470,0x10470,0x10470},  {0x10471,0x10471,0x10471}, /* 10470 */
  {0x10472,0x10472,0x10472},  {0x10473,0x10473,0x10473}, /* 10472 */
  {0x10474,0x10474,0x10474},  {0x10475,0x10475,0x10475}, /* 10474 */
  {0x10476,0x10476,0x10476},  {0x10477,0x10477,0x10477}, /* 10476 */
  {0x10478,0x10478,0x10478},  {0x10479,0x10479,0x10479}, /* 10478 */
  {0x1047A,0x1047A,0x1047A},  {0x1047B,0x1047B,0x1047B}, /* 1047A */
  {0x1047C,0x1047C,0x1047C},  {0x1047D,0x1047D,0x1047D}, /* 1047C */
  {0x1047E,0x1047E,0x1047E},  {0x1047F,0x1047F,0x1047F}, /* 1047E */
  {0x10480,0x10480,0x10480},  {0x10481,0x10481,0x10481}, /* 10480 */
  {0x10482,0x10482,0x10482},  {0x10483,0x10483,0x10483}, /* 10482 */
  {0x10484,0x10484,0x10484},  {0x10485,0x10485,0x10485}, /* 10484 */
  {0x10486,0x10486,0x10486},  {0x10487,0x10487,0x10487}, /* 10486 */
  {0x10488,0x10488,0x10488},  {0x10489,0x10489,0x10489}, /* 10488 */
  {0x1048A,0x1048A,0x1048A},  {0x1048B,0x1048B,0x1048B}, /* 1048A */
  {0x1048C,0x1048C,0x1048C},  {0x1048D,0x1048D,0x1048D}, /* 1048C */
  {0x1048E,0x1048E,0x1048E},  {0x1048F,0x1048F,0x1048F}, /* 1048E */
  {0x10490,0x10490,0x10490},  {0x10491,0x10491,0x10491}, /* 10490 */
  {0x10492,0x10492,0x10492},  {0x10493,0x10493,0x10493}, /* 10492 */
  {0x10494,0x10494,0x10494},  {0x10495,0x10495,0x10495}, /* 10494 */
  {0x10496,0x10496,0x10496},  {0x10497,0x10497,0x10497}, /* 10496 */
  {0x10498,0x10498,0x10498},  {0x10499,0x10499,0x10499}, /* 10498 */
  {0x1049A,0x1049A,0x1049A},  {0x1049B,0x1049B,0x1049B}, /* 1049A */
  {0x1049C,0x1049C,0x1049C},  {0x1049D,0x1049D,0x1049D}, /* 1049C */
  {0x1049E,0x1049E,0x1049E},  {0x1049F,0x1049F,0x1049F}, /* 1049E */
  {0x104A0,0x104A0,0x104A0},  {0x104A1,0x104A1,0x104A1}, /* 104A0 */
  {0x104A2,0x104A2,0x104A2},  {0x104A3,0x104A3,0x104A3}, /* 104A2 */
  {0x104A4,0x104A4,0x104A4},  {0x104A5,0x104A5,0x104A5}, /* 104A4 */
  {0x104A6,0x104A6,0x104A6},  {0x104A7,0x104A7,0x104A7}, /* 104A6 */
  {0x104A8,0x104A8,0x104A8},  {0x104A9,0x104A9,0x104A9}, /* 104A8 */
  {0x104AA,0x104AA,0x104AA},  {0x104AB,0x104AB,0x104AB}, /* 104AA */
  {0x104AC,0x104AC,0x104AC},  {0x104AD,0x104AD,0x104AD}, /* 104AC */
  {0x104AE,0x104AE,0x104AE},  {0x104AF,0x104AF,0x104AF}, /* 104AE */
  {0x104B0,0x104B0,0x104B0},  {0x104B1,0x104B1,0x104B1}, /* 104B0 */
  {0x104B2,0x104B2,0x104B2},  {0x104B3,0x104B3,0x104B3}, /* 104B2 */
  {0x104B4,0x104B4,0x104B4},  {0x104B5,0x104B5,0x104B5}, /* 104B4 */
  {0x104B6,0x104B6,0x104B6},  {0x104B7,0x104B7,0x104B7}, /* 104B6 */
  {0x104B8,0x104B8,0x104B8},  {0x104B9,0x104B9,0x104B9}, /* 104B8 */
  {0x104BA,0x104BA,0x104BA},  {0x104BB,0x104BB,0x104BB}, /* 104BA */
  {0x104BC,0x104BC,0x104BC},  {0x104BD,0x104BD,0x104BD}, /* 104BC */
  {0x104BE,0x104BE,0x104BE},  {0x104BF,0x104BF,0x104BF}, /* 104BE */
  {0x104C0,0x104C0,0x104C0},  {0x104C1,0x104C1,0x104C1}, /* 104C0 */
  {0x104C2,0x104C2,0x104C2},  {0x104C3,0x104C3,0x104C3}, /* 104C2 */
  {0x104C4,0x104C4,0x104C4},  {0x104C5,0x104C5,0x104C5}, /* 104C4 */
  {0x104C6,0x104C6,0x104C6},  {0x104C7,0x104C7,0x104C7}, /* 104C6 */
  {0x104C8,0x104C8,0x104C8},  {0x104C9,0x104C9,0x104C9}, /* 104C8 */
  {0x104CA,0x104CA,0x104CA},  {0x104CB,0x104CB,0x104CB}, /* 104CA */
  {0x104CC,0x104CC,0x104CC},  {0x104CD,0x104CD,0x104CD}, /* 104CC */
  {0x104CE,0x104CE,0x104CE},  {0x104CF,0x104CF,0x104CF}, /* 104CE */
  {0x104D0,0x104D0,0x104D0},  {0x104D1,0x104D1,0x104D1}, /* 104D0 */
  {0x104D2,0x104D2,0x104D2},  {0x104D3,0x104D3,0x104D3}, /* 104D2 */
  {0x104D4,0x104D4,0x104D4},  {0x104D5,0x104D5,0x104D5}, /* 104D4 */
  {0x104D6,0x104D6,0x104D6},  {0x104D7,0x104D7,0x104D7}, /* 104D6 */
  {0x104D8,0x104D8,0x104D8},  {0x104D9,0x104D9,0x104D9}, /* 104D8 */
  {0x104DA,0x104DA,0x104DA},  {0x104DB,0x104DB,0x104DB}, /* 104DA */
  {0x104DC,0x104DC,0x104DC},  {0x104DD,0x104DD,0x104DD}, /* 104DC */
  {0x104DE,0x104DE,0x104DE},  {0x104DF,0x104DF,0x104DF}, /* 104DE */
  {0x104E0,0x104E0,0x104E0},  {0x104E1,0x104E1,0x104E1}, /* 104E0 */
  {0x104E2,0x104E2,0x104E2},  {0x104E3,0x104E3,0x104E3}, /* 104E2 */
  {0x104E4,0x104E4,0x104E4},  {0x104E5,0x104E5,0x104E5}, /* 104E4 */
  {0x104E6,0x104E6,0x104E6},  {0x104E7,0x104E7,0x104E7}, /* 104E6 */
  {0x104E8,0x104E8,0x104E8},  {0x104E9,0x104E9,0x104E9}, /* 104E8 */
  {0x104EA,0x104EA,0x104EA},  {0x104EB,0x104EB,0x104EB}, /* 104EA */
  {0x104EC,0x104EC,0x104EC},  {0x104ED,0x104ED,0x104ED}, /* 104EC */
  {0x104EE,0x104EE,0x104EE},  {0x104EF,0x104EF,0x104EF}, /* 104EE */
  {0x104F0,0x104F0,0x104F0},  {0x104F1,0x104F1,0x104F1}, /* 104F0 */
  {0x104F2,0x104F2,0x104F2},  {0x104F3,0x104F3,0x104F3}, /* 104F2 */
  {0x104F4,0x104F4,0x104F4},  {0x104F5,0x104F5,0x104F5}, /* 104F4 */
  {0x104F6,0x104F6,0x104F6},  {0x104F7,0x104F7,0x104F7}, /* 104F6 */
  {0x104F8,0x104F8,0x104F8},  {0x104F9,0x104F9,0x104F9}, /* 104F8 */
  {0x104FA,0x104FA,0x104FA},  {0x104FB,0x104FB,0x104FB}, /* 104FA */
  {0x104FC,0x104FC,0x104FC},  {0x104FD,0x104FD,0x104FD}, /* 104FC */
  {0x104FE,0x104FE,0x104FE},  {0x104FF,0x104FF,0x104FF} /* 104FE */
};


MY_UNICASE_CHARACTER *my_unicase_pages_unicode520[4352]=
{
 u520p00, u520p01, u520p02, u520p03, u520p04, u520p05,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
 u520p10,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL, u520p1D, u520p1E, u520p1F,
    NULL, u520p21,    NULL,    NULL, u520p24,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL, u520p2C, u520p2D,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, u520pA6, u520pA7,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,
    NULL,    NULL,    NULL,    NULL,    NULL,    NULL,    NULL, u520pFF,
    NULL,    NULL,    NULL,    NULL,u520p104,    NULL,    NULL,    NULL,
};


MY_UNICASE_INFO my_unicase_unicode520=
{
  0x104FF,
  my_unicase_pages_unicode520
};


static inline void
my_tosort_unicode(MY_UNICASE_INFO *uni_plane, my_wc_t *wc, uint flags)
{
  if (*wc <= uni_plane->maxchar)
  {
    MY_UNICASE_CHARACTER *page;
    if ((page= uni_plane->page[*wc >> 8]))
      *wc= (flags & MY_CS_LOWER_SORT) ?
           page[*wc & 0xFF].tolower :
           page[*wc & 0xFF].sort;
  }
  else
  {
    *wc= MY_CS_REPLACEMENT_CHARACTER;
  }
}


/*
** Compare string against string with wildcard
** This function is used in UTF8 and UCS2
**
**	0 if matched
**	-1 if not matched with wildcard
**	 1 if matched with wildcard
*/

static
int my_wildcmp_unicode_impl(CHARSET_INFO *cs,
                            const char *str,const char *str_end,
                            const char *wildstr,const char *wildend,
                            int escape, int w_one, int w_many,
                            MY_UNICASE_INFO *weights, int recurse_level)
{
  int result= -1;                             /* Not found, using wildcards */
  my_wc_t s_wc, w_wc;
  int scan;
  my_charset_conv_mb_wc mb_wc= cs->cset->mb_wc;

  if (my_string_stack_guard && my_string_stack_guard(recurse_level))
    return 1;
  while (wildstr != wildend)
  {
    while (1)
    {
      my_bool escaped= 0;
      if ((scan= mb_wc(cs, &w_wc, (const uchar*)wildstr,
                       (const uchar*)wildend)) <= 0)
        return 1;

      if (w_wc == (my_wc_t) w_many)
      {
        result= 1;                                /* Found an anchor char */
        break;
      }

      wildstr+= scan;
      if (w_wc ==  (my_wc_t) escape && wildstr < wildend)
      {
        if ((scan= mb_wc(cs, &w_wc, (const uchar*)wildstr,
                         (const uchar*)wildend)) <= 0)
          return 1;
        wildstr+= scan;
        escaped= 1;
      }

      if ((scan= mb_wc(cs, &s_wc, (const uchar*)str,
                       (const uchar*)str_end)) <= 0)
        return 1;
      str+= scan;

      if (!escaped && w_wc == (my_wc_t) w_one)
      {
        result= 1;                                /* Found an anchor char */
      }
      else
      {
        if (weights)
        {
          my_tosort_unicode(weights, &s_wc, cs->state);
          my_tosort_unicode(weights, &w_wc, cs->state);
        }
        if (s_wc != w_wc)
          return 1;                               /* No match */
      }
      if (wildstr == wildend)
        return (str != str_end);                  /* Match if both are at end */
    }

    if (w_wc == (my_wc_t) w_many)
    {                                             /* Found w_many */
      /* Remove any '%' and '_' from the wild search string */
      for ( ; wildstr != wildend ; )
      {
        if ((scan= mb_wc(cs, &w_wc, (const uchar*)wildstr,
                         (const uchar*)wildend)) <= 0)
          return 1;

        if (w_wc == (my_wc_t) w_many)
        {
          wildstr+= scan;
          continue;
        } 

        if (w_wc == (my_wc_t) w_one)
        {
          wildstr+= scan;
          if ((scan= mb_wc(cs, &s_wc, (const uchar*)str,
                           (const uchar*)str_end)) <= 0)
            return 1;
          str+= scan;
          continue;
        }
        break;                                        /* Not a wild character */
      }

      if (wildstr == wildend)
        return 0;                                /* Ok if w_many is last */

      if (str == str_end)
        return -1;

      if ((scan= mb_wc(cs, &w_wc, (const uchar*)wildstr,
                       (const uchar*)wildend)) <= 0)
        return 1;
      wildstr+= scan;

      if (w_wc ==  (my_wc_t) escape)
      {
        if (wildstr < wildend)
        {
          if ((scan= mb_wc(cs, &w_wc, (const uchar*)wildstr,
                           (const uchar*)wildend)) <= 0)
            return 1;
          wildstr+= scan;
        }
      }

      while (1)
      {
        /* Skip until the first character from wildstr is found */
        while (str != str_end)
        {
          if ((scan= mb_wc(cs, &s_wc, (const uchar*)str,
                           (const uchar*)str_end)) <= 0)
            return 1;
          if (weights)
          {
            my_tosort_unicode(weights, &s_wc, cs->state);
            my_tosort_unicode(weights, &w_wc, cs->state);
          }

          if (s_wc == w_wc)
            break;
          str+= scan;
        }
        if (str == str_end)
          return -1;

        str+= scan;
        result= my_wildcmp_unicode_impl(cs, str, str_end, wildstr, wildend,
                                        escape, w_one, w_many,
                                        weights, recurse_level + 1);
        if (result <= 0)
          return result;
      }
    }
  }
  return (str != str_end ? 1 : 0);
}


int
my_wildcmp_unicode(CHARSET_INFO *cs,
                   const char *str,const char *str_end,
                   const char *wildstr,const char *wildend,
                   int escape, int w_one, int w_many,
                   MY_UNICASE_INFO *weights)
{
  return my_wildcmp_unicode_impl(cs, str, str_end,
                                 wildstr, wildend,
                                 escape, w_one, w_many, weights, 1);
}


/**
  Pad buffer with weights for space characters.
  
  @details
  This functions fills the buffer pointed by "str"
  with weights of space character. Not more than
  "nweights" weights are put. If at some iteration
  step only a half of weight can fit
  (which is possible if buffer length is an odd number)
  then a half of this weight is put - this gives
  a little bit better ORDER BY result for long strings.
  
  @str      Buffer
  @strend   End of buffer
  @nweights Number of weights
  
  @return Result length
*/

size_t
my_strxfrm_pad_nweights_unicode(uchar *str, uchar *strend, size_t nweights)
{
  uchar *str0;
  DBUG_ASSERT(str && str <= strend); 
  for (str0= str; str < strend && nweights; nweights--)
  {
    *str++= 0x00;
    if (str < strend)
      *str++= 0x20;
  }
  return str - str0;
}


/**
  Pad buffer with weights for space characters.
  
  @details
  This functions fills the buffer pointed by "str"
  with weights of space character. Putting half of weight
  (when buffer length is an odd number) is OK.
  
  @str      Buffer
  @strend   End of buffer
  
  @return Result length
*/

size_t
my_strxfrm_pad_unicode(uchar *str, uchar *strend)
{
  uchar *str0= str;
  DBUG_ASSERT(str && str <= strend); 
  for ( ; str < strend ; )
  {
    *str++= 0x00;
    if (str < strend)
      *str++= 0x20;
  }
  return str - str0;
}


/*
  For BMP-only collations that use 2 bytes per weight.
*/
size_t
my_strnxfrmlen_unicode(CHARSET_INFO *cs, size_t len)
{
  
  return ((len + cs->mbmaxlen - 1) / cs->mbmaxlen) * 2;
}

size_t
my_strnxfrm_unicode_full_bin_internal(CHARSET_INFO *cs,
                                      uchar *dst, uchar *de, uint *nweights,
                                      const uchar *src, const uchar *se)
{
  my_wc_t UNINIT_VAR(wc);
  uchar *dst0= dst;

  DBUG_ASSERT(src || !se);
  DBUG_ASSERT(cs->state & MY_CS_BINSORT);

  for (; dst < de && *nweights; (*nweights)--)
  {
    int res;
    if ((res= cs->cset->mb_wc(cs, &wc, src, se)) <= 0)
      break;
    src+= res;
    *dst++= (uchar) (wc >> 16);
    if (dst < de)
    {
      *dst++= (uchar) ((wc >> 8) & 0xFF);
      if (dst < de)
        *dst++= (uchar) (wc & 0xFF);
    }
  }
  return dst - dst0;
}


/*
  Store sorting weights using 3 bytes per character.
  This function is shared between utf8mb4_bin, utf16_bin, utf32_bin.
*/
size_t
my_strnxfrm_unicode_full_bin(CHARSET_INFO *cs,
                             uchar *dst, size_t dstlen, uint nweights,
                             const uchar *src, size_t srclen, uint flags)
{
  uchar *dst0= dst;
  uchar *de= dst + dstlen;

  dst+= my_strnxfrm_unicode_full_bin_internal(cs, dst, de, &nweights,
                                                  src, src + srclen);
  DBUG_ASSERT(dst <= de); /* Safety */

  if (flags & MY_STRXFRM_PAD_WITH_SPACE)
  {
    for ( ; dst < de && nweights; nweights--)
    {
      *dst++= 0x00;
      if (dst < de)
      {
        *dst++= 0x00;
        if (dst < de)
          *dst++= 0x20;
      }
    }
  }

  my_strxfrm_desc_and_reverse(dst0, dst, flags, 0);

  if (flags & MY_STRXFRM_PAD_TO_MAXLEN)
  {
    while (dst < de)
    {
      *dst++= 0x00;
      if (dst < de)
      {
        *dst++= 0x00;
        if (dst < de)
          *dst++= 0x20;
      }
    }
  }
  return dst - dst0;
}


size_t
my_strnxfrm_unicode_full_nopad_bin(CHARSET_INFO *cs,
                                   uchar *dst, size_t dstlen, uint nweights,
                                   const uchar *src, size_t srclen, uint flags)
{
  uchar *dst0= dst;
  uchar *de= dst + dstlen;

  dst+= my_strnxfrm_unicode_full_bin_internal(cs, dst, de, &nweights,
                                                  src, src + srclen);
  DBUG_ASSERT(dst <= de); /* Safety */

  if (dst < de && nweights && (flags & MY_STRXFRM_PAD_WITH_SPACE))
  {
    size_t len= de - dst;
    set_if_smaller(len, nweights * 3);
    memset(dst, 0x00, len);
    dst+= len;
  }

  my_strxfrm_desc_and_reverse(dst0, dst, flags, 0);

  if ((flags & MY_STRXFRM_PAD_TO_MAXLEN) && dst < de)
  {
    memset(dst, 0x00, de - dst);
    dst= de;
  }
  return dst - dst0;
}


size_t
my_strnxfrmlen_unicode_full_bin(CHARSET_INFO *cs, size_t len)
{
  return ((len + 3) / cs->mbmaxlen) * 3;
}
#endif /* HAVE_UNIDATA */


#ifdef HAVE_CHARSET_utf8

/*
  We consider bytes with code more than 127 as a letter.
  This guarantees that word boundaries work fine with regular
  expressions. Note, there is no need to mark byte 255  as a
  letter, it is illegal byte in UTF8.
*/
static const uchar ctype_utf8[] = {
    0,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 40, 40, 40, 40, 40, 32, 32,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
   72, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
  132,132,132,132,132,132,132,132,132,132, 16, 16, 16, 16, 16, 16,
   16,129,129,129,129,129,129,  1,  1,  1,  1,  1,  1,  1,  1,  1,
    1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 16, 16, 16, 16,
   16,130,130,130,130,130,130,  2,  2,  2,  2,  2,  2,  2,  2,  2,
    2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 16, 16, 16, 16, 32,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  0
};

/* The below are taken from usa7 implementation */

static const uchar to_lower_utf8[] = {
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122, 91, 92, 93, 94, 95,
   96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};

static const uchar to_upper_utf8[] = {
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
   96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};

static int my_utf8_uni(CHARSET_INFO *cs __attribute__((unused)),
                       my_wc_t * pwc, const uchar *s, const uchar *e)
{
  return my_mb_wc_utf8mb3_quick(pwc, s, e);
}


/*
  The same as above, but without range check
  for example, for a null-terminated string
*/
static int my_utf8_uni_no_range(CHARSET_INFO *cs __attribute__((unused)),
                                my_wc_t * pwc, const uchar *s)
{
  uchar c;

  c= s[0];
  if (c < 0x80)
  {
    *pwc = c;
    return 1;
  }

  if (c < 0xc2)
    return MY_CS_ILSEQ;

  if (c < 0xe0)
  {
    if (!((s[1] ^ 0x80) < 0x40))
      return MY_CS_ILSEQ;

    *pwc= UTF8MB2_CODE(c, s[1]);
    return 2;
  }

  if (c < 0xf0)
  {
    if (!IS_UTF8MB3_STEP2(c, s[1], s[2]))
      return MY_CS_ILSEQ;

    *pwc= UTF8MB3_CODE(c, s[1], s[2]);
    return 3;
  }
  return MY_CS_ILSEQ;
}


static int my_uni_utf8 (CHARSET_INFO *cs __attribute__((unused)),
                        my_wc_t wc, uchar *r, uchar *e)
{
  if (wc < 0x80)
  {
    if (r >= e)
      return MY_CS_TOOSMALL;
    *r= (uchar) wc;
    return 1;
  }
  if (wc < 0x800)
  {
    if (r + 2 > e)
      return MY_CS_TOOSMALLN(2);
    /* U+0080..U+07FF:  00000xxx.xxyyyyyy -> 110xxxxx 10yyyyyy */
    *r++= (uchar) (0xC0 | (wc >> 6));
    *r=   (uchar) (0x80 | (wc & 0x3F));
    return 2;
  }
  if (wc < 0x10000)
  {
    if (r + 3 > e)
      return MY_CS_TOOSMALLN(3);
    /* U+0800..U+FFFF: xxxxyyyy.yyzzzzzz  -> 1110xxxx 10yyyyyy 10zzzzzz */
    *r++= (uchar) (0xE0 | (wc >> 12));
    *r++= (uchar) (0x80 | ((wc >> 6) & 0x3f));
    *r=   (uchar) (0x80 | (wc & 0x3f));
    return 3;
  }
  return MY_CS_ILUNI;
}


/*
  The same as above, but without range check.
*/
static int my_uni_utf8_no_range(CHARSET_INFO *cs __attribute__((unused)),
                                my_wc_t wc, uchar *r)
{
  int count;

  if (wc < 0x80)
    count= 1;
  else if (wc < 0x800)
    count= 2;
  else if (wc < 0x10000)
    count= 3;
  else
    return MY_CS_ILUNI;

  switch (count)
  {
    case 3: r[2]= (uchar) (0x80 | (wc & 0x3f)); wc= wc >> 6; wc |= 0x800;
      /* fall through */
    case 2: r[1]= (uchar) (0x80 | (wc & 0x3f)); wc= wc >> 6; wc |= 0xc0;
      /* fall through */
    case 1: r[0]= (uchar) wc;
  }
  return count;
}


static inline void
my_tolower_utf8mb3(MY_UNICASE_INFO *uni_plane, my_wc_t *wc)
{
  MY_UNICASE_CHARACTER *page;
  if ((page= uni_plane->page[(*wc >> 8) & 0xFF]))
    *wc= page[*wc & 0xFF].tolower;
}


static inline void
my_toupper_utf8mb3(MY_UNICASE_INFO *uni_plane, my_wc_t *wc)
{
  MY_UNICASE_CHARACTER *page;
  if ((page= uni_plane->page[(*wc >> 8) & 0xFF]))
    *wc= page[*wc & 0xFF].toupper;
}


static size_t my_caseup_utf8(CHARSET_INFO *cs, const char *src, size_t srclen,
                             char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(src != dst || cs->caseup_multiply == 1);

  while ((src < srcend) &&
         (srcres= my_utf8_uni(cs, &wc, (uchar *) src, (uchar*) srcend)) > 0)
  {
    my_toupper_utf8mb3(uni_plane, &wc);
    if ((dstres= my_uni_utf8(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static void my_hash_sort_utf8_nopad(CHARSET_INFO *cs, const uchar *s, size_t slen,
                                    ulong *nr1, ulong *nr2)
{
  my_wc_t wc;
  int res;
  const uchar *e= s+slen;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  register ulong m1= *nr1, m2= *nr2;

  while ((s < e) && (res=my_utf8_uni(cs,&wc, (uchar *)s, (uchar*)e))>0 )
  {
    my_tosort_unicode(uni_plane, &wc, cs->state);
    MY_HASH_ADD_16(m1, m2, wc);
    s+= res;
  }
  *nr1= m1;
  *nr2= m2;
}


static void my_hash_sort_utf8(CHARSET_INFO *cs, const uchar *s, size_t slen,
                              ulong *nr1, ulong *nr2)
{
  const uchar *e= s+slen;
  /*
    Remove end space. We have to do this to be able to compare
    'A ' and 'A' as identical
  */
  while (e > s && e[-1] == ' ')
    e--;
  my_hash_sort_utf8_nopad(cs, s, e - s, nr1, nr2);
}


static size_t my_caseup_str_utf8(CHARSET_INFO *cs, char *src)
{
  my_wc_t wc;
  int srcres, dstres;
  char *dst= src, *dst0= src;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(cs->caseup_multiply == 1);

  while (*src &&
         (srcres= my_utf8_uni_no_range(cs, &wc, (uchar *) src)) > 0)
  {
    my_toupper_utf8mb3(uni_plane, &wc);
    if ((dstres= my_uni_utf8_no_range(cs, wc, (uchar*) dst)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  *dst= '\0';
  return (size_t) (dst - dst0);
}


static size_t my_casedn_utf8(CHARSET_INFO *cs, const char *src, size_t srclen,
                             char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(src != dst || cs->casedn_multiply == 1);

  while ((src < srcend) &&
         (srcres= my_utf8_uni(cs, &wc, (uchar*) src, (uchar*)srcend)) > 0)
  {
    my_tolower_utf8mb3(uni_plane, &wc);
    if ((dstres= my_uni_utf8(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static size_t my_casedn_str_utf8(CHARSET_INFO *cs, char *src)
{
  my_wc_t wc;
  int srcres, dstres;
  char *dst= src, *dst0= src;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(cs->casedn_multiply == 1);

  while (*src &&
         (srcres= my_utf8_uni_no_range(cs, &wc, (uchar *) src)) > 0)
  {
    my_tolower_utf8mb3(uni_plane, &wc);
    if ((dstres= my_uni_utf8_no_range(cs, wc, (uchar*) dst)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }

  /*
   In rare cases lower string can be shorter than
   the original string, for example:

   "U+0130 LATIN CAPITAL LETTER I WITH DOT ABOVE"
   (which is 0xC4B0 in utf8, i.e. two bytes)

   is converted into

   "U+0069 LATIN SMALL LETTER I"
   (which is 0x69 in utf8, i.e. one byte)

   So, we need to put '\0' terminator after converting.
  */

  *dst= '\0';
  return (size_t) (dst - dst0);
}


/*
  Compare 0-terminated UTF8 strings.

  SYNOPSIS
    my_strcasecmp_utf8()
    cs                  character set handler
    s                   First 0-terminated string to compare
    t                   Second 0-terminated string to compare

  IMPLEMENTATION

  RETURN
    - negative number if s < t
    - positive number if s > t
    - 0 is the strings are equal
*/

static
int my_strcasecmp_utf8(CHARSET_INFO *cs, const char *s, const char *t)
{
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  while (s[0] && t[0])
  {
    my_wc_t s_wc,t_wc;
    
    if ((uchar) s[0] < 128)
    {
      /* 
        s[0] is between 0 and 127.
        It represents a single byte character.
        Convert it into weight according to collation.
      */
      s_wc= my_unicase_default_page00[(uchar) s[0]].tolower;
      s++;
    }
    else
    {
      int res;
      
      /*
        Scan a multibyte character.

        In the future it is worth to write a special version of my_utf8_uni()
        for 0-terminated strings which will not take in account length. Now
        we call the regular version of my_utf8_uni() with s+3 in the
        last argument. s+3 is enough to scan any multibyte sequence.

        Calling the regular version of my_utf8_uni is safe for 0-terminated
        strings: we will never lose the end of the string:
        If we have 0 character in the middle of a multibyte sequence,
        then my_utf8_uni will always return a negative number, so the
        loop with finish.
      */
      
      res= my_utf8_uni(cs,&s_wc, (const uchar*)s, (const uchar*) s + 3);
      
      /* 
         In the case of wrong multibyte sequence we will
         call strcmp() for byte-to-byte comparison.
      */
      if (res <= 0)
        return strcmp(s, t);
      s+= res;
      
      /* Convert Unicode code into weight according to collation */
      my_tolower_utf8mb3(uni_plane, &s_wc);
    }
    
    
    /* Do the same for the second string */
    
    if ((uchar) t[0] < 128)
    {
      /* Convert single byte character into weight */
      t_wc= my_unicase_default_page00[(uchar) t[0]].tolower;
      t++;
    }
    else
    {
      int res=my_utf8_uni(cs,&t_wc, (const uchar*)t, (const uchar*) t + 3);
      if (res <= 0)
        return strcmp(s, t);
      t+= res;
      
      /* Convert code into weight */
      my_tolower_utf8mb3(uni_plane, &t_wc);
    }
    
    /* Now we have two weights, let's compare them */
    if ( s_wc != t_wc )
      return  ((int) s_wc) - ((int) t_wc);
  }
  return ((int)(uchar)s[0]) - ((int) (uchar) t[0]);
}


static
int my_wildcmp_utf8(CHARSET_INFO *cs,
		    const char *str,const char *str_end,
		    const char *wildstr,const char *wildend,
		    int escape, int w_one, int w_many)
{
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  return my_wildcmp_unicode(cs,str,str_end,wildstr,wildend,
                            escape,w_one,w_many,uni_plane); 
}


static
int my_charlen_utf8(CHARSET_INFO *cs __attribute__((unused)),
                    const uchar *s, const uchar *e)
{
  uchar c;

  if (s >= e)
    return MY_CS_TOOSMALL;

  c= s[0];
  if (c < 0xf0)
    return my_valid_mbcharlen_utf8mb3(s, e);

  return MY_CS_ILSEQ;
}


#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8
#define CHARLEN(cs,str,end)       my_charlen_utf8(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.ic"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_utf8 */


static inline int my_weight_mb1_utf8_general_ci(uchar b)
{
  return (int) my_unicase_default_page00[b & 0xFF].sort;
}


static inline int my_weight_mb2_utf8_general_ci(uchar b0, uchar b1)
{
  my_wc_t wc= UTF8MB2_CODE(b0, b1);
  MY_UNICASE_CHARACTER *page= my_unicase_default_pages[wc >> 8];
  return (int) (page ? page[wc & 0xFF].sort : wc);
}


static inline int my_weight_mb3_utf8_general_ci(uchar b0, uchar b1, uchar b2)
{
  my_wc_t wc= UTF8MB3_CODE(b0, b1, b2);
  MY_UNICASE_CHARACTER *page= my_unicase_default_pages[wc >> 8];
  return (int) (page ? page[wc & 0xFF].sort : wc);
}


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8_general_ci
#define DEFINE_STRNXFRM_UNICODE
#define DEFINE_STRNXFRM_UNICODE_NOPAD
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define UNICASE_MAXCHAR          MY_UNICASE_INFO_DEFAULT_MAXCHAR
#define UNICASE_PAGE0            my_unicase_default_page00
#define UNICASE_PAGES            my_unicase_default_pages
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            my_weight_mb1_utf8_general_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8_general_ci(x,y,z)
#include "strcoll.ic"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)    my_ ## x ## _utf8_general_nopad_ci
#define WEIGHT_ILSEQ(x)        (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)          my_weight_mb1_utf8_general_ci(x)
#define WEIGHT_MB2(x,y)        my_weight_mb2_utf8_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)      my_weight_mb3_utf8_general_ci(x,y,z)
#include "strcoll.ic"


static inline int my_weight_mb1_utf8_general_mysql500_ci(uchar b)
{
  return (int) plane00_mysql500[b & 0xFF].sort;
}


static inline int my_weight_mb2_utf8_general_mysql500_ci(uchar b0, uchar b1)
{
  my_wc_t wc= UTF8MB2_CODE(b0, b1);
  MY_UNICASE_CHARACTER *page= my_unicase_pages_mysql500[wc >> 8];
  return (int) (page ? page[wc & 0xFF].sort : wc);
}


static inline int
my_weight_mb3_utf8_general_mysql500_ci(uchar b0, uchar b1, uchar b2)
{
  my_wc_t wc= UTF8MB3_CODE(b0, b1, b2);
  MY_UNICASE_CHARACTER *page= my_unicase_pages_mysql500[wc >> 8];
  return (int) (page ? page[wc & 0xFF].sort : wc);
}


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8_general_mysql500_ci
#define DEFINE_STRNXFRM_UNICODE
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define UNICASE_MAXCHAR          MY_UNICASE_INFO_DEFAULT_MAXCHAR
#define UNICASE_PAGE0            plane00_mysql500
#define UNICASE_PAGES            my_unicase_pages_mysql500
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            my_weight_mb1_utf8_general_mysql500_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8_general_mysql500_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8_general_mysql500_ci(x,y,z)
#include "strcoll.ic"


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8_bin
#define DEFINE_STRNXFRM_UNICODE_BIN2
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb3_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            ((int) (uchar) (x))
#define WEIGHT_MB2(x,y)          ((int) UTF8MB2_CODE(x,y))
#define WEIGHT_MB3(x,y,z)        ((int) UTF8MB3_CODE(x,y,z))
#include "strcoll.ic"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)    my_ ## x ## _utf8_nopad_bin
#define WEIGHT_ILSEQ(x)        (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)          ((int) (uchar) (x))
#define WEIGHT_MB2(x,y)        ((int) UTF8MB2_CODE(x,y))
#define WEIGHT_MB3(x,y,z)      ((int) UTF8MB3_CODE(x,y,z))
#include "strcoll.ic"

/*
  TODO-10.2: join this with pad_max_char() in ctype-mb.c
*/
static void
my_fill_utf8_mb(CHARSET_INFO *cs, char *str, size_t length, int fill)
{
  char *end= str + length;
  char buf[10];
  char buflen= cs->cset->native_to_mb(cs, (my_wc_t) fill, (uchar*) buf,
                                      (uchar*) buf + sizeof(buf));
  DBUG_ASSERT(buflen > 0);
  for ( ; str + buflen <= end ; )
  {
    memcpy(str, buf, buflen);
    str+= buflen;
  }

  for ( ; str < end; )
    *str++= ' ';
}


static void
my_fill_utf8(CHARSET_INFO *cs, char *str, size_t length, int fill)
{
  if (fill < 0x80)
    my_fill_8bit(cs, str, length, fill);
  else
    my_fill_utf8_mb(cs, str, length, fill);
}


static MY_COLLATION_HANDLER my_collation_utf8_general_ci_handler =
{
    NULL,               /* init */
    my_strnncoll_utf8_general_ci,
    my_strnncollsp_utf8_general_ci,
    my_strnxfrm_utf8_general_ci,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8,
    my_strcasecmp_utf8,
    my_instr_mb,
    my_hash_sort_utf8,
    my_propagate_complex
};


static MY_COLLATION_HANDLER my_collation_utf8_general_mysql500_ci_handler =
{
    NULL,               /* init */
    my_strnncoll_utf8_general_mysql500_ci,
    my_strnncollsp_utf8_general_mysql500_ci,
    my_strnxfrm_utf8_general_mysql500_ci,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8,
    my_strcasecmp_utf8,
    my_instr_mb,
    my_hash_sort_utf8,
    my_propagate_complex
};


static MY_COLLATION_HANDLER my_collation_utf8_bin_handler =
{
    NULL,		/* init */
    my_strnncoll_utf8_bin,
    my_strnncollsp_utf8_bin,
    my_strnxfrm_utf8_bin,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_mb_bin,
    my_strcasecmp_mb_bin,
    my_instr_mb,
    my_hash_sort_mb_bin,
    my_propagate_simple
};


static MY_COLLATION_HANDLER my_collation_utf8_general_nopad_ci_handler =
{
  NULL,               /* init */
  my_strnncoll_utf8_general_ci,
  my_strnncollsp_utf8_general_nopad_ci,
  my_strnxfrm_nopad_utf8_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8,
  my_strcasecmp_utf8,
  my_instr_mb,
  my_hash_sort_utf8_nopad,
  my_propagate_complex
};


static MY_COLLATION_HANDLER my_collation_utf8_nopad_bin_handler =
{
  NULL,		/* init */
  my_strnncoll_utf8_bin,
  my_strnncollsp_utf8_nopad_bin,
  my_strnxfrm_nopad_utf8_bin,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_mb_bin,
  my_strcasecmp_mb_bin,
  my_instr_mb,
  my_hash_sort_mb_nopad_bin,
  my_propagate_simple
};


MY_CHARSET_HANDLER my_charset_utf8_handler=
{
    NULL,               /* init */
    my_numchars_mb,
    my_charpos_mb,
    my_lengthsp_8bit,
    my_numcells_mb,
    my_utf8_uni,
    my_uni_utf8,
    my_mb_ctype_mb,
    my_caseup_str_utf8,
    my_casedn_str_utf8,
    my_caseup_utf8,
    my_casedn_utf8,
    my_snprintf_8bit,
    my_long10_to_str_8bit,
    my_longlong10_to_str_8bit,
    my_fill_utf8,
    my_strntol_8bit,
    my_strntoul_8bit,
    my_strntoll_8bit,
    my_strntoull_8bit,
    my_strntod_8bit,
    my_strtoll10_8bit,
    my_strntoull10rnd_8bit,
    my_scan_8bit,
    my_charlen_utf8,
    my_well_formed_char_length_utf8,
    my_copy_fix_mb,
    my_uni_utf8,
};



struct charset_info_st my_charset_utf8_general_ci=
{
    33,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE,  /* state  */
    "utf8",             /* cs name      */
    "utf8_general_ci",  /* name         */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8,         /* ctype        */
    to_lower_utf8,      /* to_lower     */
    to_upper_utf8,      /* to_upper     */
    to_upper_utf8,      /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_unicase_default,/* caseinfo     */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* caseup_multiply  */
    1,                  /* casedn_multiply  */
    1,                  /* mbminlen     */
    3,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    1,                  /* levels_for_order   */
    &my_charset_utf8_handler,
    &my_collation_utf8_general_ci_handler
};


struct charset_info_st my_charset_utf8_general_mysql500_ci=
{
  223,0,0,                                      /* number           */
  MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE,  /* state            */
  "utf8",                                       /* cs name          */
  "utf8_general_mysql500_ci",                   /* name             */
  "",                                           /* comment          */
  NULL,                                         /* tailoring        */
  ctype_utf8,                                   /* ctype            */
  to_lower_utf8,                                /* to_lower         */
  to_upper_utf8,                                /* to_upper         */
  to_upper_utf8,                                /* sort_order       */
  NULL,                                         /* uca              */
  NULL,                                         /* tab_to_uni       */
  NULL,                                         /* tab_from_uni     */
  &my_unicase_mysql500,                         /* caseinfo         */
  NULL,                                         /* state_map        */
  NULL,                                         /* ident_map        */
  1,                                            /* strxfrm_multiply */
  1,                                            /* caseup_multiply  */
  1,                                            /* casedn_multiply  */
  1,                                            /* mbminlen         */
  3,                                            /* mbmaxlen         */
  0,                                            /* min_sort_char    */
  0xFFFF,                                       /* max_sort_char    */
  ' ',                                          /* pad char         */
  0,                          /* escape_with_backslash_is_dangerous */
  1,                                            /* levels_for_order   */
  &my_charset_utf8_handler,
  &my_collation_utf8_general_mysql500_ci_handler
};


struct charset_info_st my_charset_utf8_bin=
{
    83,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_BINSORT|MY_CS_UNICODE, /* state  */
    "utf8",             /* cs name      */
    "utf8_bin",         /* name         */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8,         /* ctype        */
    to_lower_utf8,      /* to_lower     */
    to_upper_utf8,      /* to_upper     */
    NULL,               /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_unicase_default,/* caseinfo     */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* caseup_multiply  */
    1,                  /* casedn_multiply  */
    1,                  /* mbminlen     */
    3,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    1,                  /* levels_for_order   */
    &my_charset_utf8_handler,
    &my_collation_utf8_bin_handler
};


struct charset_info_st my_charset_utf8_general_nopad_ci=
{
    MY_NOPAD_ID(33),0,0,      /* number           */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_NOPAD, /* state */
    "utf8",                   /* cs name          */
    "utf8_general_nopad_ci",  /* name             */
    "",                       /* comment          */
    NULL,                     /* tailoring        */
    ctype_utf8,               /* ctype            */
    to_lower_utf8,            /* to_lower         */
    to_upper_utf8,            /* to_upper         */
    to_upper_utf8,            /* sort_order       */
    NULL,                     /* uca              */
    NULL,                     /* tab_to_uni       */
    NULL,                     /* tab_from_uni     */
    &my_unicase_default,      /* caseinfo         */
    NULL,                     /* state_map        */
    NULL,                     /* ident_map        */
    1,                        /* strxfrm_multiply */
    1,                        /* caseup_multiply  */
    1,                        /* casedn_multiply  */
    1,                        /* mbminlen         */
    3,                        /* mbmaxlen         */
    0,                        /* min_sort_char    */
    0xFFFF,                   /* max_sort_char    */
    ' ',                      /* pad char         */
    0,                        /* escape_with_backslash_is_dangerous */
    1,                        /* levels_for_order */
    &my_charset_utf8_handler,
    &my_collation_utf8_general_nopad_ci_handler
};


struct charset_info_st my_charset_utf8_nopad_bin=
{
    MY_NOPAD_ID(83),0,0,/* number           */
    MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_BINSORT|MY_CS_UNICODE|MY_CS_NOPAD,
    "utf8",             /* cs name          */
    "utf8_nopad_bin",   /* name             */
    "",                 /* comment          */
    NULL,               /* tailoring        */
    ctype_utf8,         /* ctype            */
    to_lower_utf8,      /* to_lower         */
    to_upper_utf8,      /* to_upper         */
    NULL,               /* sort_order       */
    NULL,               /* uca              */
    NULL,               /* tab_to_uni       */
    NULL,               /* tab_from_uni     */
    &my_unicase_default,/* caseinfo         */
    NULL,               /* state_map        */
    NULL,               /* ident_map        */
    1,                  /* strxfrm_multiply */
    1,                  /* caseup_multiply  */
    1,                  /* casedn_multiply  */
    1,                  /* mbminlen         */
    3,                  /* mbmaxlen         */
    0,                  /* min_sort_char    */
    0xFFFF,             /* max_sort_char    */
    ' ',                /* pad char         */
    0,                  /* escape_with_backslash_is_dangerous */
    1,                  /* levels_for_order */
    &my_charset_utf8_handler,
    &my_collation_utf8_nopad_bin_handler
};


#ifdef HAVE_UTF8_GENERAL_CS

/*
 * These functions bacically do the same as their original, except
 * that they return 0 only when two comparing unicode strings are
 * strictly the same in case-sensitive way.  See "save_diff" local
 * variable to what they actually do.
 */

static int my_strnncoll_utf8_cs(CHARSET_INFO *cs, 
                                const uchar *s, size_t slen,
                                const uchar *t, size_t tlen,
                                my_bool t_is_prefix)
{
  int s_res,t_res;
  my_wc_t s_wc,t_wc;
  const uchar *se=s+slen;
  const uchar *te=t+tlen;
  int save_diff = 0;
  int diff;
  MY_UNICASE_INFO *const *uni_plane= cs->caseinfo;

  while ( s < se && t < te )
  {
    s_res=my_utf8_uni(cs,&s_wc, s, se);
    t_res=my_utf8_uni(cs,&t_wc, t, te);
    
    if ( s_res <= 0 || t_res <= 0 )

    {
      /* Incorrect string, compare by char value */
      return ((int)s[0]-(int)t[0]); 
    }
    
    if ( save_diff == 0 )
    {
      save_diff = ((int)s_wc) - ((int)t_wc);
    }

    my_tosort_unicode(uni_plane, &s_wc, cs->state);
    my_tosort_unicode(uni_plane, &t_wc, cs->state);

    if ( s_wc != t_wc )
    {
      return  ((int) s_wc) - ((int) t_wc);
    }
    
    s+=s_res;
    t+=t_res;
  }
  diff = ( (se-s) - (te-t) );
  return t_is_prefix ? t-te : ((diff == 0) ? save_diff : diff);
}

static int my_strnncollsp_utf8_cs(CHARSET_INFO *cs, 
                                  const uchar *s, size_t slen,
                                  const uchar *t, size_t tlen)
{
  int s_res, t_res, res;
  my_wc_t s_wc, t_wc;
  const uchar *se= s + slen;
  const uchar *te= t + tlen;
  int save_diff= 0;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;

  while ( s < se && t < te )
  {
    s_res=my_utf8_uni(cs,&s_wc, s, se);
    t_res=my_utf8_uni(cs,&t_wc, t, te);
    
    if ( s_res <= 0 || t_res <= 0 )
    {
      /* Incorrect string, compare by char value */
      return ((int)s[0]-(int)t[0]); 
    }
    
    if ( save_diff == 0 )
    {
      save_diff = ((int)s_wc) - ((int)t_wc);
    }

    my_tosort_unicode(uni_plane, &s_wc, cs->state);
    my_tosort_unicode(uni_plane, &t_wc, cs->state);

    if ( s_wc != t_wc )
    {
      return  ((int) s_wc) - ((int) t_wc);
    }
    
    s+=s_res;
    t+=t_res;
  }
  
  slen= se-s;
  tlen= te-t;

  /*
    The following code uses the fact that in UTF-8
    all multibyte characters are greater than space,
    and all multibyte head characters are greater than
    space. It means if we meet a character greater
    than space, it always means that the longer string
    is greater. So we can reuse the same loop from the
    8bit version, without having to process full multibute
    sequences.
  */
  if ((res= slen == tlen ? 0 :
            slen < tlen  ?
              -my_strnncollsp_padspace_bin(t, tlen) :
              my_strnncollsp_padspace_bin(s, slen)))
    return res;
  return save_diff;
}

static MY_COLLATION_HANDLER my_collation_cs_handler =
{
    NULL,		/* init */
    my_strnncoll_utf8_cs,
    my_strnncollsp_utf8_cs,
    my_strnxfrm_utf8_general_ci,
    my_strnxfrmlen_unicode,
    my_like_range_simple,
    my_wildcmp_mb,
    my_strcasecmp_utf8,
    my_instr_mb,
    my_hash_sort_utf8,
    my_propagate_simple
};

struct charset_info_st my_charset_utf8_general_cs=
{
    254,0,0,		/* number       */
    MY_CS_COMPILED|MY_CS_UNICODE,	/* state  */
    "utf8",		/* cs name      */
    "utf8_general_cs",	/* name         */
    "",			/* comment      */
    NULL,		/* tailoring    */
    ctype_utf8,		/* ctype        */
    to_lower_utf8,	/* to_lower     */
    to_upper_utf8,	/* to_upper     */
    to_upper_utf8,	/* sort_order   */
    NULL,		/* uca          */
    NULL,		/* tab_to_uni   */
    NULL,		/* tab_from_uni */
    &my_unicase_default,/* caseinfo     */
    NULL,		/* state_map    */
    NULL,		/* ident_map    */
    1,			/* strxfrm_multiply */
    1,                  /* caseup_multiply  */
    1,                  /* casedn_multiply  */
    1,			/* mbminlen     */
    3,			/* mbmaxlen     */
    0,			/* min_sort_char */
    255,		/* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    1,                  /* levels_for_order   */
    &my_charset_utf8_handler,
    &my_collation_cs_handler
};
#endif	/* Cybozu Hack */


/*
  File system encoding components:

Code range Pattern            Number   Used Unused  Blocks
-----------------------------------------------------------------------------
00C0..017F [.][0..4][g..z] 5*20= 100   97     3  Latin1 Supplement + Ext A
0370..03FF [.][5..9][g..z] 5*20= 100   88    12  Greek + Coptic
0400..052F [.][g..z][0..6] 20*7= 140  140   137  Cyrillic
0530..058F [.][g..z][7..8] 20*2=  40   38     2  Armenian
2160..217F [.][g..z][9]    20*1=  20   16     4  Number Forms
0180..02AF [.][g..z][a..k] 28*11=220  203    17  Latin Ext B + IPA
1E00..0EFF [.][g..z][l..r] 20*7= 140  136     4  Latin Additional Extended
1F00..1FFF [.][g..z][s..z] 20*8= 160  144    16  Greek Extended
....  .... [.][a..f][g..z] 6*20= 120    0   120  RESERVED
24B6..24E9 [.][@][a..z]           26   26     0  Enclosed Alphanumerics
FF21..FF5A [.][a..z][@]           26   26     0  Full Width forms

All other characters are encoded using five bytes:

[.][0..9a..z][0..9a..z][0..9a..z][0..9a..z]

*/


static const uint16 touni[5994]=
{
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00C0,
  0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x00C7,0x00C8,
  0x00C9,0x00CA,0x00CB,0x00CC,0x00CD,0x00CE,0x00CF,0x00D0,
  0x00D1,0x00D2,0x00D3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00E0,
  0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x00E7,0x00E8,
  0x00E9,0x00EA,0x00EB,0x00EC,0x00ED,0x00EE,0x00EF,0x00F0,
  0x00F1,0x00F2,0x00F3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00D4,
  0x00D5,0x00D6,0x0000,0x00D8,0x00D9,0x00DA,0x00DB,0x00DC,
  0x00DD,0x00DE,0x0178,0x0100,0x0102,0x0104,0x0106,0x0108,
  0x010A,0x010C,0x010E,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00F4,
  0x00F5,0x00F6,0x00DF,0x00F8,0x00F9,0x00FA,0x00FB,0x00FC,
  0x00FD,0x00FE,0x00FF,0x0101,0x0103,0x0105,0x0107,0x0109,
  0x010B,0x010D,0x010F,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0110,
  0x0112,0x0114,0x0116,0x0118,0x011A,0x011C,0x011E,0x0120,
  0x0122,0x0124,0x0126,0x0128,0x012A,0x012C,0x012E,0x0000,
  0x0132,0x0134,0x0136,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0111,
  0x0113,0x0115,0x0117,0x0119,0x011B,0x011D,0x011F,0x0121,
  0x0123,0x0125,0x0127,0x0129,0x012B,0x012D,0x012F,0x0131,
  0x0133,0x0135,0x0137,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0139,0x013B,0x013D,0x013F,0x0141,0x0143,0x0145,0x0147,
  0x0000,0x014A,0x014C,0x014E,0x0150,0x0152,0x0154,0x0156,
  0x0158,0x015A,0x015C,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0138,
  0x013A,0x013C,0x013E,0x0140,0x0142,0x0144,0x0146,0x0148,
  0x0149,0x014B,0x014D,0x014F,0x0151,0x0153,0x0155,0x0157,
  0x0159,0x015B,0x015D,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x015E,
  0x0160,0x0162,0x0164,0x0166,0x0168,0x016A,0x016C,0x016E,
  0x0170,0x0172,0x0174,0x0176,0x0179,0x017B,0x017D,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x015F,
  0x0161,0x0163,0x0165,0x0167,0x0169,0x016B,0x016D,0x016F,
  0x0171,0x0173,0x0175,0x0177,0x017A,0x017C,0x017E,0x017F,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0390,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0386,
  0x0388,0x0389,0x038A,0x0000,0x0391,0x0000,0x0393,0x0394,
  0x0395,0x0396,0x0397,0x0000,0x0399,0x0000,0x039B,0x039C,
  0x039D,0x039E,0x039F,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03AC,
  0x03AD,0x03AE,0x03AF,0x03B0,0x03B1,0x03B2,0x03B3,0x03B4,
  0x03B5,0x03B6,0x03B7,0x03B8,0x03B9,0x03BA,0x03BB,0x03BC,
  0x03BD,0x03BE,0x03BF,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x03A1,0x0000,0x0000,0x03A4,0x03A5,0x0000,0x03A7,0x03A8,
  0x03A9,0x03AA,0x03AB,0x038C,0x038E,0x038F,0x0000,0x0392,
  0x0398,0x03D2,0x03D3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03C0,
  0x03C1,0x03C2,0x03C3,0x03C4,0x03C5,0x03C6,0x03C7,0x03C8,
  0x03C9,0x03CA,0x03CB,0x03CC,0x03CD,0x03CE,0x0000,0x03D0,
  0x03D1,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03D4,
  0x03A6,0x03A0,0x0000,0x0000,0x03DA,0x03DC,0x03DE,0x03E0,
  0x03E2,0x03E4,0x03E6,0x03E8,0x03EA,0x03EC,0x03EE,0x039A,
  0x0000,0x03A3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x03D5,0x03D6,0x03D7,0x03D9,0x03DB,0x03DD,0x03DF,0x03E1,
  0x03E3,0x03E5,0x03E7,0x03E9,0x03EB,0x03ED,0x03EF,0x03F0,
  0x03F1,0x03F2,0x03F3,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x03FD,0x03FE,0x03FF,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03F5,
  0x03F6,0x03F8,0x03FB,0x03FC,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x24B6,0x24B7,0x24B8,0x24B9,0x24BA,0x24BB,0x24BC,
  0x24BD,0x24BE,0x24BF,0x24C0,0x24C1,0x24C2,0x24C3,0x24C4,
  0x24C5,0x24C6,0x24C7,0x24C8,0x24C9,0x24CA,0x24CB,0x24CC,
  0x24CD,0x24CE,0x24CF,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x24D0,0x24D1,0x24D2,0x24D3,0x24D4,0x24D5,0x24D6,
  0x24D7,0x24D8,0x24D9,0x24DA,0x24DB,0x24DC,0x24DD,0x24DE,
  0x24DF,0x24E0,0x24E1,0x24E2,0x24E3,0x24E4,0x24E5,0x24E6,
  0x24E7,0x24E8,0x24E9,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF21,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF22,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF23,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF24,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF25,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF26,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0410,0x0424,0x0408,0x0478,0x04A6,0x04CD,0x04F4,0x0000,
  0x0544,0x2160,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF27,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01B3,0x01DE,0x0208,0x0230,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E00,0x1E28,0x1E50,0x1E78,
  0x1E60,0x1EBE,0x1EE6,0x1F08,0x1F2A,0x0000,0x1F6C,0x1F88,
  0x1FAC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0411,0x0425,0x0409,0x047A,0x04A8,0x0000,0x04F6,0x0531,
  0x0545,0x2161,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF28,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0182,0x01B5,0x01E0,0x020A,0x0232,0x0000,0x0000,
  0x019D,0x0000,0x0000,0x0000,0x1E02,0x1E2A,0x1E52,0x1E7A,
  0x0000,0x1EC0,0x1EE8,0x1F09,0x1F2B,0x0000,0x1F6D,0x1F89,
  0x1FAD,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0412,0x0426,0x040A,0x047C,0x04AA,0x04D0,0x04F8,0x0532,
  0x0546,0x2162,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF29,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0184,0x01B8,0x01E2,0x020C,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E04,0x1E2C,0x1E54,0x1E7C,
  0x0000,0x1EC2,0x1EEA,0x1F0A,0x1F2C,0x0000,0x1F6E,0x1F8A,
  0x1FAE,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0413,0x0427,0x040B,0x047E,0x04AC,0x04D2,0x0000,0x0533,
  0x0547,0x2163,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0187,0x0000,0x01E4,0x020E,0x0000,0x0000,0x0193,
  0x0000,0x01AE,0x0000,0x0000,0x1E06,0x1E2E,0x1E56,0x1E7E,
  0x0000,0x1EC4,0x1EEC,0x1F0B,0x1F2D,0x0000,0x1F6F,0x1F8B,
  0x1FAF,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0414,0x0428,0x040C,0x0480,0x04AE,0x04D4,0x0000,0x0534,
  0x0548,0x2164,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018B,0x0000,0x01E6,0x0210,0x0000,0x0000,0x0000,
  0x019F,0x0000,0x0000,0x0000,0x1E08,0x1E30,0x1E58,0x1E80,
  0x0000,0x1EC6,0x1EEE,0x1F0C,0x1F2E,0x0000,0x1FBA,0x1F8C,
  0x1FB8,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0415,0x0429,0x040D,0x0000,0x04B0,0x04D6,0x0000,0x0535,
  0x0549,0x2165,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x01E8,0x0212,0x0000,0x0000,0x0000,
  0x0000,0x01B1,0x0000,0x0000,0x1E0A,0x1E32,0x1E5A,0x1E82,
  0x1EA0,0x1EC8,0x1EF0,0x1F0D,0x1F2F,0x1F59,0x1FBB,0x1F8D,
  0x1FB9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0416,0x042A,0x040E,0x048A,0x04B2,0x04D8,0x0000,0x0536,
  0x054A,0x2166,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0191,0x0000,0x01EA,0x0214,0x0000,0x0000,0x0194,
  0x0000,0x01B2,0x0000,0x0000,0x1E0C,0x1E34,0x1E5C,0x1E84,
  0x1EA2,0x1ECA,0x1EF2,0x1F0E,0x1F38,0x0000,0x1FC8,0x1F8E,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0417,0x042B,0x040F,0x048C,0x04B4,0x04DA,0x0000,0x0537,
  0x054B,0x2167,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01F6,0x01BC,0x01EC,0x0216,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E0E,0x1E36,0x1E5E,0x1E86,
  0x1EA4,0x1ECC,0x1EF4,0x1F0F,0x1F39,0x1F5B,0x1FC9,0x1F8F,
  0x1FBC,0x1FE8,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0418,0x042C,0x0460,0x048E,0x04B6,0x04DC,0x0000,0x0538,
  0x054C,0x2168,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF2F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0198,0x01C4,0x01EE,0x0218,0x023A,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E10,0x1E38,0x0000,0x1E88,
  0x1EA6,0x1ECE,0x1EF6,0x1F18,0x1F3A,0x0000,0x1FCA,0x1F98,
  0x0000,0x1FE9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0419,0x042D,0x0462,0x0490,0x04B8,0x04DE,0x0500,0x0539,
  0x054D,0x2169,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF30,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x023D,0x01C7,0x0000,0x021A,0x023B,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E12,0x1E3A,0x1E62,0x1E8A,
  0x1EA8,0x1ED0,0x1EF8,0x1F19,0x1F3B,0x1F5D,0x1FCB,0x1F99,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041A,0x042E,0x0464,0x0492,0x04BA,0x04E0,0x0502,0x053A,
  0x054E,0x216A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF31,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01CA,0x01F1,0x021C,0x023E,0x0181,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E14,0x1E3C,0x1E64,0x1E8C,
  0x1EAA,0x1ED2,0x0000,0x1F1A,0x1F3C,0x0000,0x1FDA,0x1F9A,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041B,0x042F,0x0466,0x0494,0x04BC,0x04E2,0x0504,0x053B,
  0x054F,0x216B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF32,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0220,0x01CD,0x01F4,0x021E,0x0000,0x0186,0x0197,
  0x0000,0x0000,0x0000,0x0000,0x1E16,0x1E3E,0x1E66,0x1E8E,
  0x1EAC,0x1ED4,0x0000,0x1F1B,0x1F3D,0x1F5F,0x1FDB,0x1F9B,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041C,0x0400,0x0468,0x0496,0x04BE,0x04E4,0x0506,0x053C,
  0x0550,0x216C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF33,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A0,0x01CF,0x01F8,0x0000,0x0000,0x0000,0x0196,
  0x0000,0x0000,0x0000,0x0000,0x1E18,0x1E40,0x1E68,0x1E90,
  0x1EAE,0x1ED6,0x0000,0x1F1C,0x1F3E,0x0000,0x1FF8,0x1F9C,
  0x0000,0x1FEC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041D,0x0401,0x046A,0x0498,0x04C0,0x04E6,0x0508,0x053D,
  0x0551,0x216D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF34,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A2,0x01D1,0x01FA,0x0222,0x0000,0x0189,0x0000,
  0x0000,0x01B7,0x0000,0x0000,0x1E1A,0x1E42,0x1E6A,0x1E92,
  0x1EB0,0x1ED8,0x0000,0x1F1D,0x1F3F,0x0000,0x1FF9,0x1F9D,
  0x1FCC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041E,0x0402,0x046C,0x049A,0x04C1,0x04E8,0x050A,0x053E,
  0x0552,0x216E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF35,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A4,0x01D3,0x01FC,0x0224,0x0000,0x018A,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E1C,0x1E44,0x1E6C,0x1E94,
  0x1EB2,0x1EDA,0x0000,0x0000,0x1F48,0x0000,0x1FEA,0x1F9E,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x041F,0x0403,0x046E,0x049C,0x04C3,0x04EA,0x050C,0x053F,
  0x0553,0x216F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF36,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A7,0x01D5,0x01FE,0x0226,0x0000,0x0000,0x0000,
  0x01A6,0x0241,0x0000,0x0000,0x1E1E,0x1E46,0x1E6E,0x0000,
  0x1EB4,0x1EDC,0x0000,0x0000,0x1F49,0x0000,0x1FEB,0x1F9F,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0420,0x0404,0x0470,0x049E,0x04C5,0x04EC,0x050E,0x0540,
  0x0554,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF37,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01D7,0x0200,0x0228,0x0000,0x018F,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E20,0x1E48,0x1E70,0x0000,
  0x1EB6,0x1EDE,0x0000,0x0000,0x1F4A,0x1F68,0x1FFA,0x1FA8,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0421,0x0405,0x0472,0x04A0,0x04C7,0x04EE,0x0000,0x0541,
  0x0555,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF38,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x01D9,0x0202,0x022A,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E22,0x1E4A,0x1E72,0x0000,
  0x1EB8,0x1EE0,0x0000,0x0000,0x1F4B,0x1F69,0x1FFB,0x1FA9,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0422,0x0406,0x0474,0x04A2,0x04C9,0x04F0,0x0000,0x0542,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF39,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AC,0x01DB,0x0204,0x022C,0x0000,0x0190,0x019C,
  0x01A9,0x0000,0x0000,0x0000,0x1E24,0x1E4C,0x1E74,0x0000,
  0x1EBA,0x1EE2,0x0000,0x1F28,0x1F4C,0x1F6A,0x0000,0x1FAA,
  0x1FD8,0x1FFC,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0423,0x0407,0x0476,0x04A4,0x04CB,0x04F2,0x0000,0x0543,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF3A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AF,0x018E,0x0206,0x022E,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x1E26,0x1E4E,0x1E76,0x0000,
  0x1EBC,0x1EE4,0x0000,0x1F29,0x1F4D,0x1F6B,0x0000,0x1FAB,
  0x1FD9,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF41,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF42,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF43,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF44,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF45,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF46,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0430,0x0444,0x0458,0x0479,0x04A7,0x04CE,0x04F5,0x0000,
  0x0574,0x2170,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF47,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0180,0x01B4,0x01DF,0x0209,0x0231,0x0000,0x025D,
  0x0271,0x0285,0x0299,0x02AD,0x1E01,0x1E29,0x1E51,0x1E79,
  0x1E9B,0x1EBF,0x1EE7,0x1F00,0x1F22,0x0000,0x1F64,0x1F80,
  0x1FA4,0x1FD2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0431,0x0445,0x0459,0x047B,0x04A9,0x0000,0x04F7,0x0561,
  0x0575,0x2171,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF48,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0183,0x01B6,0x01E1,0x020B,0x0233,0x0000,0x025E,
  0x0272,0x0286,0x029A,0x02AE,0x1E03,0x1E2B,0x1E53,0x1E7B,
  0x0000,0x1EC1,0x1EE9,0x1F01,0x1F23,0x0000,0x1F65,0x1F81,
  0x1FA5,0x1FD3,0x1FF6,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0432,0x0446,0x045A,0x047D,0x04AB,0x04D1,0x04F9,0x0562,
  0x0576,0x2172,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF49,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0185,0x01B9,0x01E3,0x020D,0x0234,0x0000,0x025F,
  0x0273,0x0287,0x029B,0x02AF,0x1E05,0x1E2D,0x1E55,0x1E7D,
  0x0000,0x1EC3,0x1EEB,0x1F02,0x1F24,0x0000,0x1F66,0x1F82,
  0x1FA6,0x0000,0x1FF7,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0433,0x0447,0x045B,0x047F,0x04AD,0x04D3,0x0000,0x0563,
  0x0577,0x2173,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0188,0x01BA,0x01E5,0x020F,0x0235,0x0000,0x0260,
  0x0274,0x0288,0x029C,0x0000,0x1E07,0x1E2F,0x1E57,0x1E7F,
  0x0000,0x1EC5,0x1EED,0x1F03,0x1F25,0x0000,0x1F67,0x1F83,
  0x1FA7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0434,0x0448,0x045C,0x0481,0x04AF,0x04D5,0x0000,0x0564,
  0x0578,0x2174,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018C,0x01BB,0x01E7,0x0211,0x0236,0x0000,0x0261,
  0x0275,0x0289,0x029D,0x0000,0x1E09,0x1E31,0x1E59,0x1E81,
  0x0000,0x1EC7,0x1EEF,0x1F04,0x1F26,0x1F50,0x1F70,0x1F84,
  0x1FB0,0x1FD6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0435,0x0449,0x045D,0x0000,0x04B1,0x04D7,0x0000,0x0565,
  0x0579,0x2175,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x018D,0x0000,0x01E9,0x0213,0x0237,0x0000,0x0262,
  0x0276,0x028A,0x029E,0x0000,0x1E0B,0x1E33,0x1E5B,0x1E83,
  0x1EA1,0x1EC9,0x1EF1,0x1F05,0x1F27,0x1F51,0x1F71,0x1F85,
  0x1FB1,0x1FD7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0436,0x044A,0x045E,0x048B,0x04B3,0x04D9,0x0000,0x0566,
  0x057A,0x2176,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0192,0x01BE,0x01EB,0x0215,0x0238,0x0000,0x0263,
  0x0277,0x028B,0x029F,0x0000,0x1E0D,0x1E35,0x1E5D,0x1E85,
  0x1EA3,0x1ECB,0x1EF3,0x1F06,0x1F30,0x1F52,0x1F72,0x1F86,
  0x1FB2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0437,0x044B,0x045F,0x048D,0x04B5,0x04DB,0x0000,0x0567,
  0x057B,0x2177,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0195,0x01BD,0x01ED,0x0217,0x0239,0x0250,0x0264,
  0x0278,0x028C,0x02A0,0x0000,0x1E0F,0x1E37,0x1E5F,0x1E87,
  0x1EA5,0x1ECD,0x1EF5,0x1F07,0x1F31,0x1F53,0x1F73,0x1F87,
  0x1FB3,0x1FE0,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0438,0x044C,0x0461,0x048F,0x04B7,0x04DD,0x0000,0x0568,
  0x057C,0x2178,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF4F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0199,0x01C6,0x01EF,0x0219,0x0000,0x0251,0x0265,
  0x0279,0x028D,0x02A1,0x0000,0x1E11,0x1E39,0x1E61,0x1E89,
  0x1EA7,0x1ECF,0x1EF7,0x1F10,0x1F32,0x1F54,0x1F74,0x1F90,
  0x1FB4,0x1FE1,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0439,0x044D,0x0463,0x0491,0x04B9,0x04DF,0x0501,0x0569,
  0x057D,0x2179,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF50,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019A,0x01C9,0x01F0,0x021B,0x023C,0x0252,0x0266,
  0x027A,0x028E,0x02A2,0x0000,0x1E13,0x1E3B,0x1E63,0x1E8B,
  0x1EA9,0x1ED1,0x1EF9,0x1F11,0x1F33,0x1F55,0x1F75,0x1F91,
  0x0000,0x1FE2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043A,0x044E,0x0465,0x0493,0x04BB,0x04E1,0x0503,0x056A,
  0x057E,0x217A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF51,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019B,0x01CC,0x01F3,0x021D,0x0000,0x0253,0x0267,
  0x027B,0x028F,0x02A3,0x0000,0x1E15,0x1E3D,0x1E65,0x1E8D,
  0x1EAB,0x1ED3,0x0000,0x1F12,0x1F34,0x1F56,0x1F76,0x1F92,
  0x1FB6,0x1FE3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043B,0x044F,0x0467,0x0495,0x04BD,0x04E3,0x0505,0x056B,
  0x057F,0x217B,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF52,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x019E,0x01CE,0x01F5,0x021F,0x023F,0x0254,0x0268,
  0x027C,0x0290,0x02A4,0x0000,0x1E17,0x1E3F,0x1E67,0x1E8F,
  0x1EAD,0x1ED5,0x0000,0x1F13,0x1F35,0x1F57,0x1F77,0x1F93,
  0x1FB7,0x1FE4,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043C,0x0450,0x0469,0x0497,0x04BF,0x04E5,0x0507,0x056C,
  0x0580,0x217C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF53,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A1,0x01D0,0x01F9,0x0221,0x0240,0x0255,0x0269,
  0x027D,0x0291,0x02A5,0x0000,0x1E19,0x1E41,0x1E69,0x1E91,
  0x1EAF,0x1ED7,0x0000,0x1F14,0x1F36,0x0000,0x1F78,0x1F94,
  0x1FC2,0x1FE5,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043D,0x0451,0x046B,0x0499,0x0000,0x04E7,0x0509,0x056D,
  0x0581,0x217D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF54,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A3,0x01D2,0x01FB,0x0223,0x0000,0x0256,0x026A,
  0x027E,0x0292,0x02A6,0x0000,0x1E1B,0x1E43,0x1E6B,0x1E93,
  0x1EB1,0x1ED9,0x0000,0x1F15,0x1F37,0x0000,0x1F79,0x1F95,
  0x1FC3,0x1FE6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043E,0x0452,0x046D,0x049B,0x04C2,0x04E9,0x050B,0x056E,
  0x0582,0x217E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF55,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A5,0x01D4,0x01FD,0x0225,0x0000,0x0257,0x026B,
  0x027F,0x0293,0x02A7,0x0000,0x1E1D,0x1E45,0x1E6D,0x1E95,
  0x1EB3,0x1EDB,0x0000,0x0000,0x1F40,0x0000,0x1F7A,0x1F96,
  0x1FC4,0x1FE7,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x043F,0x0453,0x046F,0x049D,0x04C4,0x04EB,0x050D,0x056F,
  0x0583,0x217F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF56,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01A8,0x01D6,0x01FF,0x0227,0x0000,0x0258,0x026C,
  0x0280,0x0294,0x02A8,0x0000,0x1E1F,0x1E47,0x1E6F,0x1E96,
  0x1EB5,0x1EDD,0x0000,0x0000,0x1F41,0x0000,0x1F7B,0x1F97,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0440,0x0454,0x0471,0x049F,0x04C6,0x04ED,0x050F,0x0570,
  0x0584,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF57,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AA,0x01D8,0x0201,0x0229,0x0000,0x0259,0x026D,
  0x0281,0x0295,0x02A9,0x0000,0x1E21,0x1E49,0x1E71,0x1E97,
  0x1EB7,0x1EDF,0x0000,0x0000,0x1F42,0x1F60,0x1F7C,0x1FA0,
  0x1FC6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0441,0x0455,0x0473,0x04A1,0x04C8,0x04EF,0x0000,0x0571,
  0x0585,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF58,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AB,0x01DA,0x0203,0x022B,0x0000,0x025A,0x026E,
  0x0282,0x0296,0x02AA,0x0000,0x1E23,0x1E4B,0x1E73,0x1E98,
  0x1EB9,0x1EE1,0x0000,0x0000,0x1F43,0x1F61,0x1F7D,0x1FA1,
  0x1FC7,0x1FF2,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0442,0x0456,0x0475,0x04A3,0x04CA,0x04F1,0x0000,0x0572,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF59,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01AD,0x01DC,0x0205,0x022D,0x0000,0x025B,0x026F,
  0x0283,0x0297,0x02AB,0x0000,0x1E25,0x1E4D,0x1E75,0x1E99,
  0x1EBB,0x1EE3,0x0000,0x1F20,0x1F44,0x1F62,0x0000,0x1FA2,
  0x1FD0,0x1FF3,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0443,0x0457,0x0477,0x04A5,0x04CC,0x04F3,0x0000,0x0573,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0xFF5A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x01B0,0x01DD,0x0207,0x022F,0x0000,0x025C,0x0270,
  0x0284,0x0298,0x02AC,0x0000,0x1E27,0x1E4F,0x1E77,0x1E9A,
  0x1EBD,0x1EE5,0x0000,0x1F21,0x1F45,0x1F63,0x0000,0x1FA3,
  0x1FD1,0x1FF4
};


/* 00C0-05FF */
static const uint16 uni_0C00_05FF[1344]=
{
  0x0017,0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,
  0x001F,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,
  0x0027,0x0028,0x0029,0x002A,0x0067,0x0068,0x0069,0x0000,
  0x006B,0x006C,0x006D,0x006E,0x006F,0x0070,0x0071,0x008A,
  0x0037,0x0038,0x0039,0x003A,0x003B,0x003C,0x003D,0x003E,
  0x003F,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,
  0x0047,0x0048,0x0049,0x004A,0x0087,0x0088,0x0089,0x0000,
  0x008B,0x008C,0x008D,0x008E,0x008F,0x0090,0x0091,0x0092,
  0x0073,0x0093,0x0074,0x0094,0x0075,0x0095,0x0076,0x0096,
  0x0077,0x0097,0x0078,0x0098,0x0079,0x0099,0x007A,0x009A,
  0x00B7,0x00D7,0x00B8,0x00D8,0x00B9,0x00D9,0x00BA,0x00DA,
  0x00BB,0x00DB,0x00BC,0x00DC,0x00BD,0x00DD,0x00BE,0x00DE,
  0x00BF,0x00DF,0x00C0,0x00E0,0x00C1,0x00E1,0x00C2,0x00E2,
  0x00C3,0x00E3,0x00C4,0x00E4,0x00C5,0x00E5,0x00C6,0x00E6,
  0x0000,0x00E7,0x00C8,0x00E8,0x00C9,0x00E9,0x00CA,0x00EA,
  0x0127,0x0108,0x0128,0x0109,0x0129,0x010A,0x012A,0x010B,
  0x012B,0x010C,0x012C,0x010D,0x012D,0x010E,0x012E,0x010F,
  0x012F,0x0130,0x0111,0x0131,0x0112,0x0132,0x0113,0x0133,
  0x0114,0x0134,0x0115,0x0135,0x0116,0x0136,0x0117,0x0137,
  0x0118,0x0138,0x0119,0x0139,0x011A,0x013A,0x0157,0x0177,
  0x0158,0x0178,0x0159,0x0179,0x015A,0x017A,0x015B,0x017B,
  0x015C,0x017C,0x015D,0x017D,0x015E,0x017E,0x015F,0x017F,
  0x0160,0x0180,0x0161,0x0181,0x0162,0x0182,0x0163,0x0183,
  0x0072,0x0164,0x0184,0x0165,0x0185,0x0166,0x0186,0x0187,
  0x1161,0x0A86,0x07B1,0x11B1,0x0801,0x1201,0x0AD6,0x0851,
  0x1251,0x0B76,0x0BC6,0x08A1,0x12A1,0x12F1,0x0D52,0x0C66,
  0x0D06,0x0941,0x1341,0x0857,0x0947,0x1391,0x0B27,0x0AD7,
  0x09E1,0x13E1,0x1431,0x1481,0x0D07,0x07B8,0x14D1,0x08A8,
  0x0B21,0x1521,0x0B71,0x1571,0x0BC1,0x15C1,0x0C18,0x0C11,
  0x1611,0x0D08,0x1661,0x16B1,0x0D01,0x1701,0x0859,0x0D51,
  0x1751,0x08F9,0x0949,0x0762,0x1162,0x07B2,0x11B2,0x0B79,
  0x0802,0x1202,0x1252,0x12A2,0x0992,0x1392,0x1342,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x09E2,0x0000,0x13E2,0x0A32,
  0x0000,0x1432,0x0A82,0x0000,0x1482,0x0AD2,0x14D2,0x0B22,
  0x1522,0x0B72,0x1572,0x0BC2,0x15C2,0x0C12,0x1612,0x0C62,
  0x1662,0x0CB2,0x16B2,0x0D02,0x1702,0x1752,0x0763,0x1163,
  0x07B3,0x11B3,0x0803,0x1203,0x0853,0x1253,0x08A3,0x12A3,
  0x08F3,0x12F3,0x0943,0x1343,0x0993,0x1393,0x09E3,0x13E3,
  0x1433,0x0A83,0x0000,0x1483,0x0AD3,0x14D3,0x0991,0x0000,
  0x0B23,0x1523,0x0B73,0x1573,0x0BC3,0x15C3,0x0C13,0x1613,
  0x0C63,0x1663,0x0CB3,0x16B3,0x0D03,0x1703,0x0D53,0x1753,
  0x0764,0x1164,0x07B4,0x11B4,0x0804,0x1204,0x0854,0x1254,
  0x08A4,0x12A4,0x08F4,0x12F4,0x0944,0x1344,0x0994,0x1394,
  0x09E4,0x13E4,0x0A34,0x1434,0x0A84,0x1484,0x0AD4,0x14D4,
  0x0AD1,0x1524,0x0B74,0x1574,0x0BC4,0x15C4,0x0C14,0x1614,
  0x0C64,0x1664,0x0CB4,0x16B4,0x0D04,0x1704,0x0D54,0x1754,
  0x0765,0x1165,0x07B5,0x11B5,0x1205,0x1255,0x12A5,0x12F5,
  0x1345,0x1395,0x09E5,0x0A35,0x1435,0x0A31,0x0A85,0x14D5,
  0x1525,0x0C19,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x1396,0x13E6,0x1436,0x1486,0x14D6,0x1526,0x1576,0x15C6,
  0x1616,0x1666,0x16B6,0x1706,0x1756,0x1167,0x11B7,0x1207,
  0x1257,0x12A7,0x12F7,0x1347,0x1397,0x13E7,0x1437,0x1487,
  0x14D7,0x1527,0x1577,0x15C7,0x1617,0x1667,0x16B7,0x1707,
  0x1757,0x1168,0x11B8,0x1208,0x1258,0x12A8,0x12F8,0x1348,
  0x1398,0x13E8,0x1438,0x1488,0x14D8,0x1528,0x1578,0x15C8,
  0x1618,0x1668,0x16B8,0x1708,0x1758,0x1169,0x11B9,0x1209,
  0x1259,0x12A9,0x12F9,0x1349,0x1399,0x13E9,0x1439,0x1489,
  0x14D9,0x1529,0x1579,0x15C9,0x1619,0x1669,0x16B9,0x1709,
  0x1759,0x116A,0x11BA,0x120A,0x125A,0x12AA,0x12FA,0x134A,
  0x139A,0x13EA,0x143A,0x148A,0x14DA,0x152A,0x157A,0x15CA,
  0x161A,0x166A,0x16BA,0x170A,0x175A,0x116B,0x11BB,0x120B,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x01F7,0x0000,
  0x01F8,0x01F9,0x01FA,0x0000,0x0253,0x0000,0x0254,0x0255,
  0x01D9,0x01FC,0x0257,0x01FE,0x01FF,0x0200,0x0201,0x0202,
  0x0258,0x0204,0x02A7,0x0206,0x0207,0x0208,0x0209,0x020A,
  0x0299,0x0248,0x0000,0x02A9,0x024B,0x024C,0x0298,0x024E,
  0x024F,0x0250,0x0251,0x0252,0x0217,0x0218,0x0219,0x021A,
  0x021B,0x021C,0x021D,0x021E,0x021F,0x0220,0x0221,0x0222,
  0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022A,
  0x0267,0x0268,0x0269,0x026A,0x026B,0x026C,0x026D,0x026E,
  0x026F,0x0270,0x0271,0x0272,0x0273,0x0274,0x0275,0x0000,
  0x0277,0x0278,0x0259,0x025A,0x0297,0x02B8,0x02B9,0x02BA,
  0x0000,0x02BB,0x029C,0x02BC,0x029D,0x02BD,0x029E,0x02BE,
  0x029F,0x02BF,0x02A0,0x02C0,0x02A1,0x02C1,0x02A2,0x02C2,
  0x02A3,0x02C3,0x02A4,0x02C4,0x02A5,0x02C5,0x02A6,0x02C6,
  0x02C7,0x02C8,0x02C9,0x02CA,0x0000,0x0307,0x0308,0x0000,
  0x0309,0x0000,0x0000,0x030A,0x030B,0x02EC,0x02ED,0x02EE,
  0x0AF1,0x0B41,0x0B91,0x0BE1,0x0C31,0x0C81,0x0CD1,0x0D21,
  0x0732,0x0782,0x07D2,0x0822,0x0872,0x08C2,0x0912,0x0962,
  0x0730,0x0780,0x07D0,0x0820,0x0870,0x08C0,0x0910,0x0960,
  0x09B0,0x0A00,0x0A50,0x0AA0,0x0AF0,0x0B40,0x0B90,0x0BE0,
  0x0C30,0x0C80,0x0CD0,0x0D20,0x0731,0x0781,0x07D1,0x0821,
  0x0871,0x08C1,0x0911,0x0961,0x09B1,0x0A01,0x0A51,0x0AA1,
  0x1130,0x1180,0x11D0,0x1220,0x1270,0x12C0,0x1310,0x1360,
  0x13B0,0x1400,0x1450,0x14A0,0x14F0,0x1540,0x1590,0x15E0,
  0x1630,0x1680,0x16D0,0x1720,0x1131,0x1181,0x11D1,0x1221,
  0x1271,0x12C1,0x1311,0x1361,0x13B1,0x1401,0x1451,0x14A1,
  0x14F1,0x1541,0x1591,0x15E1,0x1631,0x1681,0x16D1,0x1721,
  0x1132,0x1182,0x11D2,0x1222,0x1272,0x12C2,0x1312,0x1362,
  0x09B2,0x13B2,0x0A02,0x1402,0x0A52,0x1452,0x0AA2,0x14A2,
  0x0AF2,0x14F2,0x0B42,0x1542,0x0B92,0x1592,0x0BE2,0x15E2,
  0x0C32,0x1632,0x0C82,0x1682,0x0CD2,0x16D2,0x0D22,0x1722,
  0x0733,0x1133,0x0783,0x1183,0x07D3,0x11D3,0x0823,0x1223,
  0x0873,0x1273,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0913,0x1313,0x0963,0x1363,0x09B3,0x13B3,
  0x0A03,0x1403,0x0A53,0x1453,0x0AA3,0x14A3,0x0AF3,0x14F3,
  0x0B43,0x1543,0x0B93,0x1593,0x0BE3,0x15E3,0x0C33,0x1633,
  0x0C83,0x1683,0x0CD3,0x16D3,0x0D23,0x1723,0x0734,0x1134,
  0x0784,0x1184,0x07D4,0x11D4,0x0824,0x1224,0x0874,0x1274,
  0x08C4,0x12C4,0x0914,0x1314,0x0964,0x1364,0x09B4,0x13B4,
  0x0A04,0x1404,0x0A54,0x1454,0x0AA4,0x14A4,0x0AF4,0x14F4,
  0x0B44,0x0B94,0x1594,0x0BE4,0x15E4,0x0C34,0x1634,0x0C84,
  0x1684,0x0CD4,0x16D4,0x0D24,0x1724,0x0735,0x1135,0x0000,
  0x07D5,0x11D5,0x0825,0x1225,0x0875,0x1275,0x08C5,0x12C5,
  0x0915,0x1315,0x0965,0x1365,0x09B5,0x13B5,0x0A05,0x1405,
  0x0A55,0x1455,0x0AA5,0x14A5,0x0AF5,0x14F5,0x0B45,0x1545,
  0x0B95,0x1595,0x0BE5,0x15E5,0x0C35,0x1635,0x0C85,0x1685,
  0x0CD5,0x16D5,0x0D25,0x1725,0x0736,0x1136,0x0786,0x1186,
  0x07D6,0x11D6,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0A06,0x1406,0x0A56,0x1456,0x0AA6,0x14A6,0x0AF6,0x14F6,
  0x0B46,0x1546,0x0B96,0x1596,0x0BE6,0x15E6,0x0C36,0x1636,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0787,0x07D7,0x0827,0x0877,0x08C7,0x0917,0x0967,
  0x09B7,0x0A07,0x0A57,0x0AA7,0x0AF7,0x0B47,0x0B97,0x0BE7,
  0x0C37,0x0C87,0x0CD7,0x0D27,0x0738,0x0788,0x07D8,0x0828,
  0x0878,0x08C8,0x0918,0x0968,0x09B8,0x0A08,0x0A58,0x0AA8,
  0x0AF8,0x0B48,0x0B98,0x0BE8,0x0C38,0x0C88,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x1187,0x11D7,0x1227,0x1277,0x12C7,0x1317,0x1367,
  0x13B7,0x1407,0x1457,0x14A7,0x14F7,0x1547,0x1597,0x15E7,
  0x1637,0x1687,0x16D7,0x1727,0x1138,0x1188,0x11D8,0x1228,
  0x1278,0x12C8,0x1318,0x1368,0x13B8,0x1408,0x1458,0x14A8,
  0x14F8,0x1548,0x1598,0x15E8,0x1638,0x1688,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000
};


/* 1E00-1FFF */
static uint16 uni_1E00_1FFF[512]=
{
  0x076C,0x116C,0x07BC,0x11BC,0x080C,0x120C,0x085C,0x125C,
  0x08AC,0x12AC,0x08FC,0x12FC,0x094C,0x134C,0x099C,0x139C,
  0x09EC,0x13EC,0x0A3C,0x143C,0x0A8C,0x148C,0x0ADC,0x14DC,
  0x0B2C,0x152C,0x0B7C,0x157C,0x0BCC,0x15CC,0x0C1C,0x161C,
  0x0C6C,0x166C,0x0CBC,0x16BC,0x0D0C,0x170C,0x0D5C,0x175C,
  0x076D,0x116D,0x07BD,0x11BD,0x080D,0x120D,0x085D,0x125D,
  0x08AD,0x12AD,0x08FD,0x12FD,0x094D,0x134D,0x099D,0x139D,
  0x09ED,0x13ED,0x0A3D,0x143D,0x0A8D,0x148D,0x0ADD,0x14DD,
  0x0B2D,0x152D,0x0B7D,0x157D,0x0BCD,0x15CD,0x0C1D,0x161D,
  0x0C6D,0x166D,0x0CBD,0x16BD,0x0D0D,0x170D,0x0D5D,0x175D,
  0x076E,0x116E,0x07BE,0x11BE,0x080E,0x120E,0x085E,0x125E,
  0x08AE,0x12AE,0x08FE,0x12FE,0x094E,0x134E,0x099E,0x139E,
  0x0770,0x13EE,0x0A3E,0x143E,0x0A8E,0x148E,0x0ADE,0x14DE,
  0x0B2E,0x152E,0x0B7E,0x157E,0x0BCE,0x15CE,0x0C1E,0x161E,
  0x0C6E,0x166E,0x0CBE,0x16BE,0x0D0E,0x170E,0x0D5E,0x175E,
  0x076F,0x116F,0x07BF,0x11BF,0x080F,0x120F,0x085F,0x125F,
  0x08AF,0x12AF,0x08FF,0x12FF,0x094F,0x134F,0x099F,0x139F,
  0x09EF,0x13EF,0x0A3F,0x143F,0x0A8F,0x148F,0x0ADF,0x14DF,
  0x0B2F,0x152F,0x0B7F,0x157F,0x0BCF,0x15CF,0x161F,0x166F,
  0x16BF,0x170F,0x175F,0x1170,0x0000,0x0000,0x0000,0x0000,
  0x0900,0x1300,0x0950,0x1350,0x09A0,0x13A0,0x09F0,0x13F0,
  0x0A40,0x1440,0x0A90,0x1490,0x0AE0,0x14E0,0x0B30,0x1530,
  0x0B80,0x1580,0x0BD0,0x15D0,0x0C20,0x1620,0x0C70,0x1670,
  0x0CC0,0x16C0,0x0D10,0x1710,0x0D60,0x1760,0x0771,0x1171,
  0x07C1,0x11C1,0x0811,0x1211,0x0861,0x1261,0x08B1,0x12B1,
  0x0901,0x1301,0x0951,0x1351,0x09A1,0x13A1,0x09F1,0x13F1,
  0x0A41,0x1441,0x0A91,0x1491,0x0AE1,0x14E1,0x0B31,0x1531,
  0x0B81,0x1581,0x0BD1,0x15D1,0x0C21,0x1621,0x0C71,0x1671,
  0x0CC1,0x16C1,0x0D11,0x1711,0x0D61,0x1761,0x0772,0x1172,
  0x07C2,0x11C2,0x0812,0x1212,0x0862,0x1262,0x08B2,0x12B2,
  0x0902,0x1302,0x0952,0x1352,0x09A2,0x13A2,0x09F2,0x13F2,
  0x0A42,0x1442,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x1173,0x11C3,0x1213,0x1263,0x12B3,0x1303,0x1353,0x13A3,
  0x0773,0x07C3,0x0813,0x0863,0x08B3,0x0903,0x0953,0x09A3,
  0x13F3,0x1443,0x1493,0x14E3,0x1533,0x1583,0x0000,0x0000,
  0x09F3,0x0A43,0x0A93,0x0AE3,0x0B33,0x0B83,0x0000,0x0000,
  0x1713,0x1763,0x1174,0x11C4,0x1214,0x1264,0x12B4,0x1304,
  0x0D13,0x0D63,0x0774,0x07C4,0x0814,0x0864,0x08B4,0x0904,
  0x1354,0x13A4,0x13F4,0x1444,0x1494,0x14E4,0x1534,0x1584,
  0x0954,0x09A4,0x09F4,0x0A44,0x0A94,0x0AE4,0x0B34,0x0B84,
  0x15D4,0x1624,0x1674,0x16C4,0x1714,0x1764,0x0000,0x0000,
  0x0BD4,0x0C24,0x0C74,0x0CC4,0x0D14,0x0D64,0x0000,0x0000,
  0x12B5,0x1305,0x1355,0x13A5,0x13F5,0x1445,0x1495,0x14E5,
  0x0000,0x0905,0x0000,0x09A5,0x0000,0x0A45,0x0000,0x0AE5,
  0x1675,0x16C5,0x1715,0x1765,0x1176,0x11C6,0x1216,0x1266,
  0x0C75,0x0CC5,0x0D15,0x0D65,0x0776,0x07C6,0x0816,0x0866,
  0x12B6,0x1306,0x1356,0x13A6,0x13F6,0x1446,0x1496,0x14E6,
  0x1536,0x1586,0x15D6,0x1626,0x1676,0x16C6,0x0000,0x0000,
  0x1177,0x11C7,0x1217,0x1267,0x12B7,0x1307,0x1357,0x13A7,
  0x0777,0x07C7,0x0817,0x0867,0x08B7,0x0907,0x0957,0x09A7,
  0x13F7,0x1447,0x1497,0x14E7,0x1537,0x1587,0x15D7,0x1627,
  0x09F7,0x0A47,0x0A97,0x0AE7,0x0B37,0x0B87,0x0BD7,0x0C27,
  0x1677,0x16C7,0x1717,0x1767,0x1178,0x11C8,0x1218,0x1268,
  0x0C77,0x0CC7,0x0D17,0x0D67,0x0778,0x07C8,0x0818,0x0868,
  0x12B8,0x1308,0x1358,0x13A8,0x13F8,0x0000,0x1498,0x14E8,
  0x08B8,0x0908,0x08B6,0x0906,0x09A8,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x1538,0x1588,0x15D8,0x0000,0x1678,0x16C8,
  0x0956,0x09A6,0x09F6,0x0A46,0x0B88,0x0000,0x0000,0x0000,
  0x1718,0x1768,0x1179,0x11C9,0x0000,0x0000,0x12B9,0x1309,
  0x0D18,0x0D68,0x0A96,0x0AE6,0x0000,0x0000,0x0000,0x0000,
  0x13A9,0x13F9,0x1449,0x1499,0x14E9,0x1539,0x1589,0x15D9,
  0x09A9,0x09F9,0x0BD6,0x0C26,0x0B39,0x0000,0x0000,0x0000,
  0x0000,0x0000,0x16C9,0x1719,0x0000,0x0000,0x11CA,0x121A,
  0x0B36,0x0B86,0x0C76,0x0CC6,0x0D19,0x0000,0x0000,0x0000
};


/* 2160-217F */
static const uint16 uni_2160_217F[32]=
{
  0x0739,0x0789,0x07D9,0x0829,0x0879,0x08C9,0x0919,0x0969,
  0x09B9,0x0A09,0x0A59,0x0AA9,0x0AF9,0x0B49,0x0B99,0x0BE9,
  0x1139,0x1189,0x11D9,0x1229,0x1279,0x12C9,0x1319,0x1369,
  0x13B9,0x1409,0x1459,0x14A9,0x14F9,0x1549,0x1599,0x15E9
};


/* 24B0-24EF */
static const uint16 uni_24B0_24EF[64]=
{
  0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0511,0x0512,
  0x0513,0x0514,0x0515,0x0516,0x0517,0x0518,0x0519,0x051A,
  0x051B,0x051C,0x051D,0x051E,0x051F,0x0520,0x0521,0x0522,
  0x0523,0x0524,0x0525,0x0526,0x0527,0x0528,0x0529,0x052A,
  0x0531,0x0532,0x0533,0x0534,0x0535,0x0536,0x0537,0x0538,
  0x0539,0x053A,0x053B,0x053C,0x053D,0x053E,0x053F,0x0540,
  0x0541,0x0542,0x0543,0x0544,0x0545,0x0546,0x0547,0x0548,
  0x0549,0x054A,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000
};


/* FF20-FF5F */
static const uint16 uni_FF20_FF5F[64]=
{
  0x0000,0x0560,0x05B0,0x0600,0x0650,0x06A0,0x06F0,0x0740,
  0x0790,0x07E0,0x0830,0x0880,0x08D0,0x0920,0x0970,0x09C0,
  0x0A10,0x0A60,0x0AB0,0x0B00,0x0B50,0x0BA0,0x0BF0,0x0C40,
  0x0C90,0x0CE0,0x0D30,0x0000,0x0000,0x0000,0x0000,0x0000,
  0x0000,0x0F60,0x0FB0,0x1000,0x1050,0x10A0,0x10F0,0x1140,
  0x1190,0x11E0,0x1230,0x1280,0x12D0,0x1320,0x1370,0x13C0,
  0x1410,0x1460,0x14B0,0x1500,0x1550,0x15A0,0x15F0,0x1640,
  0x1690,0x16E0,0x1730,0x0000,0x0000,0x0000,0x0000,0x0000
};





/*
  Returns
   a number 0..15, if a valid HEX digit in lower case,
   -1 otherwise.
*/

static int hexlo(int x)
{
  static const char hex_lo_digit[256]=
  {
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /*  !"#$%&'()*+,-./ */
     0, 1, 2, 3, 4, 5, 6, 7, 8, 9,-1,-1,-1,-1,-1,-1, /* 0123456789:;<=>? */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* @ABCDEFGHIJKLMNO */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* PQRSTUVWXYZ[\]^_ */
    -1,10,11,12,13,14,15,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* `abcdefghijklmno */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* pqrstuvwxyz{|}~. */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, /* ................ */
  };
  return hex_lo_digit[(unsigned int) x];
}


/*
  Safe characters:
   '\0'  NULL
   A..Z  capital letters,
   a..z  small letters
   0..9  digits
   _     underscore
*/
static const char filename_safe_char[128]=
{
  1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* ................ */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* ................ */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /*  !"#$%&'()*+,-./ */
  1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0, /* 0123456789:;<=>? */
  0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* @ABCDEFGHIJKLMNO */
  1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,1, /* PQRSTUVWXYZ[\]^_ */
  0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* `abcdefghijklmno */
  1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0, /* pqrstuvwxyz{|}~. */
};

#define MY_FILENAME_ESCAPE '@'

/*
  note, that we cannot trust 'e' here, it's may be fake,
  see strconvert()
*/
static int
my_mb_wc_filename(CHARSET_INFO *cs __attribute__((unused)),
                  my_wc_t *pwc, const uchar *s, const uchar *e)
{
  int byte1, byte2;
  if (s >= e)
    return MY_CS_TOOSMALL;

  if (*s < 128 && filename_safe_char[*s])  
  {
    *pwc= *s;
    return 1;
  }
  
  if (*s != MY_FILENAME_ESCAPE)
    return MY_CS_ILSEQ;
  
  if (s + 3 > e)
    return MY_CS_TOOSMALL3;
  
  byte1= s[1];
  if (byte1 == 0)
    return MY_CS_ILSEQ; /* avoid possible out-of-bounds read */
  byte2= s[2];
  
  if (byte1 >= 0x30 && byte1 <= 0x7F &&
      byte2 >= 0x30 && byte2 <= 0x7F)
  {
    int code= (byte1 - 0x30) * 80 + byte2 - 0x30;
    if (code < 5994 && touni[code])
    {
      *pwc= touni[code];
      return 3;
    }
    if (byte1 == '@' && byte2 == '@')
    {
     *pwc= 0;
      return 3;
    }
  }
  
  if (s + 4 > e)
    return MY_CS_TOOSMALL4;

  if ((byte1= hexlo(byte1)) >= 0 &&
      (byte2= hexlo(byte2)) >= 0)
  {
    int byte3= hexlo(s[3]);
    int byte4= hexlo(s[3] ? s[4] : 0);
    if (byte3 >=0 && byte4 >=0)
    {
      *pwc= (byte1 << 12) + (byte2 << 8) + (byte3 << 4) + byte4;
      return 5;
    }
  }
  
  return MY_CS_ILSEQ;
}


static int
my_wc_mb_filename(CHARSET_INFO *cs __attribute__((unused)),
                  my_wc_t wc, uchar *s, uchar *e)
{
  int code;
  static const char hex[]= "0123456789abcdef";

  if (s >= e)
    return MY_CS_TOOSMALL;

  if (wc < 128 && filename_safe_char[wc])
  {
    *s= (uchar) wc;
    return 1;
  }
  
  if (s + 3 > e)
    return MY_CS_TOOSMALL3;

  *s++= MY_FILENAME_ESCAPE;
  if ((wc >= 0x00C0 && wc <= 0x05FF && (code= uni_0C00_05FF[wc - 0x00C0])) ||
      (wc >= 0x1E00 && wc <= 0x1FFF && (code= uni_1E00_1FFF[wc - 0x1E00])) ||
      (wc >= 0x2160 && wc <= 0x217F && (code= uni_2160_217F[wc - 0x2160])) ||
      (wc >= 0x24B0 && wc <= 0x24EF && (code= uni_24B0_24EF[wc - 0x24B0])) ||
      (wc >= 0xFF20 && wc <= 0xFF5F && (code= uni_FF20_FF5F[wc - 0xFF20])))
  {
    *s++= (code / 80) + 0x30;
    *s++= (code % 80) + 0x30;
    return 3;
  }

  /* Non letter */
  if (s + 4 > e)
    return MY_CS_TOOSMALL5;

  *s++= hex[(wc >> 12) & 15];
  *s++= hex[(wc >> 8) & 15];
  *s++= hex[(wc >> 4) & 15];
  *s++= hex[(wc)      & 15];
  return 5;
}


static int
my_charlen_filename(CHARSET_INFO *cs, const uchar *str, const uchar *end)
{
  my_wc_t wc;
  return cs->cset->mb_wc(cs, &wc, str, end);
}


#define MY_FUNCTION_NAME(x)       my_ ## x ## _filename
#define CHARLEN(cs,str,end)       my_charlen_filename(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.ic"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_filename */

#define MY_FUNCTION_NAME(x)      my_ ## x ## _filename
#define DEFINE_STRNNCOLL         0
#define DEFINE_STRNXFRM_UNICODE
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_filename(cs, pwc, s, e)
#define OPTIMIZE_ASCII           0
#define UNICASE_MAXCHAR          MY_UNICASE_INFO_DEFAULT_MAXCHAR
#define UNICASE_PAGE0            my_unicase_default_page00
#define UNICASE_PAGES            my_unicase_default_pages

/*
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(x)            my_weight_mb1_utf8_general_ci(x)
#define WEIGHT_MB2(x,y)          my_weight_mb2_utf8_general_ci(x,y)
#define WEIGHT_MB3(x,y,z)        my_weight_mb3_utf8_general_ci(x,y,z)
*/
#include "strcoll.ic"


static MY_COLLATION_HANDLER my_collation_filename_handler =
{
    NULL,               /* init */
    my_strnncoll_simple,
    my_strnncollsp_simple,
    my_strnxfrm_filename,
    my_strnxfrmlen_unicode,
    my_like_range_mb,
    my_wildcmp_utf8,
    my_strcasecmp_utf8,
    my_instr_mb,
    my_hash_sort_utf8,
    my_propagate_complex
};

static MY_CHARSET_HANDLER my_charset_filename_handler=
{
    NULL,               /* init */
    my_numchars_mb,
    my_charpos_mb,
    my_lengthsp_8bit,
    my_numcells_mb,
    my_mb_wc_filename,
    my_wc_mb_filename,
    my_mb_ctype_mb,
    my_caseup_str_utf8,
    my_casedn_str_utf8,
    my_caseup_utf8,
    my_casedn_utf8,
    my_snprintf_8bit,
    my_long10_to_str_8bit,
    my_longlong10_to_str_8bit,
    my_fill_8bit,
    my_strntol_8bit,
    my_strntoul_8bit,
    my_strntoll_8bit,
    my_strntoull_8bit,
    my_strntod_8bit,
    my_strtoll10_8bit,
    my_strntoull10rnd_8bit,
    my_scan_8bit,
    my_charlen_filename,
    my_well_formed_char_length_filename,
    my_copy_fix_mb,
    my_wc_mb_filename,
};



struct charset_info_st my_charset_filename=
{
    17,0,0,             /* number       */
    MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_HIDDEN|MY_CS_NONASCII,
    "filename",         /* cs name      */
    "filename",         /* name         */
    "",                 /* comment      */
    NULL,               /* tailoring    */
    ctype_utf8,         /* ctype        */
    to_lower_utf8,      /* to_lower     */
    to_upper_utf8,      /* to_upper     */
    to_upper_utf8,      /* sort_order   */
    NULL,               /* uca          */
    NULL,               /* tab_to_uni   */
    NULL,               /* tab_from_uni */
    &my_unicase_default,/* caseinfo     */
    NULL,               /* state_map    */
    NULL,               /* ident_map    */
    1,                  /* strxfrm_multiply */
    1,                  /* caseup_multiply  */
    1,                  /* casedn_multiply  */
    1,                  /* mbminlen     */
    5,                  /* mbmaxlen     */
    0,                  /* min_sort_char */
    0xFFFF,             /* max_sort_char */
    ' ',                /* pad char      */
    0,                  /* escape_with_backslash_is_dangerous */
    1,                  /* levels_for_order   */
    &my_charset_filename_handler,
    &my_collation_filename_handler
};


#endif /* HAVE_CHARSET_UTF8 */



#ifdef HAVE_CHARSET_utf8mb4

/*
  We consider bytes with code more than 127 as a letter.
  This guarantees that word boundaries work fine with regular
  expressions. Note, there is no need to mark byte 255  as a
  letter, it is illegal byte in UTF8.
*/
static uchar ctype_utf8mb4[]=
{
    0,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 40, 40, 40, 40, 40, 32, 32,
   32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
   72, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
  132,132,132,132,132,132,132,132,132,132, 16, 16, 16, 16, 16, 16,
   16,129,129,129,129,129,129,  1,  1,  1,  1,  1,  1,  1,  1,  1,
    1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 16, 16, 16, 16,
   16,130,130,130,130,130,130,  2,  2,  2,  2,  2,  2,  2,  2,  2,
    2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 16, 16, 16, 16, 32,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
    3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  0
};


static uchar to_lower_utf8mb4[]=
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122, 91, 92, 93, 94, 95,
   96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};


static uchar to_upper_utf8mb4[]=
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
   16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
   32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
   48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
   64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
   96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
   80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};


static int
my_mb_wc_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                 my_wc_t * pwc, const uchar *s, const uchar *e)
{
  return my_mb_wc_utf8mb4_quick(pwc, s, e);
}


/*
  The same as above, but without range check
  for example, for a null-terminated string
*/
static int
my_mb_wc_utf8mb4_no_range(CHARSET_INFO *cs __attribute__((unused)),
                          my_wc_t *pwc, const uchar *s)
{
  uchar c;

  c= s[0];
  if (c < 0x80)
  {
    *pwc = c;
    return 1;
  }

  if (c < 0xc2)
    return MY_CS_ILSEQ;

  if (c < 0xe0)
  {
    if (!IS_CONTINUATION_BYTE(s[1]))
      return MY_CS_ILSEQ;

    *pwc= UTF8MB2_CODE(c, s[1]);
    return 2;
  }

  if (c < 0xf0)
  {
    if (!IS_UTF8MB3_STEP2(c, s[1], s[2]))
      return MY_CS_ILSEQ;
    *pwc= UTF8MB3_CODE(c, s[1], s[2]);
    return 3;
  }
  else if (c < 0xf5)
  {
    if (!IS_UTF8MB4_STEP2(c, s[1], s[2], s[3]))
      return MY_CS_ILSEQ;
    *pwc= UTF8MB4_CODE(c, s[1], s[2], s[3]);
    return 4;
  }
  return MY_CS_ILSEQ;
}


static int
my_wc_mb_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                 my_wc_t wc, uchar *r, uchar *e)
{
  int count;

  if (r >= e)
    return MY_CS_TOOSMALL;

  if (wc < 0x80)
    count= 1;
  else if (wc < 0x800)
    count= 2;
  else if (wc < 0x10000)
    count= 3;
  else if (wc < 0x200000)
    count= 4;
  else return MY_CS_ILUNI;

  if (r + count > e)
    return MY_CS_TOOSMALLN(count);

  switch (count) {
    case 4: r[3] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x10000;
      /* fall through */
    case 3: r[2] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
      /* fall through */
    case 2: r[1] = (uchar) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
      /* fall through */
    case 1: r[0] = (uchar) wc;
  }
  return count;
}


/*
  The same as above, but without range check.
*/
static int
my_wc_mb_utf8mb4_no_range(CHARSET_INFO *cs __attribute__((unused)),
                          my_wc_t wc, uchar *r)
{
  int count;

  if (wc < 0x80)
    count= 1;
  else if (wc < 0x800)
    count= 2;
  else if (wc < 0x10000)
    count= 3;
  else if (wc < 0x200000)
    count= 4;
  else
    return MY_CS_ILUNI;

  switch (count)
  {
    case 4: r[3]= (uchar) (0x80 | (wc & 0x3f)); wc= wc >> 6; wc |= 0x10000;
      /* fall through */
    case 3: r[2]= (uchar) (0x80 | (wc & 0x3f)); wc= wc >> 6; wc |= 0x800;
      /* fall through */
    case 2: r[1]= (uchar) (0x80 | (wc & 0x3f)); wc= wc >> 6; wc |= 0xc0;
      /* fall through */
    case 1: r[0]= (uchar) wc;
  }
  return count;
}


static inline void
my_tolower_utf8mb4(MY_UNICASE_INFO *uni_plane, my_wc_t *wc)
{
  if (*wc <= uni_plane->maxchar)
  {
    MY_UNICASE_CHARACTER *page;
    if ((page= uni_plane->page[(*wc >> 8)]))
      *wc= page[*wc & 0xFF].tolower;
  }
}


static inline void
my_toupper_utf8mb4(MY_UNICASE_INFO *uni_plane, my_wc_t *wc)
{
  if (*wc <= uni_plane->maxchar)
  {
    MY_UNICASE_CHARACTER *page;
    if ((page= uni_plane->page[(*wc >> 8)]))
      *wc= page[*wc & 0xFF].toupper;
  }
}


static size_t
my_caseup_utf8mb4(CHARSET_INFO *cs, const char *src, size_t srclen,
                  char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(src != dst || cs->caseup_multiply == 1);

  while ((src < srcend) &&
         (srcres= my_mb_wc_utf8mb4(cs, &wc,
                                   (uchar *) src, (uchar*) srcend)) > 0)
  {
    my_toupper_utf8mb4(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static void
my_hash_sort_utf8mb4_nopad(CHARSET_INFO *cs, const uchar *s, size_t slen,
                     ulong *nr1, ulong *nr2)
{
  my_wc_t wc;
  int res;
  const uchar *e= s + slen;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  register ulong m1= *nr1, m2= *nr2;

  while ((res= my_mb_wc_utf8mb4(cs, &wc, (uchar*) s, (uchar*) e)) > 0)
  {
    my_tosort_unicode(uni_plane, &wc, cs->state);
    MY_HASH_ADD_16(m1, m2, (uint) (wc & 0xFFFF));
    if (wc > 0xFFFF)
    {
       /*
        Put the highest byte only if it is non-zero,
        to make hash functions for utf8mb3 and utf8mb4
        compatible for BMP characters.
        This is useful to keep order of records in
        test results, e.g. for "SHOW GRANTS".
      */
      MY_HASH_ADD(m1, m2, (uint) ((wc >> 16) & 0xFF));
    }
    s+= res;
  }
  *nr1= m1;
  *nr2= m2;
}


static void
my_hash_sort_utf8mb4(CHARSET_INFO *cs, const uchar *s, size_t slen,
                     ulong *nr1, ulong *nr2)
{
  const uchar *e= s + slen;
  /*
    Remove end space. We do this to be able to compare
    'A ' and 'A' as identical
  */
  while (e > s && e[-1] == ' ')
    e--;
  my_hash_sort_utf8mb4_nopad(cs, s, e - s, nr1, nr2);
}


static size_t
my_caseup_str_utf8mb4(CHARSET_INFO *cs, char *src)
{
  my_wc_t wc;
  int srcres, dstres;
  char *dst= src, *dst0= src;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(cs->caseup_multiply == 1);

  while (*src &&
         (srcres= my_mb_wc_utf8mb4_no_range(cs, &wc, (uchar *) src)) > 0)
  {
    my_toupper_utf8mb4(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4_no_range(cs, wc, (uchar*) dst)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  *dst= '\0';
  return (size_t) (dst - dst0);
}


static size_t
my_casedn_utf8mb4(CHARSET_INFO *cs,
                  const char *src, size_t srclen,
                  char *dst, size_t dstlen)
{
  my_wc_t wc;
  int srcres, dstres;
  const char *srcend= src + srclen;
  char *dstend= dst + dstlen, *dst0= dst;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(src != dst || cs->casedn_multiply == 1);

  while ((src < srcend) &&
         (srcres= my_mb_wc_utf8mb4(cs, &wc,
                                   (uchar*) src, (uchar*) srcend)) > 0)
  {
    my_tolower_utf8mb4(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4(cs, wc, (uchar*) dst, (uchar*) dstend)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }
  return (size_t) (dst - dst0);
}


static size_t
my_casedn_str_utf8mb4(CHARSET_INFO *cs, char *src)
{
  my_wc_t wc;
  int srcres, dstres;
  char *dst= src, *dst0= src;
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  DBUG_ASSERT(cs->casedn_multiply == 1);

  while (*src &&
         (srcres= my_mb_wc_utf8mb4_no_range(cs, &wc, (uchar *) src)) > 0)
  {
    my_tolower_utf8mb4(uni_plane, &wc);
    if ((dstres= my_wc_mb_utf8mb4_no_range(cs, wc, (uchar*) dst)) <= 0)
      break;
    src+= srcres;
    dst+= dstres;
  }

  /*
   In rare cases lower string can be shorter than
   the original string, for example:

   "U+0130 LATIN CAPITAL LETTER I WITH DOT ABOVE"
   (which is 0xC4B0 in utf8, i.e. two bytes)

   is converted into

   "U+0069 LATIN SMALL LETTER I"
   (which is 0x69 in utf8, i.e. one byte)

   So, we need to put '\0' terminator after converting.
  */

  *dst= '\0';
  return (size_t) (dst - dst0);
}


/**
  Compare 0-terminated UTF8 strings.

  @param  cs                  character set handler
  @param  s                   First 0-terminated string to compare
  @param  t                   Second 0-terminated string to compare

  @return Comparison result.
    @retval negative number if s < t
    @retval positive number if s > t
    @retval 0 is the strings are equal
*/

static int
my_strcasecmp_utf8mb4(CHARSET_INFO *cs, const char *s, const char *t)
{
  MY_UNICASE_INFO *uni_plane= cs->caseinfo;
  while (s[0] && t[0])
  {
    my_wc_t s_wc,t_wc;
    
    if ((uchar) s[0] < 128)
    {
      /* 
        s[0] is between 0 and 127.
        It represents a single byte character.
        Convert it into weight according to collation.
      */
      s_wc= my_unicase_default_page00[(uchar) s[0]].tolower;
      s++;
    }
    else
    {
      int res= my_mb_wc_utf8mb4_no_range(cs, &s_wc, (const uchar*) s);
      
      /* 
         In the case of wrong multibyte sequence we will
         call strcmp() for byte-to-byte comparison.
      */
      if (res <= 0)
        return strcmp(s, t);
      s+= res;
      
      my_tolower_utf8mb4(uni_plane, &s_wc);
    }
    
    
    /* Do the same for the second string */
    
    if ((uchar) t[0] < 128)
    {
      /* Convert single byte character into weight */
      t_wc= my_unicase_default_page00[(uchar) t[0]].tolower;
      t++;
    }
    else
    {
      int res= my_mb_wc_utf8mb4_no_range(cs, &t_wc, (const uchar*) t);
      if (res <= 0)
        return strcmp(s, t);
      t+= res;
      
      my_tolower_utf8mb4(uni_plane, &t_wc);
    }
    
    /* Now we have two weights, let's compare them */
    if ( s_wc != t_wc )
      return  ((int) s_wc) - ((int) t_wc);
  }
  return ((int) (uchar) s[0]) - ((int) (uchar) t[0]);
}


static int
my_wildcmp_utf8mb4(CHARSET_INFO *cs,
                   const char *str, const char *strend,
                   const char *wildstr, const char *wildend,
                   int escape, int w_one, int w_many)
{
  return my_wildcmp_unicode(cs, str, strend, wildstr, wildend,
                            escape, w_one, w_many, cs->caseinfo); 
}


static int
my_charlen_utf8mb4(CHARSET_INFO *cs __attribute__((unused)),
                   const uchar *s, const uchar *e)
{
  uchar c;

  if (s >= e)
    return MY_CS_TOOSMALL;

  c= s[0];
  if (c < 0xf0)
    return my_valid_mbcharlen_utf8mb3(s, e);

  if (c < 0xf5)
  {
    if (s + 4 > e) /* We need 4 characters */
      return MY_CS_TOOSMALL4;

    if (!IS_UTF8MB4_STEP2(c, s[1], s[2], s[3]))
      return MY_CS_ILSEQ;

    return 4;
  }

  return MY_CS_ILSEQ;
}


#define MY_FUNCTION_NAME(x)       my_ ## x ## _utf8mb4
#define CHARLEN(cs,str,end)       my_charlen_utf8mb4(cs,str,end)
#define DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
#include "ctype-mb.ic"
#undef MY_FUNCTION_NAME
#undef CHARLEN
#undef DEFINE_WELL_FORMED_CHAR_LENGTH_USING_CHARLEN
/* my_well_formed_char_length_utf8mb4 */


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_general_ci
#define DEFINE_STRNXFRM_UNICODE
#define DEFINE_STRNXFRM_UNICODE_NOPAD
#define MY_MB_WC(cs, pwc, s, e)  my_mb_wc_utf8mb4_quick(pwc, s, e)
#define OPTIMIZE_ASCII           1
#define UNICASE_MAXCHAR          MY_UNICASE_INFO_DEFAULT_MAXCHAR
#define UNICASE_PAGE0            my_unicase_default_page00
#define UNICASE_PAGES            my_unicase_default_pages
#define IS_MB4_CHAR(b0,b1,b2,b3) IS_UTF8MB4_STEP3(b0,b1,b2,b3)
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           my_weight_mb1_utf8_general_ci(b0)
#define WEIGHT_MB2(b0,b1)        my_weight_mb2_utf8_general_ci(b0,b1)
#define WEIGHT_MB3(b0,b1,b2)     my_weight_mb3_utf8_general_ci(b0,b1,b2)
/*
  All non-BMP characters have the same weight.
*/
#define WEIGHT_MB4(b0,b1,b2,b3)  MY_CS_REPLACEMENT_CHARACTER
#include "strcoll.ic"


#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_bin
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           ((int) (uchar) (b0))
#define WEIGHT_MB2(b0,b1)        ((int) UTF8MB2_CODE(b0,b1))
#define WEIGHT_MB3(b0,b1,b2)     ((int) UTF8MB3_CODE(b0,b1,b2))
#define WEIGHT_MB4(b0,b1,b2,b3)  ((int) UTF8MB4_CODE(b0,b1,b2,b3))
#include "strcoll.ic"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_general_nopad_ci
#define IS_MB4_CHAR(b0,b1,b2,b3) IS_UTF8MB4_STEP3(b0,b1,b2,b3)
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           my_weight_mb1_utf8_general_ci(b0)
#define WEIGHT_MB2(b0,b1)        my_weight_mb2_utf8_general_ci(b0,b1)
#define WEIGHT_MB3(b0,b1,b2)     my_weight_mb3_utf8_general_ci(b0,b1,b2)
/*
  All non-BMP characters have the same weight.
*/
#define WEIGHT_MB4(b0,b1,b2,b3)  MY_CS_REPLACEMENT_CHARACTER
#include "strcoll.ic"


#define DEFINE_STRNNCOLLSP_NOPAD
#define MY_FUNCTION_NAME(x)      my_ ## x ## _utf8mb4_nopad_bin
#define WEIGHT_ILSEQ(x)          (0xFF0000 + (uchar) (x))
#define WEIGHT_MB1(b0)           ((int) (uchar) (b0))
#define WEIGHT_MB2(b0,b1)        ((int) UTF8MB2_CODE(b0,b1))
#define WEIGHT_MB3(b0,b1,b2)     ((int) UTF8MB3_CODE(b0,b1,b2))
#define WEIGHT_MB4(b0,b1,b2,b3)  ((int) UTF8MB4_CODE(b0,b1,b2,b3))
#include "strcoll.ic"


static MY_COLLATION_HANDLER my_collation_utf8mb4_general_ci_handler=
{
  NULL,               /* init */
  my_strnncoll_utf8mb4_general_ci,
  my_strnncollsp_utf8mb4_general_ci,
  my_strnxfrm_utf8mb4_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8mb4,
  my_strcasecmp_utf8mb4,
  my_instr_mb,
  my_hash_sort_utf8mb4,
  my_propagate_complex
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_bin_handler =
{
    NULL,              /* init */
    my_strnncoll_utf8mb4_bin,
    my_strnncollsp_utf8mb4_bin,
    my_strnxfrm_unicode_full_bin,
    my_strnxfrmlen_unicode_full_bin,
    my_like_range_mb,
    my_wildcmp_mb_bin,
    my_strcasecmp_mb_bin,
    my_instr_mb,
    my_hash_sort_mb_bin,
    my_propagate_simple
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_general_nopad_ci_handler=
{
  NULL,               /* init */
  my_strnncoll_utf8mb4_general_ci,
  my_strnncollsp_utf8mb4_general_nopad_ci,
  my_strnxfrm_nopad_utf8mb4_general_ci,
  my_strnxfrmlen_unicode,
  my_like_range_mb,
  my_wildcmp_utf8mb4,
  my_strcasecmp_utf8mb4,
  my_instr_mb,
  my_hash_sort_utf8mb4_nopad,
  my_propagate_complex
};


static MY_COLLATION_HANDLER my_collation_utf8mb4_nopad_bin_handler =
{
  NULL,		/* init */
  my_strnncoll_utf8mb4_bin,
  my_strnncollsp_utf8mb4_nopad_bin,
  my_strnxfrm_unicode_full_nopad_bin,
  my_strnxfrmlen_unicode_full_bin,
  my_like_range_mb,
  my_wildcmp_mb_bin,
  my_strcasecmp_mb_bin,
  my_instr_mb,
  my_hash_sort_mb_nopad_bin,
  my_propagate_simple
};


MY_CHARSET_HANDLER my_charset_utf8mb4_handler=
{
  NULL,               /* init */
  my_numchars_mb,
  my_charpos_mb,
  my_lengthsp_8bit,
  my_numcells_mb,
  my_mb_wc_utf8mb4,
  my_wc_mb_utf8mb4,
  my_mb_ctype_mb,
  my_caseup_str_utf8mb4,
  my_casedn_str_utf8mb4,
  my_caseup_utf8mb4,
  my_casedn_utf8mb4,
  my_snprintf_8bit,
  my_long10_to_str_8bit,
  my_longlong10_to_str_8bit,
  my_fill_utf8,
  my_strntol_8bit,
  my_strntoul_8bit,
  my_strntoll_8bit,
  my_strntoull_8bit,
  my_strntod_8bit,
  my_strtoll10_8bit,
  my_strntoull10rnd_8bit,
  my_scan_8bit,
  my_charlen_utf8mb4,
  my_well_formed_char_length_utf8mb4,
  my_copy_fix_mb,
  my_wc_mb_utf8mb4,
};



struct charset_info_st my_charset_utf8mb4_general_ci=
{
  45,0,0,              /* number       */
  MY_CS_COMPILED|MY_CS_PRIMARY|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_UNICODE_SUPPLEMENT,  /* state  */
  MY_UTF8MB4,         /* cs name      */
  MY_UTF8MB4_GENERAL_CI,/* name       */
  "UTF-8 Unicode",    /* comment      */
  NULL,               /* tailoring    */
  ctype_utf8mb4,      /* ctype        */
  to_lower_utf8mb4,   /* to_lower     */
  to_upper_utf8mb4,   /* to_upper     */
  to_upper_utf8mb4,   /* sort_order   */
  NULL,               /* uca          */
  NULL,               /* tab_to_uni   */
  NULL,               /* tab_from_uni */
  &my_unicase_default,/* caseinfo     */
  NULL,               /* state_map    */
  NULL,               /* ident_map    */
  1,                  /* strxfrm_multiply */
  1,                  /* caseup_multiply  */
  1,                  /* casedn_multiply  */
  1,                  /* mbminlen     */
  4,                  /* mbmaxlen     */
  0,                  /* min_sort_char */
  0xFFFF,             /* max_sort_char */
  ' ',                /* pad char      */
  0,                  /* escape_with_backslash_is_dangerous */
  1,                  /* levels_for_order   */
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_general_ci_handler
};


struct charset_info_st my_charset_utf8mb4_bin=
{
  46,0,0,             /* number       */
  MY_CS_COMPILED|MY_CS_BINSORT|MY_CS_STRNXFRM|MY_CS_UNICODE|
  MY_CS_UNICODE_SUPPLEMENT, /* state  */
  MY_UTF8MB4,         /* cs name      */
  MY_UTF8MB4_BIN,     /* name         */
  "UTF-8 Unicode",    /* comment      */
  NULL,               /* tailoring    */
  ctype_utf8mb4,      /* ctype        */
  to_lower_utf8mb4,   /* to_lower     */
  to_upper_utf8mb4,   /* to_upper     */
  NULL,               /* sort_order   */
  NULL,               /* uca          */
  NULL,               /* tab_to_uni   */
  NULL,               /* tab_from_uni */
  &my_unicase_default,/* caseinfo     */
  NULL,               /* state_map    */
  NULL,               /* ident_map    */
  1,                  /* strxfrm_multiply */
  1,                  /* caseup_multiply  */
  1,                  /* casedn_multiply  */
  1,                  /* mbminlen     */
  4,                  /* mbmaxlen     */
  0,                  /* min_sort_char */
  0xFFFF,             /* max_sort_char */
  ' ',                /* pad char      */
  0,                  /* escape_with_backslash_is_dangerous */
  1,                  /* levels_for_order   */
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_bin_handler
};


struct charset_info_st my_charset_utf8mb4_general_nopad_ci=
{
  MY_NOPAD_ID(45),0,0,/* number       */
  MY_CS_COMPILED|MY_CS_STRNXFRM|MY_CS_UNICODE|MY_CS_UNICODE_SUPPLEMENT|
  MY_CS_NOPAD,        /* state            */
  MY_UTF8MB4,         /* cs name          */
  MY_UTF8MB4_GENERAL_NOPAD_CI,/* name     */
  "UTF-8 Unicode",    /* comment          */
  NULL,               /* tailoring        */
  ctype_utf8mb4,      /* ctype            */
  to_lower_utf8mb4,   /* to_lower         */
  to_upper_utf8mb4,   /* to_upper         */
  to_upper_utf8mb4,   /* sort_order       */
  NULL,               /* uca              */
  NULL,               /* tab_to_uni       */
  NULL,               /* tab_from_uni     */
  &my_unicase_default,/* caseinfo         */
  NULL,               /* state_map        */
  NULL,               /* ident_map        */
  1,                  /* strxfrm_multiply */
  1,                  /* caseup_multiply  */
  1,                  /* casedn_multiply  */
  1,                  /* mbminlen         */
  4,                  /* mbmaxlen         */
  0,                  /* min_sort_char    */
  0xFFFF,             /* max_sort_char    */
  ' ',                /* pad char         */
  0,                  /* escape_with_backslash_is_dangerous */
  1,                  /* levels_for_order */
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_general_nopad_ci_handler
};


struct charset_info_st my_charset_utf8mb4_nopad_bin=
{
  MY_NOPAD_ID(46),0,0, /* number          */
  MY_CS_COMPILED|MY_CS_BINSORT|MY_CS_STRNXFRM|MY_CS_UNICODE|
  MY_CS_UNICODE_SUPPLEMENT|MY_CS_NOPAD, /* state  */
  MY_UTF8MB4,         /* cs name          */
  MY_UTF8MB4_NOPAD_BIN, /* name           */
  "UTF-8 Unicode",    /* comment          */
  NULL,               /* tailoring        */
  ctype_utf8mb4,      /* ctype            */
  to_lower_utf8mb4,   /* to_lower         */
  to_upper_utf8mb4,   /* to_upper         */
  NULL,               /* sort_order       */
  NULL,               /* uca              */
  NULL,               /* tab_to_uni       */
  NULL,               /* tab_from_uni     */
  &my_unicase_default,/* caseinfo         */
  NULL,               /* state_map        */
  NULL,               /* ident_map        */
  1,                  /* strxfrm_multiply */
  1,                  /* caseup_multiply  */
  1,                  /* casedn_multiply  */
  1,                  /* mbminlen         */
  4,                  /* mbmaxlen         */
  0,                  /* min_sort_char    */
  0xFFFF,             /* max_sort_char    */
  ' ',                /* pad char         */
  0,                  /* escape_with_backslash_is_dangerous */
  1,                  /* levels_for_order */
  &my_charset_utf8mb4_handler,
  &my_collation_utf8mb4_nopad_bin_handler
};

#endif /* HAVE_CHARSET_utf8mb4 */
