/* Copyright (c) 2007, 2011, Oracle and/or its affiliates.
   Copyright (c) 2009, 2017, MariaDB Corporation.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1335  USA */

#ifndef MY_BIT_INCLUDED
#define MY_BIT_INCLUDED

/*
  Some useful bit functions
*/

C_MODE_START

extern const uchar _my_bits_reverse_table[256];

/*
  Find smallest X in 2^X >= value
  This can be used to divide a number with value by doing a shift instead
*/

static inline uint my_bit_log2(ulong value)
{
  uint bit;
  for (bit=0 ; value > 1 ; value>>=1, bit++) ;
  return bit;
}


/*
Count bits in 32bit integer

  Algorithm by Sean Anderson, according to:
  http://graphics.stanford.edu/~seander/bithacks.html
  under "Counting bits set, in parallel"

 (Original code public domain).
*/
static inline uint my_count_bits_uint32(uint32 v)
{
  v = v - ((v >> 1) & 0x55555555);
  v = (v & 0x33333333) + ((v >> 2) & 0x33333333);
  return (((v + (v >> 4)) & 0xF0F0F0F) * 0x1010101) >> 24;
}


static inline uint my_count_bits(ulonglong x)
{
  return my_count_bits_uint32((uint32)x) + my_count_bits_uint32((uint32)(x >> 32));
}




/*
  Next highest power of two

  SYNOPSIS
    my_round_up_to_next_power()
    v		Value to check

  RETURN
    Next or equal power of 2
    Note: 0 will return 0

  NOTES
    Algorithm by Sean Anderson, according to:
    http://graphics.stanford.edu/~seander/bithacks.html
    (Original code public domain)

    Comments shows how this works with 01100000000000000000000000001011
*/

static inline uint32 my_round_up_to_next_power(uint32 v)
{
  v--;			/* 01100000000000000000000000001010 */
  v|= v >> 1;		/* 01110000000000000000000000001111 */
  v|= v >> 2;		/* 01111100000000000000000000001111 */
  v|= v >> 4;		/* 01111111110000000000000000001111 */
  v|= v >> 8;		/* 01111111111111111100000000001111 */
  v|= v >> 16;		/* 01111111111111111111111111111111 */
  return v+1;		/* 10000000000000000000000000000000 */
}

static inline uint32 my_clear_highest_bit(uint32 v)
{
  uint32 w=v >> 1;
  w|= w >> 1;
  w|= w >> 2;
  w|= w >> 4;
  w|= w >> 8;
  w|= w >> 16;
  return v & w;
}

static inline uint32 my_reverse_bits(uint32 key)
{
  return
    ((uint32)_my_bits_reverse_table[ key      & 255] << 24) |
    ((uint32)_my_bits_reverse_table[(key>> 8) & 255] << 16) |
    ((uint32)_my_bits_reverse_table[(key>>16) & 255] <<  8) |
     (uint32)_my_bits_reverse_table[(key>>24)      ];
}

/*
  a number with the n lowest bits set
  an overflow-safe version of  (1 << n) - 1
*/
static inline uint64 my_set_bits(int n)
{
  return (((1ULL << (n - 1)) - 1) << 1) | 1;
}

/* Create a mask of the significant bits for the last byte (1,3,7,..255) */
static inline uchar last_byte_mask(uint bits)
{
  /* Get the number of used bits-1 (0..7) in the last byte */
  unsigned int const used = (bits - 1U) & 7U;
  /* Return bitmask for the significant bits */
  return ((2U << used) - 1);
}

#ifdef _MSC_VER
#include <intrin.h>
#endif

#define MY_FIND_FIRST_BIT_END sizeof(ulonglong)*8
/*
  Find the position of the first(least significant) bit set in
  the argument. Returns 64 if the argument was 0.
*/
static inline uint my_find_first_bit(ulonglong n)
{
  if(!n)
    return MY_FIND_FIRST_BIT_END;
#if defined(__GNUC__)
  return __builtin_ctzll(n);
#elif defined(_MSC_VER)
#if defined(_M_IX86)
  unsigned long bit;
  if( _BitScanForward(&bit, (uint)n))
    return bit;
  _BitScanForward(&bit, (uint)(n>>32));
  return bit + 32;
#else
  unsigned long bit;
  _BitScanForward64(&bit, n);
  return bit;
#endif
#else
  /* Generic case */
  uint  shift= 0;
  static const uchar last_bit[16] = { 32, 0, 1, 0,
                                      2, 0, 1, 0,
                                      3, 0, 1, 0,
                                      2, 0, 1, 0};
  uint bit;
  while ((bit = last_bit[(n >> shift) & 0xF]) == 32)
    shift+= 4;
  return shift+bit;
#endif
}
C_MODE_END

#endif /* MY_BIT_INCLUDED */
